CREATE OR REPLACE PACKAGE plpdf_xfdf is
--v2.0.0
/* Simple XFDF Composer and Parser*/

-- type
type t_xml_inputs is table of plx_sc.maxv2 index by plx_sc.normalv2;

-- variable
g_xml_doc clob;

-- constants
c_xml_version    constant plx_sc.maxv2 := '1.0';
c_xml_encoding   constant plx_sc.maxv2 := 'utf-8';
c_xml_standalone constant plx_sc.maxv2 := null;
c_xmlns          constant plx_sc.maxv2 := 'http://ns.adobe.com/xfdf/';
c_xml_space      constant plx_sc.maxv2 := 'preserve';
t_xmlns    constant plx_sc.normalv2 := 'xmlns';
t_xml      constant plx_sc.normalv2 := 'xml';
t_space    constant plx_sc.normalv2 := 'space';
t_xfdf     constant plx_sc.normalv2 := 'xfdf';
t_original constant plx_sc.normalv2 := 'original';
t_modified constant plx_sc.normalv2 := 'modified';
t_ids      constant plx_sc.normalv2 := 'ids';
t_fields   constant plx_sc.normalv2 := 'fields';
t_name     constant plx_sc.normalv2 := 'name';
t_field    constant plx_sc.normalv2 := 'field';
t_value    constant plx_sc.normalv2 := 'value';

/* Subprograms */

-- Start the document
procedure start_;

-- End the document
procedure end_;

-- Open xfdf tag
procedure startxfdf;

-- Close xfdf tag
procedure endxfdf;

-- Add ids tag with original and modified attributes
procedure ids(
  p_original_id plx_sc.maxv2,
  p_modified_id plx_sc.maxv2
  );

-- Open fields tag
procedure startfields;

-- Close fields tag
procedure endfields;

-- Open field tag and add name attribute
procedure startfield(
  p_name plx_sc.maxv2
  );

-- Close field tag
procedure endfield;

-- Open value tag
procedure startvalue;

-- Close value tag
procedure endvalue;

-- Add value
procedure value(
    p_value plx_sc.maxv2
);

-- Get blob type from clob
function getblob(
  p_xml_doc clob
  ) return blob;

/*
-TEST CREATE
declare
v_nodes plpdf_xfdf.t_xml_inputs;
v_blob blob;
begin
v_nodes('CAR') := 'Ford';
v_nodes('MOTORBIKE') := 'Suzuki';
v_nodes('BIKE') := 'KTM';
v_nodes('BOAT') := 'Bayliner';
v_blob := plpdf_xfdf.create_xfdf(v_nodes, sys_guid());
end;
*/

-- Generate an XFDF document from PL/SQL table
function create_xfdf(
  p_xml_inputs t_xml_inputs,
  p_id varchar2 default null
  ) return blob;

/*
-TEST PARSE
declare
p_file blob;
v_clob clob;
v_varchar varchar2(32767);
v_start pls_integer := 1;
v_buffer pls_integer := 32767;
xml_ret plpdf_xfdf.t_xml_inputs;
v_i plx_sc.normalv2;
begin
    select blob_file into p_file from store_blob_tk where filename = 'test_input.xml';
    dbms_lob.createtemporary(v_clob, true);

    for i in 1..ceil(dbms_lob.getlength(p_file) / v_buffer)
        loop
            v_varchar := utl_raw.cast_to_varchar2(dbms_lob.substr(p_file, v_buffer, v_start));
            dbms_lob.writeappend(v_clob, length(v_varchar), v_varchar);
            v_start := v_start + v_buffer;
        end loop;

    xml_ret := plpdf_xfdf.parse_xfdf(v_clob);

    dbms_output.enable(1000000);
    v_i := xml_ret.first;
    while v_i is not null loop
        dbms_output.put_line('xml_ret(' || v_i || ') = ' || xml_ret(v_i));
        v_i := xml_ret.next(v_i);
    end loop;
end;
*/
-- Processing an XFDF document into PL/SQL table
function parse_xfdf(
  p_file clob
  ) return t_xml_inputs;

end plpdf_xfdf;
/

CREATE OR REPLACE PACKAGE BODY plpdf_xfdf wrapped 
a000000
b2
abcd
abcd
abcd
abcd
abcd
abcd
abcd
abcd
abcd
abcd
abcd
abcd
abcd
abcd
abcd
b
f1f 562
AdWbP3WNK167ctc22244jK4R/Hswgw0r2tAFVy+8I6pEmFpbkFJSiKj7EffCWGmJZwmHIGMb
tPwYif5Ogs+3wpFXZ2ZwsWbBgRCCamsbaBBqFxjc70KEJPn4CHMQOrVz+Nrg/EMB8QuA+lY+
CasM7TTKAxBRFxuPMgbE6TJb1WYLeCjDffw0aDm83cJIv8OHlo62xbfJHRjQQf/FhybDfo5y
Pl6z8Eb0TseDpq4vR4A1+yGUGtdYaiMTeOffk2SkVJvEQdL/mebPr6PZj2CdLndITZ+k+7bg
Nooh9qOtCjnetvJN1DgfxXLHSKXFw94LP3dG2oM2g8lS6huX3qH67B3k1B2a/8xmME/oTJz9
4gTMjX+HYPrdKIuZ7689qSCxCwtjHYnEF4nYY8rYqq6nhOTo28KcuUdlcPbybnIBq84HomoJ
//VkjN3VSSe1okgjv2AFlM+PuGRAmLnT5upZN1uSB/yRcXgi28c+YVMcvwNcOoHDigEGGDpZ
qi3DPMn9VbyYAPUS/4q81fcQVdizEKQh7ZT76FypP2+QQDjw/8NPzjDA8zR7/HDGkqfDcgnK
jDdLHgIOC41QsHL62PDACiD5VQ1a58a2oJHgE2oHWZi/yNqhLeifemy0WVkDK0hKkgIThHw/
FceVwphwQqJGMzCLq/RYToB6ghzJ2UH3WulfwN3kI1l2kw+prfWGViGSrPnpkuYC7X1oYlXo
v2oVYtdxKrj9mUX/9Wion/0c6UeYKD2CJqDrMO183MxDlFjm4PVZ9tvGUpv+eZ/5QntYzOCY
bIpTHEeGRBMRjiMiurrBOO0dunXfYgbWawKI6q1Cqm2VldPA021E++ZtjZqt1JYDp66BtEZj
l7rYeK1MLG00MRVNIdrKy2KcCqBDOlIoVvpXdo+Wz+XOh/29TFdeRcNlBcmxPlLFItS5b064
UWWeShPhC9C5CxN5sI8H66OvIB97lZD33ISiJI/epDl04/c3TfB4mSFKmb/BxUHYid8WfK2y
JwjcRBDXCeq+5j28R0Jvr/8TZ6ow4fzS6AgxqZfBswedJHeTI5mEtK0eDI2tyy2SzFFk471X
igeES3NbEFkXSI3kXZzee13R/dINy/BMie9YdM79srHjA3G6206n3Unih8YFUynrZHk3DJRk
Brg1Y6GLzRgnH+4GsKnFqINUfLflZw9Y7sNseDLzbGFO1uYEjN/8xh9etis8XVQ8FBdLXvOl
VcqnvsTOttOMXesnoVFy+bewFO4GRp+WUwSCeODqQp5OyIHesG7Q9jqktrggZEDL0UTCzAfT
Ajph2TAwazqnuD/qAQgD4hg0/En0Qw7rEtx5ABo488mgzBPOoiPaQQWdP+FI2d0W

/
