CREATE OR REPLACE PACKAGE plpdf_toolkit is
/**
 <config>
  <pck-name>PL/PDF Toolkit</pck-name>
  <pck-title>User's Guide</pck-title>
  <pck-version>v3.0</pck-version>
  <header-prc>plpdf_doc_xml.plpdf_header</header-prc>
  <header-size>15</header-size>
  <footer-prc>plpdf_doc_xml.plpdf_footer</footer-prc>
  <footer-size>10</footer-size>
 </config>
*/

/**
<h1>Introdution</h1>
<br/>
<p>
 PL/PDF Toolkit is a java library with PL/SQL interface for manipulating existing PDF documents.
</p>
<newpage/>
*/

/**
<h1>Certification</h1>
<h2>GetCertKey</h2>
<type>Function</type>
<desc>
  Generates preliminary certification key.
  This function should be run once when the PL/PDF package is installed. The function generates a string
  that should be sent to <a>mailto:info@plpdf.com</a>. Our representatives will send a certificate key back that should
  be used as the return value in the <i>pl_pdf_cert.SetCertKey</i> function.
</desc>
<params>-</params>
<return>
 <return-def>varchar2</return-def>
 <return-desc>Preliminary certificate key</return-desc>
</return>
<newpage/>
*/
function GetCertKey
  return varchar2;

/**
<h1>Types And Constants</h1>
<h2>t14font</h2>
<type>Subtype, VARHCAR2(30 CHAR)</type>
<desc>
 Standard Type 1 Fonts<br/>
 The PostScript names of 14 Type 1 fonts, known as the standard fonts.
 These fonts, or their font metrics and suitable substitution fonts, are required to be available to the viewer application.
</desc>
<br/>
*/
subtype t_14font is varchar2(30 char);

/**
<h2>t_color</h2>
<type>Record</type>
<desc>
Represents the RGB color triplet.
</desc>
<fields>
 <field>
  <field-def>r number</field-def>
  <field-desc>Value of the red component</field-desc>
 </field>
 <field>
  <field-def>g number</field-def>
  <field-desc>Value of the green component</field-desc>
 </field>
 <field>
  <field-def>b number</field-def>
  <field-desc>Value of the blue component</field-desc>
 </field>
</fields>
*/
type t_color is record (
  r number,
  g number,
  b number
  );
/**
<br/>
<h2>times-roman</h2>
<type>Constant</type>
<constant>
 <constant-def>t_14font</constant-def>
 <constant-desc>Times-Roman</constant-desc>
</constant>

<h2>times_bold</h2>
<type>Constant</type>
<constant>
 <constant-def>t_14font</constant-def>
 <constant-desc>Times-Bold</constant-desc>
</constant>

<h2>times_italic</h2>
<type>Constant</type>
<constant>
 <constant-def>t_14font</constant-def>
 <constant-desc>Times-Italic</constant-desc>
</constant>

<h2>times_bold_italic</h2>
<type>Constant</type>
<constant>
 <constant-def>t_14font</constant-def>
 <constant-desc>Times-BoldItalic</constant-desc>
</constant>

<h2>helvetica</h2>
<type>Constant</type>
<constant>
 <constant-def>t_14font</constant-def>
 <constant-desc>Helvetica</constant-desc>
</constant>

<h2>helvetica_bold</h2>
<type>Constant</type>
<constant>
 <constant-def>t_14font</constant-def>
 <constant-desc>Helvetica-Bold</constant-desc>
</constant>

<h2>helvetica_oblique</h2>
<type>Constant</type>
<constant>
 <constant-def>t_14font</constant-def>
 <constant-desc>Helvetica-Oblique</constant-desc>
</constant>

<h2>helvetica_bold_oblique</h2>
<type>Constant</type>
<constant>
 <constant-def>t_14font</constant-def>
 <constant-desc>Helvetica-BoldOblique</constant-desc>
</constant>

<h2>courier</h2>
<type>Constant</type>
<constant>
 <constant-def>t_14font</constant-def>
 <constant-desc>Courier</constant-desc>
</constant>

<h2>courier_bold</h2>
<type>Constant</type>
<constant>
 <constant-def>t_14font</constant-def>
 <constant-desc>Courier-Bold</constant-desc>
</constant>

<h2>courier_oblique</h2>
<type>Constant</type>
<constant>
 <constant-def>t_14font</constant-def>
 <constant-desc>Courier-Oblique</constant-desc>
</constant>

<h2>courier_bold_oblique</h2>
<type>Constant</type>
<constant>
 <constant-def>t_14font</constant-def>
 <constant-desc>Courier-BoldOblique</constant-desc>
</constant>

<h2>symbol</h2>
<type>Constant</type>
<constant>
 <constant-def>t_14font</constant-def>
 <constant-desc>Symbol</constant-desc>
</constant>

<h2>zapf_dingbats</h2>
<type>Constant</type>
<constant>
 <constant-def>t_14font</constant-def>
 <constant-desc>ZapfDingbats</constant-desc>
</constant>
<newpage/>
*/
  times_roman constant t_14font := 'Times-Roman';
  times_bold constant t_14font := 'Times-Bold';
  times_italic constant t_14font := 'Times-Italic';
  times_bold_italic constant t_14font := 'Times-BoldItalic';
  helvetica constant t_14font := 'Helvetica';
  helvetica_bold constant t_14font := 'Helvetica-Bold';
  helvetica_oblique constant t_14font := 'Helvetica-Oblique';
  helvetica_bold_oblique constant t_14font := 'Helvetica-BoldOblique';
  courier constant t_14font := 'Courier';
  courier_bold constant t_14font := 'Courier-Bold';
  courier_oblique constant t_14font := 'Courier-Oblique';
  courier_bold_oblique constant t_14font := 'Courier-BoldOblique';
  symbol constant t_14font := 'Symbol';
  zapf_dingbats constant t_14font := 'ZapfDingbats';

/**
<br/>
<h2>Metadata</h2>
<type>Record</type>
<desc>
  A PDF document may include general information such as the document?s title,
  author, and creation and modification dates. Such global information about the
  document itself (as opposed to its content or structure) is called metadata, and is
  intended to assist in cataloguing and searching for documents in external databases.
</desc>
<fields>
 <field>
  <field-def>Title</field-def>
  <field-desc>The document?s title</field-desc>
 </field>
 <field>
  <field-def>Author</field-def>
  <field-desc>The name of the person who created the document</field-desc>
 </field>
 <field>
  <field-def>Subject</field-def>
  <field-desc>The subject of the document</field-desc>
 </field>
 <field>
  <field-def>Creator</field-def>
  <field-desc>If the document was converted to PDF from another format, the
   name of the application (for example, Adobe FrameMaker?) that created the
   original document from which it was converted.
  </field-desc>
 </field>
 <field>
  <field-def>Keywords</field-def>
  <field-desc>Keywords associated with the document</field-desc>
 </field>
 <field>
  <field-def>Producer</field-def>
  <field-desc>If the document was converted to PDF from another format, the
   name of the application (for example, Acrobat Distiller) that converted it to PDF.
  </field-desc>
 </field>
 <field>
  <field-def>CreationDate</field-def>
  <field-desc>The date and time the document was created, in human-readable form</field-desc>
 </field>
 <field>
  <field-def>ModDate</field-def>
  <field-desc>The date and time the document was most recently modified, in human-readable form</field-desc>
 </field>
</fields>
*/
type t_metadata is record (
  Title varchar2(2000 char),
  Author varchar2(2000 char),
  Subject varchar2(2000 char),
  Creator varchar2(2000 char),
  Keywords varchar2(2000 char),
  Producer varchar2(2000 char),
  CreationDate date,
  ModDate date
  );

/**
<br/>
<h2>Merge input</h2>
<type>Record</type>
<desc>
  PDF document with some properties.
</desc>
<fields>
 <field>
  <field-def>inputfile</field-def>
  <field-desc>Source PDF file</field-desc>
 </field>
 <field>
  <field-def>password</field-def>
  <field-desc>The owner password of the PDF file (if encrypted).</field-desc>
 </field>
 <field>
  <field-def>overwrite_rect_x</field-def>
  <field-desc>Overwrite rectangle x coordinate</field-desc>
 </field>
 <field>
  <field-def>overwrite_rect_y</field-def>
  <field-desc>Overwrite rectangle y coordinate</field-desc>
 </field>
 <field>
  <field-def>overwrite_rect_w</field-def>
  <field-desc>Overwrite rectangle width</field-desc>
 </field>
 <field>
  <field-def>overwrite_rect_h</field-def>
  <field-desc>Overwrite rectangle heigth</field-desc>
 </field><field>
  <field-def>overwrite_rect_color</field-def>
  <field-desc>Overwrite rectangle fill color</field-desc>
 </field>
</fields>
*/
type t_merge_input is record (
  inputfile blob,
  password varchar2(255 char),
  overwrite_rect_x number,
  overwrite_rect_y number,
  overwrite_rect_w number,
  overwrite_rect_h number,
  overwrite_rect_color t_color
  );

/**
<br/>
<h2>Merge inputs</h2>
<type>Table</type>
<desc>
  Table of Merge input records.
</desc>
<fields>
 <field>
  <field-def>Input record</field-def>
  <field-desc>PDF document with some properties</field-desc>
 </field>
</fields>
*/
type t_merge_inputs is table of t_merge_input index by pls_integer;

/**
<newpage/>
<h1>Functions</h1>
<br/>
<h2>Encryption</h2>
<p>
  A PDF document can be encrypted to protect its contents from unauthorized
  access. Encryption applies to all strings and streams in the document?s
  PDF file, but not to other object types such as integers and boolean values, which
  are used primarily to convey information about the document?s structure rather
  than its content. Leaving these values unencrypted allows random access to the
  objects within a document, while encrypting the strings and streams protects the
  document?s substantive contents.
</p>
*/

/**
<h2>Encrypt40</h2>
<type>Function</type>
<desc>
  Encrypt any PDF document using standard PDF 40 RC4 bit encryption.
</desc>
<version>v2.0.0</version>
<params>
 <param>
  <param-def>p_inputfile blob</param-def>
  <param-desc>source PDF file</param-desc>
 </param>
 <param>
  <param-def>p_opassword varchar2</param-def>
  <param-desc>the owner password to the PDF file</param-desc>
 </param>
 <param>
  <param-def>p_upassword varchar2 default null</param-def>
  <param-desc>the user password to the PDF file</param-desc>
 </param>
 <param>
  <param-def>p_canextractcontent boolean default false</param-def>
  <param-desc>The user is allowed to copy or otherwise
    extract text and graphics from the document, including extracting text and graphics
    (in support of accessibility to disabled users or for other purposes).
  </param-desc>
 </param>
 <param>
  <param-def>p_canmodify boolean default false</param-def>
  <param-desc>The user is allowed to modify the contents of the document.</param-desc>
 </param>
 <param>
  <param-def>p_canmodifyannotations boolean default false</param-def>
  <param-desc>The user is allowed to add or modify text annotations,
    fill in interactive form fields, and, if "modify" is also set, create or modify interactive form fields
    (including signature fields).
  </param-desc>
 </param>
 <param>
  <param-def>p_canprint boolean default false</param-def>
  <param-desc>The user is allowed to print the document.</param-desc>
 </param>
</params>
<return>
 <return-def>blob</return-def>
 <return-desc>result PDF document</return-desc>
</return>
*/
function Encrypt40(
  p_inputfile blob,
  p_opassword varchar2,
  p_upassword varchar2 default null,
  p_canextractcontent boolean default false,
  p_canmodify boolean default false,
  p_canmodifyannotations boolean default false,
  p_canprint boolean default false
  ) return blob;

/**
<h2>Encrypt128</h2>
<type>Function</type>
<desc>
  Encrypt any PDF document using standard PDF 128 RC4 bit encryption.
</desc>
<version>v2.0.0</version>
<params>
 <param>
  <param-def>p_inputfile blob</param-def>
  <param-desc>source PDF file</param-desc>
 </param>
 <param>
  <param-def>p_opassword varchar2</param-def>
  <param-desc>the owner password to the PDF file</param-desc>
 </param>
 <param>
  <param-def>p_upassword varchar2 default null</param-def>
  <param-desc>the user password to the PDF file</param-desc>
 </param>
 <param>
  <param-def>p_canassemble boolean default false</param-def>
  <param-desc>The user is allowed to assemble the document
   (insert, rotate, or delete pages and create bookmarks or thumbnail images).
  </param-desc>
 </param>
 <param>
  <param-def>p_canextractcontent boolean default false</param-def>
  <param-desc>The user is allowed to copy or otherwise
    extract text and graphics from the document.
  </param-desc>
 </param>
 <param>
  <param-def>p_canextractforaccessibility boolean default false</param-def>
  <param-desc>The user is allowed to extract text and graphics
   (in support of accessibility to disabled users or for other purposes).
  </param-desc>
 </param>
 <param>
  <param-def>p_canfillinform boolean default false</param-def>
  <param-desc>The user is allowed to fill in existing interactive form fields
   (including signature fields)
  </param-desc>
 </param>
 <param>
  <param-def>p_canmodify boolean default false</param-def>
  <param-desc>The user is allowed to modify the contents of the document</param-desc>
 </param>
 <param>
  <param-def>p_canmodifyannotations boolean default false</param-def>
  <param-desc>Add or modify text annotations, fill in interactive form fields.</param-desc>
 </param>
 <param>
  <param-def>p_canprint boolean default false</param-def>
  <param-desc>Print the document to a representation from which a faithful digital copy
  of the PDF content could be generated.</param-desc>
 </param>
 <param>
  <param-def>p_canmodifyprintresolution boolean default false</param-def>
  <param-desc>The user is not allowed to print in highest quality.</param-desc>
 </param>
</params>
<return>
 <return-def>blob</return-def>
 <return-desc>result PDF document</return-desc>
</return>
*/
function Encrypt128(
  p_inputfile blob,
  p_opassword varchar2,
  p_upassword varchar2 default null,
  p_canassemble boolean default false,
  p_canextractcontent boolean default false,
  p_canextractforaccessibility boolean default false,
  p_canfillinform boolean default false,
  p_canmodify boolean default false,
  p_canmodifyannotations boolean default false,
  p_canprint boolean default false,
  p_canmodifyprintresolution boolean default false
  ) return blob;

/**
<h2>AESEncrypt128</h2>
<type>Function</type>
<desc>
  Encrypt any PDF document using standard PDF 128 AES bit encryption.
</desc>
<version>v2.0.0</version>
<params>
 <param>
  <param-def>p_inputfile blob</param-def>
  <param-desc>source PDF file</param-desc>
 </param>
 <param>
  <param-def>p_opassword varchar2</param-def>
  <param-desc>the owner password to the PDF file</param-desc>
 </param>
 <param>
  <param-def>p_upassword varchar2 default null</param-def>
  <param-desc>the user password to the PDF file</param-desc>
 </param>
 <param>
  <param-def>p_canassemble boolean default false</param-def>
  <param-desc>The user is allowed to assemble the document
   (insert, rotate, or delete pages and create bookmarks or thumbnail images).
  </param-desc>
 </param>
 <param>
  <param-def>p_canextractcontent boolean default false</param-def>
  <param-desc>The user is allowed to copy or otherwise
    extract text and graphics from the document.
  </param-desc>
 </param>
 <param>
  <param-def>p_canextractforaccessibility boolean default false</param-def>
  <param-desc>The user is allowed to extract text and graphics
   (in support of accessibility to disabled users or for other purposes).
  </param-desc>
 </param>
 <param>
  <param-def>p_canfillinform boolean default false</param-def>
  <param-desc>The user is allowed to fill in existing interactive form fields
   (including signature fields)
  </param-desc>
 </param>
 <param>
  <param-def>p_canmodify boolean default false</param-def>
  <param-desc>The user is allowed to modify the contents of the document</param-desc>
 </param>
 <param>
  <param-def>p_canmodifyannotations boolean default false</param-def>
  <param-desc>Add or modify text annotations, fill in interactive form fields.</param-desc>
 </param>
 <param>
  <param-def>p_canprint boolean default false</param-def>
  <param-desc>Print the document to a representation from which a faithful digital copy
  of the PDF content could be generated.</param-desc>
 </param>
 <param>
  <param-def>p_canmodifyprintresolution boolean default false</param-def>
  <param-desc>The user is not allowed to print in highest quality.</param-desc>
 </param>
</params>
<return>
 <return-def>blob</return-def>
 <return-desc>result PDF document</return-desc>
</return>
*/
function AESEncrypt128(
  p_inputfile blob,
  p_opassword varchar2,
  p_upassword varchar2 default null,
  p_canassemble boolean default false,
  p_canextractcontent boolean default false,
  p_canextractforaccessibility boolean default false,
  p_canfillinform boolean default false,
  p_canmodify boolean default false,
  p_canmodifyannotations boolean default false,
  p_canprint boolean default false,
  p_canmodifyprintresolution boolean default false
  ) return blob;


/**
<h2>PublicKeyEncrypt40</h2>
<type>Function</type>
<desc>
  Encrypt any PDF document using standard PDF 40 RC4 bit encryption and public key certificate.
</desc>
<version>v2.0.0</version>
<params>
 <param>
  <param-def>p_inputfile blob</param-def>
  <param-desc>source PDF file</param-desc>
 </param>
 <param>
  <param-def>p_certfile blob</param-def>
  <param-desc>the certificate file</param-desc>
 </param>
 <param>
  <param-def>p_canextractcontent boolean default false</param-def>
  <param-desc>The user is allowed to copy or otherwise
    extract text and graphics from the document, including extracting text and graphics
    (in support of accessibility to disabled users or for other purposes).
  </param-desc>
 </param>
 <param>
  <param-def>p_canmodify boolean default false</param-def>
  <param-desc>The user is allowed to modify the contents of the document.</param-desc>
 </param>
 <param>
  <param-def>p_canmodifyannotations boolean default false</param-def>
  <param-desc>The user is allowed to add or modify text annotations,
    fill in interactive form fields, and, if "modify" is also set, create or modify interactive form fields
    (including signature fields).
  </param-desc>
 </param>
 <param>
  <param-def>p_canprint boolean default false</param-def>
  <param-desc>The user is allowed to print the document.</param-desc>
 </param>
</params>
<return>
 <return-def>blob</return-def>
 <return-desc>result PDF document</return-desc>
</return>
*/
function PublicKeyEncrypt40(
  p_inputfile blob,
  p_certfile blob,
  p_canextractcontent boolean default false,
  p_canmodify boolean default false,
  p_canmodifyannotations boolean default false,
  p_canprint boolean default false
  ) return blob;

/**
<h2>PublicKeyEncrypt128</h2>
<type>Function</type>
<desc>
  Encrypt any PDF document using standard PDF 128 RC4 bit encryption and public key certificate.
</desc>
<version>v2.0.0</version>
<params>
 <param>
  <param-def>p_inputfile blob</param-def>
  <param-desc>source PDF file</param-desc>
 </param>
 <param>
  <param-def>p_certfile blob</param-def>
  <param-desc>the certificate file</param-desc>
 </param>
 <param>
  <param-def>p_canassemble boolean default false</param-def>
  <param-desc>The user is allowed to assemble the document
   (insert, rotate, or delete pages and create bookmarks or thumbnail images).
  </param-desc>
 </param>
 <param>
  <param-def>p_canextractcontent boolean default false</param-def>
  <param-desc>The user is allowed to copy or otherwise
    extract text and graphics from the document.
  </param-desc>
 </param>
 <param>
  <param-def>p_canextractforaccessibility boolean default false</param-def>
  <param-desc>The user is allowed to extract text and graphics
   (in support of accessibility to disabled users or for other purposes).
  </param-desc>
 </param>
 <param>
  <param-def>p_canfillinform boolean default false</param-def>
  <param-desc>The user is allowed to fill in existing interactive form fields
   (including signature fields)
  </param-desc>
 </param>
 <param>
  <param-def>p_canmodify boolean default false</param-def>
  <param-desc>The user is allowed to modify the contents of the document</param-desc>
 </param>
 <param>
  <param-def>p_canmodifyannotations boolean default false</param-def>
  <param-desc>Add or modify text annotations, fill in interactive form fields.</param-desc>
 </param>
 <param>
  <param-def>p_canprint boolean default false</param-def>
  <param-desc>Print the document to a representation from which a faithful digital copy
  of the PDF content could be generated.</param-desc>
 </param>
 <param>
  <param-def>p_canmodifyprintresolution boolean default false</param-def>
  <param-desc>The user is not allowed to print in highest quality.</param-desc>
 </param>
</params>
<return>
 <return-def>blob</return-def>
 <return-desc>result PDF document</return-desc>
</return>
*/
function PublicKeyEncrypt128(
  p_inputfile blob,
  p_certfile blob,
  p_canassemble boolean default false,
  p_canextractcontent boolean default false,
  p_canextractforaccessibility boolean default false,
  p_canfillinform boolean default false,
  p_canmodify boolean default false,
  p_canmodifyannotations boolean default false,
  p_canprint boolean default false,
  p_canmodifyprintresolution boolean default false
  ) return blob;

/**
<h2>Decrypt</h2>
<type>Function</type>
<desc>
  Decrypt an encrypted (40 or 128 bits) PDF document.
</desc>
<version>v2.0.0</version>
<params>
 <param>
  <param-def>p_inputfile blob</param-def>
  <param-desc>source PDF file</param-desc>
 </param>
 <param>
  <param-def>p_password varchar2</param-def>
  <param-desc>the owner password to the PDF file</param-desc>
 </param>
 </params>
<return>
  <return-def>blob</return-def>
  <return-desc>result PDF document</return-desc>
</return>
*/
function Decrypt(
  p_inputfile blob,
  p_password varchar2
  ) return blob;

/**
<h2>conv</h2>
<type>Function</type>
<desc>
  Util function for convert a measure to point (pt).
</desc>
<version>v2.0.0</version>
<params>
 <param>
  <param-def>p_mes number</param-def>
  <param-desc>original measure</param-desc>
  <param-ul>
   <param-li>pt: point</param-li>
   <param-li>mm: millimeter</param-li>
   <param-li>cm: centimeter</param-li>
   <param-li>in: inch</param-li>
  </param-ul>
 </param>
 <param>
  <param-def>p_unit varchar2</param-def>
  <param-desc>original unit of measure</param-desc>
 </param>
</params>
<return>
 <return-def>number</return-def>
 <return-desc>measure in point</return-desc>
</return>
*/
function conv(
  p_mes number,
  p_unit varchar2   --pt,mm,cm,in
  ) return number;  --pt

/**
<h2>ExtractText</h2>
<type>Function</type>
<desc>
  Extract text content of any PDF document.
</desc>
<version>v2.0.0</version>
<params>
 <param>
  <param-def>p_inputfile blob</param-def>
  <param-desc>source PDF file</param-desc>
 </param>
 <param>
  <param-def>p_encoding varchar2 default null</param-def>
  <param-desc>encoding of result text file (case-sensitive) see <a>http://java.sun.com/j2se/1.4.2/docs/guide/intl/encoding.doc.html</a>
  </param-desc>
 </param>
 <param>
  <param-def>p_startPage number default null</param-def>
  <param-desc>the start page to extract.</param-desc>
 </param>
 <param>
  <param-def>p_endPage number default null</param-def>
  <param-desc>the end page to extract.</param-desc>
 </param>
 <param>
  <param-def>p_password varchar2</param-def>
  <param-desc>the owner password of the PDF file (if encrypted).</param-desc>
 </param>
</params>
<return>
 <return-def>blob</return-def>
 <return-desc>result TEXT document</return-desc>
</return>
*/
function ExtractText(
  p_inputfile blob,
  p_encoding varchar2 default null,
  p_startPage number default null,
  p_endPage number default null,
  p_password varchar2 default null
  ) return blob;

--v2.0.7
function ExtractFormXObjectText(
  p_inputfile blob,
  p_encoding varchar2 default null,
  p_startPage number default null,
  p_endPage number default null,
  p_password varchar2 default null
  ) return blob;

/**
<h2>Merge</h2>
<type>Function</type>
<desc>
Merge two PDF documents to a new file.
</desc>
<version>v2.0.0</version>
<params>
 <param>
  <param-def>p_inputfile1 blob</param-def>
  <param-desc>source PDF file</param-desc>
 </param>
 <param>
  <param-def>p_inputfile2 blob</param-def>
  <param-desc>source PDF file</param-desc>
 </param>
 <param>
  <param-def>p_password1 varchar2 default null</param-def>
  <param-desc>the owner password of the PDF file (if encrypted).</param-desc>
 </param>
 <param>
  <param-def>p_password2 varchar2 default null</param-def>
  <param-desc>the owner password of the PDF file (if encrypted).</param-desc>
 </param>
</params>
<return>
 <return-def>blob</return-def>
 <return-desc>result PDF document</return-desc>
</return>
*/
function Merge(
  p_inputfile1 blob,
  p_inputfile2 blob,
  p_password1 varchar2 default null,
  p_password2 varchar2 default null
  ) return blob;

/**
<h2>MergeX</h2>
<type>Function</type>
<desc>
Merge PDF documents to a new file.
</desc>
<version>v2.0.0</version>
<params>
 <param>
  <param-def>p_inputfiles t_merge_inputs</param-def>
  <param-desc>source PDF files</param-desc>
 </param>
 <param>
  <param-def>p_x number default 0</param-def>
  <param-desc>X coordinate (top left) for the numbering, -1:center, -2: right
   (unit in plpdf_toolkit_par.GetDefaultUOM function)</param-desc>
 </param>
 <param>
  <param-def>p_y number default 0</param-def>
  <param-desc>Y coordinate (top left) for the numbering, -1:center, -2: bottom
   (unit in plpdf_toolkit_par.GetDefaultUOM function)</param-desc>
 </param>
 <param>
  <param-def> p_font t_14font</param-def>
  <param-desc>Standard Base 14 Font</param-desc>
 </param>
 <param>
  <param-def> p_fontSize number</param-def>
  <param-desc> font size in point</param-desc>
 </param>
 <param>
  <param-def> p_color t_color default null</param-def>
  <param-desc>color of text
   see <a>http://www.forret.com/tools/color_palette.asp</a>
  </param-desc>
 </param>
 <param>
  <param-def>p_startNumber number default 1</param-def>
  <param-desc>Start number of the page numbering</param-desc>
 </param>
 <param>
  <param-def>p_beginNumber number default 1</param-def>
  <param-desc>Start page index from which you want to start the page numbering</param-desc>
 </param>
 <param>
  <param-def>p_nbAlias varchar2 default null</param-def>
  <param-desc>Page number alias, example 'nb'</param-desc>
 </param>
 <param>
  <param-def>p_cpAlias varchar2 default null</param-def>
  <param-desc>Current page number alias, example 'cp'</param-desc>
 </param>
 <param>
  <param-def>p_format varchar2 default null</param-def>
  <param-desc>Page numbering format, example 'cp / nb'</param-desc>
 </param>
</params>
<return>
 <return-def>blob</return-def>
 <return-desc>result PDF document</return-desc>
</return>
*/
function MergeX(
  p_inputfiles t_merge_inputs,
  p_x number default 0, -- -1:center, -2:right
  p_y number default 0, -- -1:center, -2: bottom
  p_font t_14font,
  p_fontSize number,
  p_color t_color default null,
  p_startNumber number default 1,
  p_beginNumber number default 1,
  p_nbAlias varchar2 default null,--LL20110920
  p_cpAlias varchar2 default null, --LL20110920
  p_format varchar2 default null  --LL20110920
  ) return blob;

/**
<h2>MergeX</h2>
<type>Function</type>
<desc>
Merge PDF documents to a new file. PDF files stored in plpdf_tk_merge_inputs table.
</desc>
<version>v2.0.0</version>
<params>
 <param>
  <param-def>p_tr_id number</param-def>
  <param-desc>Transaction ID of the source PDF files</param-desc>
 </param>
 <param>
  <param-def>p_x number default 0</param-def>
  <param-desc>X coordinate (top left) for the numbering, -1:center, -2: right
   (unit in plpdf_toolkit_par.GetDefaultUOM function)</param-desc>
 </param>
 <param>
  <param-def>p_y number default 0</param-def>
  <param-desc>Y coordinate (top left) for the numbering, -1:center, -2: bottom
   (unit in plpdf_toolkit_par.GetDefaultUOM function)</param-desc>
 </param>
 <param>
  <param-def> p_font t_14font</param-def>
  <param-desc>Standard Base 14 Font</param-desc>
 </param>
 <param>
  <param-def> p_fontSize number</param-def>
  <param-desc> font size in point</param-desc>
 </param>
 <param>
  <param-def> p_color t_color default null</param-def>
  <param-desc>color of text
   see <a>http://www.forret.com/tools/color_palette.asp</a>
  </param-desc>
 </param>
 <param>
  <param-def>p_startNumber number default 1</param-def>
  <param-desc>Start number of the page numbering</param-desc>
 </param>
 <param>
  <param-def>p_beginNumber number default 1</param-def>
  <param-desc>Start page index from which you want to start the page numbering</param-desc>
 </param>
 <param>
  <param-def>p_nbAlias varchar2 default null</param-def>
  <param-desc>Page number alias, example 'nb'</param-desc>
 </param>
 <param>
  <param-def>p_cpAlias varchar2 default null</param-def>
  <param-desc>Current page number alias, example 'cp'</param-desc>
 </param>
 <param>
  <param-def>p_format varchar2 default null</param-def>
  <param-desc>Page numbering format, example 'cp / nb'</param-desc>
 </param>
</params>
<return>
 <return-def>blob</return-def>
 <return-desc>result PDF document</return-desc>
</return>
*/
function MergeX(
  p_tr_id number,
  p_x number default 0, -- -1:center, -2:right
  p_y number default 0, -- -1:center, -2: bottom
  p_font t_14font,
  p_fontSize number,
  p_color t_color default null,
  p_startNumber number default 1,
  p_beginNumber number default 1,
  p_nbAlias varchar2 default null,
  p_cpAlias varchar2 default null,
  p_format varchar2 default null
  ) return blob;

--20170605
function MergeXTOC(
  p_tr_id number,
  p_pageNum number,
  p_x number,
  p_y number,
  p_tocItemHeight number,
  p_tocItemWidth number,
  p_tocSeparator varchar2,
  p_font t_14font,
  p_fontSize number,
  p_color t_color default null
  ) return blob;

/**
<h2>MergeEncrypt128</h2>
<type>Function</type>
<desc>
  Merge two PDF files and encrypt using standard PDF 128 RC4 bit encryption.
</desc>
<version>v2.0.5</version>
<params>
 <param>
  <param-def>p_inputfile1 blob</param-def>
  <param-desc>source PDF file</param-desc>
 </param>
 <param>
  <param-def>p_inputfile2 blob</param-def>
  <param-desc>source PDF file</param-desc>
 </param>
 <param>
  <param-def>p_password1 varchar2 default null</param-def>
  <param-desc>the owner password of the PDF file (if encrypted).</param-desc>
 </param>
 <param>
  <param-def>p_password2 varchar2 default null</param-def>
  <param-desc>the owner password of the PDF file (if encrypted).</param-desc>
 </param>
 <param>
  <param-def>p_opassword varchar2</param-def>
  <param-desc>the owner password to the PDF file</param-desc>
 </param>
 <param>
  <param-def>p_upassword varchar2 default null</param-def>
  <param-desc>the user password to the PDF file</param-desc>
 </param>
 <param>
  <param-def>p_canassemble boolean default false</param-def>
  <param-desc>The user is allowed to assemble the document
   (insert, rotate, or delete pages and create bookmarks or thumbnail images).
  </param-desc>
 </param>
 <param>
  <param-def>p_canextractcontent boolean default false</param-def>
  <param-desc>The user is allowed to copy or otherwise
    extract text and graphics from the document.
  </param-desc>
 </param>
 <param>
  <param-def>p_canextractforaccessibility boolean default false</param-def>
  <param-desc>The user is allowed to extract text and graphics
   (in support of accessibility to disabled users or for other purposes).
  </param-desc>
 </param>
 <param>
  <param-def>p_canfillinform boolean default false</param-def>
  <param-desc>The user is allowed to fill in existing interactive form fields
   (including signature fields)
  </param-desc>
 </param>
 <param>
  <param-def>p_canmodify boolean default false</param-def>
  <param-desc>The user is allowed to modify the contents of the document</param-desc>
 </param>
 <param>
  <param-def>p_canmodifyannotations boolean default false</param-def>
  <param-desc>Add or modify text annotations, fill in interactive form fields.</param-desc>
 </param>
 <param>
  <param-def>p_canprint boolean default false</param-def>
  <param-desc>Print the document to a representation from which a faithful digital copy
  of the PDF content could be generated.</param-desc>
 </param>
 <param>
  <param-def>p_canmodifyprintresolution boolean default false</param-def>
  <param-desc>The user is not allowed to print in highest quality.</param-desc>
 </param>
</params>
<return>
 <return-def>blob</return-def>
 <return-desc>result PDF document</return-desc>
</return>
*/
function MergeEncrypt128(
  p_inputfile1 blob,
  p_inputfile2 blob,
  p_password1 varchar2 default null,
  p_password2 varchar2 default null,
  p_opassword varchar2,
  p_upassword varchar2 default null,
  p_canassemble boolean default false,
  p_canextractcontent boolean default false,
  p_canextractforaccessibility boolean default false,
  p_canfillinform boolean default false,
  p_canmodify boolean default false,
  p_canmodifyannotations boolean default false,
  p_canprint boolean default false,
  p_canmodifyprintresolution boolean default false
  ) return blob;

/**
<h2>ReplaceUrl</h2>
<type>Function</type>
<desc>
  Replace an URL in a PDF document.
</desc>
<version>v2.0.0</version>
<params>
 <param>
  <param-def>p_inputfile blob</param-def>
  <param-desc>source PDF file</param-desc>
 </param>
 <param>
  <param-def>p_oldURI varchar2</param-def>
  <param-desc>original URL</param-desc>
 </param>
 <param>
  <param-def>p_newURI varchar2</param-def>
  <param-desc>new URL</param-desc>
 </param>
 <param>
  <param-def>p_password varchar2 default null</param-def>
  <param-desc>the owner password of the PDF file (if encrypted).</param-desc>
 </param>
</params>
<return>
 <return-def>blob</return-def>
 <return-desc>result PDF document</return-desc>
</return>
*/
function ReplaceUrl(
   p_inputfile blob,
   p_oldURI varchar2,
   p_newURI varchar2,
   p_password varchar2 default null
  ) return blob;

/**
<h2>JPGStamp</h2>
<type>Function</type>
<desc>
  Stamp content into a PDF document with a JPG image.
</desc>
<version>v2.0.0</version>
<params>
 <param>
  <param-def>p_inputfile blob</param-def>
  <param-desc>source PDF file</param-desc>
 </param>
 <param>
  <param-def>p_imagefile blob</param-def>
  <param-desc>JPG image</param-desc>
 </param>
 <param>
  <param-def>p_x number</param-def>
  <param-desc>X coordinate (top left) for the image, -1: center, -2: right
    (unit in plpdf_toolkit_par.GetDefaultUOM function)
  </param-desc>
 </param>
 <param>
  <param-def>p_y number</param-def>
  <param-desc>Y coordinate (top left) for the image, -1: center, -2: bottom
   (unit in plpdf_toolkit_par.GetDefaultUOM function)
  </param-desc>
 </param>
 <param>
  <param-def>p_width number default -1</param-def>
  <param-desc>Image width, if -1 then the image is displayed in its actual width
   (unit in plpdf_toolkit_par.GetDefaultUOM function)
  </param-desc>
 </param>
 <param>
  <param-def>p_height number default -1</param-def>
  <param-desc>Image height, if -1 then the image is displayed in its actual height
   (unit in plpdf_toolkit_par.GetDefaultUOM function)
  </param-desc>
 </param>
 <param>
  <param-def>p_password varchar2 default null</param-def>
  <param-desc>the owner password of the PDF file (if encrypted).</param-desc>
 </param>
 <param>
  <param-def>p_opacity number default 1</param-def>
  <param-desc>the constant opacity value to be used for the transparent imaging model, opacity vary from 0.0 (no contribution) to 1.0 (maximum contribution)</param-desc>
 </param>
</params>
<return>
 <return-def>blob</return-def>
 <return-desc>result PDF document</return-desc>
</return>
*/
function JPGStamp(
  p_inputfile blob,
  p_imagefile blob,
  p_x number,
  p_y number,
  p_width number default -1,
  p_height number default -1,
  p_password varchar2 default null,
  p_opacity number default 1
  ) return blob;

/*3 jpgstamp + merge*/
--v2.0.7
function JPGStamp3(
  p_inputfile1 blob,
  p_inputfile2 blob,
  p_inputfile3 blob,
  p_imagefile1 blob,
  p_imagefile2 blob,
  p_imagefile3 blob,
  p_x number,
  p_y number,
  p_width number default -1,
  p_height number default -1,
  p_password1 varchar2 default null,
  p_password2 varchar2 default null,
  p_password3 varchar2 default null,
  p_opacity number default 1
  ) return blob;

--v2.0.7
function JPGTextStamp(
  p_inputfile blob,
  --image
  p_imagefile blob,
  p_image_x number,
  p_image_y number,
  p_image_width number default -1,
  p_image_height number default -1,
  p_image_opacity number default 1,
  --text
  p_text_message varchar2,
  p_text_x number,
  p_text_y number,
  p_text_font t_14font,
  p_text_fontSize number,
  p_text_color t_color default null,
  p_text_rotation number default 0,
  p_text_opacity number default 1,
  --
  p_password varchar2 default null
  ) return blob;

/**
<h2>JPGWaterMark</h2>
<type>Function</type>
<desc>
  Watermark content into a PDF document with a JPG image.
</desc>
<version>v2.0.0</version>
<params>
 <param>
  <param-def> p_inputfile blob</param-def>
  <param-desc>source PDF file</param-desc>
 </param>
 <param>
  <param-def> p_imagefile blob</param-def>
  <param-desc>JPG image</param-desc>
 </param>
 <param>
  <param-def> p_x number</param-def>
  <param-desc>X coordinate (top left) for the image, -1:center, -2: right
   (unit in plpdf_toolkit_par.GetDefaultUOM function)
  </param-desc>
 </param>
 <param>
  <param-def> p_y number</param-def>
  <param-desc>Y coordinate (top left) for the image, -1:center, -2: bottom
   (unit in plpdf_toolkit_par.GetDefaultUOM function)
  </param-desc>
 </param>
 <param>
  <param-def> p_width number default -1</param-def>
  <param-desc>Image width, if -1 then the image is displayed in its actual width
   (unit in plpdf_toolkit_par.GetDefaultUOM function)
  </param-desc>
 </param>
 <param>
  <param-def> p_height number default -1</param-def>
  <param-desc>Image height, if -1 then the image is displayed in its actual height
   (unit in plpdf_toolkit_par.GetDefaultUOM function)
  </param-desc>
 </param>
 <param>
  <param-def> p_password varchar2 default null</param-def>
  <param-desc>the owner password of the PDF file (if encrypted).</param-desc>
 </param>
 <param>
  <param-def>p_opacity number default 1</param-def>
  <param-desc>the constant opacity value to be used for the transparent imaging model, opacity vary from 0.0 (no contribution) to 1.0 (maximum contribution)</param-desc>
 </param>
 </params>
 <return>
  <return-def>blob</return-def>
  <return-desc>result PDF document</return-desc>
 </return>
*/
function JPGWaterMark(
  p_inputfile blob,
  p_imagefile blob,
  p_x number,
  p_y number,
  p_width number default -1,
  p_height number default -1,
  p_password varchar2 default null,
  p_opacity number default 1
  ) return blob;

/**
<h2>MetadataFromXML</h2>
<type>Function</type>
<desc>Set PDF metadata from an XML file.<br/>
  an example XML file:<br/>
  <i>
  &lt;?xml version="1.0" encoding="UTF-8"?&gt;<br/>
  &lt;pdf-metadata&gt;<br/>
  &lt;title&gt;PL/PDF Toolkit - User's Guide&lt;/title&gt;<br/>
  &lt;author&gt;PLPDF&lt;/author&gt;<br/>
  &lt;subject&gt;plpdf toolkit&lt;/subject&gt;<br/>
  &lt;creator&gt;PL/PDF v2.0.0&lt;/creator&gt;<br/>
  &lt;keywords&gt;Oracle PDF&lt;/keywords&gt;<br/>
  &lt;producer&gt;www.plpdf.com&lt;/producer&gt;<br/>
  &lt;createtime&gt;20080903183019&lt;/createtime&gt;<br/>
  &lt;modificationdate&gt;20080904114023&lt;/modificationdate&gt;<br/>
  &lt;/pdf-metadata&gt;<br/>
  </i>
  createtime and modificationdate format are YYYYMMDDHH24MISS
</desc>
<version>v2.0.0</version>
<params>
 <param>
  <param-def> p_inputfile blob</param-def>
  <param-desc>source PDF file</param-desc>
 </param>
 <param>
  <param-def> p_xmlfile blob</param-def>
  <param-desc>metadata XML file</param-desc>
 </param>
 <param>
  <param-def> p_password varchar2 default null</param-def>
  <param-desc>the owner password of the PDF file (if encrypted).</param-desc>
 </param>
</params>
<return>
 <return-def>blob</return-def>
 <return-desc>result PDF document</return-desc>
</return>
*/
function MetadataFromXML(
  p_inputfile blob,
  p_xmlfile blob,
  p_password varchar2 default null
  ) return blob;

/**
<h2>MetadataFromPars</h2>
<type>Function</type>
<desc>
-
</desc>
<version>v2.0.0</version>
<params>
 <param>
  <param-def> p_inputfile blob</param-def>
  <param-desc>source PDF file</param-desc>
 </param>
 <param>
  <param-def> p_title varchar2 default null</param-def>
  <param-desc> The document?s title.</param-desc>
 </param>
 <param>
  <param-def> p_author varchar2 default null</param-def>
  <param-desc>The name of the person who created the document.</param-desc>
 </param>
 <param>
  <param-def> p_subject varchar2 default null</param-def>
  <param-desc>The subject of the document.</param-desc>
 </param>
 <param>
  <param-def> p_creator varchar2 default null</param-def>
  <param-desc>If the document was converted to PDF from another format, the
    name of the application (for example, Adobe FrameMaker?) that created the
    original document from which it was converted.
  </param-desc>
 </param>
 <param>
  <param-def> p_keywords varchar2 default null</param-def>
  <param-desc>Keywords associated with the document.</param-desc>
 </param>
 <param>
  <param-def> p_producer varchar2 default null</param-def>
  <param-desc>If the document was converted to PDF from another format, the
   name of the application (for example, Acrobat Distiller) that converted it to PDF.
  </param-desc>
 </param>
 <param>
  <param-def> p_createtime date default null</param-def>
  <param-desc>The date and time the document was created, in human-readable form.</param-desc>
 </param>
 <param>
  <param-def> p_modificationdate date default null</param-def>
  <param-desc>The date and time the document was most recently modified,
   in human-readable form.
  </param-desc>
 </param>
 <param>
  <param-def> p_password varchar2 default null</param-def>
  <param-desc>the owner password of the PDF file (if encrypted).</param-desc>
 </param>
</params>
<return>
 <return-def>blob</return-def>
 <return-desc>result PDF document</return-desc>
</return>
*/
function MetadataFromPars(
  p_inputfile blob,
  p_title varchar2 default null,
  p_author varchar2 default null,
  p_subject varchar2 default null,
  p_creator varchar2 default null,
  p_keywords varchar2 default null,
  p_producer varchar2 default null,
  p_createtime date default null,
  p_modificationdate date default null,
  p_password varchar2 default null
  ) return blob;

/**
<h2>MetadataToXML</h2>
<type>Function</type>
<desc>
  Extract PDF metadata into an XML file.<br/>
  An example XML file:<br/>
  <i>
  &lt;?xml version="1.0" encoding="UTF-8"?&gt;<br/>
  &lt;pdf-metadata&gt;<br/>
  &lt;title&gt;PL/PDF Toolkit - User's Guide&lt;/title&gt;<br/>
  &lt;author&gt;PLPDF&lt;/author&gt;<br/>
  &lt;subject&gt;plpdf toolkit&lt;/subject&gt;<br/>
  &lt;creator&gt;PL/PDF v2.0.0&lt;/creator&gt;<br/>
  &lt;keywords&gt;Oracle PDF&lt;/keywords&gt;<br/>
  &lt;producer&gt;www.plpdf.com&lt;/producer&gt;<br/>
  &lt;createtime&gt;20080903183019&lt;/createtime&gt;<br/>
  &lt;modificationdate&gt;20080904114023&lt;/modificationdate&gt;<br/>
  &lt;/pdf-metadata&gt;<br/>
  </i>
  createtime and modificationdate format are YYYYMMDDHH24MISS
</desc>
<version>v2.0.0</version>
<params>
 <param>
  <param-def> p_inputfile blob</param-def>
  <param-desc>source PDF file</param-desc>
 </param>
 <param>
  <param-def> p_password varchar2 default null</param-def>
  <param-desc>the owner password of the PDF file (if encrypted).</param-desc>
 </param>
</params>
<return>
 <return-def>blob</return-def>
 <return-desc>result XML document</return-desc>
</return>
*/
function MetadataToXML(
  p_inputfile blob,
  p_password varchar2 default null
  ) return blob;

/**
<h2>MetadataToRecord</h2>
<type>Function</type>
<desc>
 Extract PDF metadata into the t_metadata record.
</desc>
<version>v2.0.0</version>
<params>
<param>
<param-def>p_inputfile blob</param-def>
<param-desc>source PDF file</param-desc>
</param>
<param>
<param-def>p_password varchar2 default null</param-def>
<param-desc>the owner password of the PDF file (if encrypted).</param-desc>
</param>
</params>
<return>
<return-def>t_metadata</return-def>
<return-desc>PDF metadata</return-desc>
</return>
*/
function MetadataToRecord(
  p_inputfile blob,
  p_password varchar2 default null
  ) return t_metadata;

/**
<h2>TextStamp</h2>
<type>Function</type>
<desc>
  Stamp content into a PDF document with a text.
</desc>
<version>v2.0.0</version>
<params>
 <param>
  <param-def> p_inputfile blob</param-def>
  <param-desc>source PDF file</param-desc>
 </param>
 <param>
  <param-def> p_message</param-def>
  <param-desc>stamp text (encoding: plpdf_toolkit_par.GetDefaultEncoding)</param-desc>
 </param>
 <param>
  <param-def> p_x number</param-def>
  <param-desc>X coordinate (top left) for the image, -1:center, -2: right
   (unit in plpdf_toolkit_par.GetDefaultUOM function)
  </param-desc>
 </param>
 <param>
  <param-def> p_y number</param-def>
  <param-desc>Y coordinate (top left) for the image, -1:center, -2: bottom
   (unit in plpdf_toolkit_par.GetDefaultUOM function)
  </param-desc>
 </param>
 <param>
  <param-def> p_font t_14font</param-def>
  <param-desc>Standard Base 14 Font</param-desc>
 </param>
 <param>
  <param-def> p_fontSize number</param-def>
  <param-desc> font size in point</param-desc>
 </param>
 <param>
  <param-def> p_color t_color default null</param-def>
  <param-desc>color of text
   see <a>http://www.forret.com/tools/color_palette.asp</a>
  </param-desc>
 </param>
 <param>
 <param-def> p_password varchar2 default null</param-def>
 <param-desc>the owner password of the PDF file (if encrypted).</param-desc>
 </param>
 <param>
  <param-def>p_startPage number default null</param-def>
  <param-desc>the start page to extract.</param-desc>
 </param>
 <param>
  <param-def>p_endPage number default null</param-def>
  <param-desc>the end page to extract.</param-desc>
 </param>
 <param>
  <param-def>p_rotation number default 0</param-def>
  <param-desc>Text rotation angle</param-desc>
 </param>
 <param>
  <param-def>p_opacity number default 1</param-def>
  <param-desc>the constant opacity value to be used for the transparent imaging model, opacity vary from 0.0 (no contribution) to 1.0 (maximum contribution)</param-desc>
 </param>
</params>
<return>
 <return-def>blob</return-def>
 <return-desc>result PDF document</return-desc>
</return>
*/
function TextStamp(
  p_inputfile blob,
  p_message varchar2,
  p_x number,
  p_y number,
  p_font t_14font,
  p_fontSize number,
  p_color t_color default null,
  p_password varchar2 default null,
  p_startPage number default null,
  p_endPage number default null,
  p_rotation number default 0,
  p_opacity number default 1
  ) return blob;

/**
<h2>TextStampTTF</h2>
<type>Function</type>
<desc>
  Stamp content into a PDF document with a text (TTF font type).
</desc>
<version>v2.0.0</version>
<params>
 <param>
  <param-def> p_inputfile blob</param-def>
  <param-desc>source PDF file</param-desc>
 </param>
 <param>
  <param-def> p_message</param-def>
  <param-desc>stamp text (encoding: plpdf_toolkit_par.GetDefaultEncoding)</param-desc>
 </param>
 <param>
  <param-def> p_x number</param-def>
  <param-desc>X coordinate (top left) for the image, -1:center, -2: right
   (unit in plpdf_toolkit_par.GetDefaultUOM function)
  </param-desc>
 </param>
 <param>
  <param-def> p_y number</param-def>
  <param-desc>Y coordinate (top left) for the image, -1:center, -2: bottom
   (unit in plpdf_toolkit_par.GetDefaultUOM function)
  </param-desc>
 </param>
 <param>
  <param-def> p_font blob</param-def>
  <param-desc>TTF (TrueType Font) file</param-desc>
 </param>
 <param>
  <param-def> p_fontSize number</param-def>
  <param-desc>font size in point</param-desc>
 </param>
 <param>
  <param-def> p_color t_color default null</param-def>
  <param-desc>color of text
   see <a>http://www.forret.com/tools/color_palette.asp</a>
  </param-desc>
 </param>
 <param>
 <param-def> p_password varchar2 default null</param-def>
 <param-desc>the owner password of the PDF file (if encrypted).</param-desc>
 </param>
 <param>
  <param-def>p_startPage number default null</param-def>
  <param-desc>the start page to extract.</param-desc>
 </param>
 <param>
  <param-def>p_endPage number default null</param-def>
  <param-desc>the end page to extract.</param-desc>
 </param>
 <param>
  <param-def>p_rotation number default 0</param-def>
  <param-desc>Text rotation angle</param-desc>
 </param>
 <param>
  <param-def>p_opacity number default 1</param-def>
  <param-desc>the constant opacity value to be used for the transparent imaging model, opacity vary from 0.0 (no contribution) to 1.0 (maximum contribution)</param-desc>
 </param>
</params>
<return>
 <return-def>blob</return-def>
 <return-desc>result PDF document</return-desc>
</return>
*/
function TextStampTTF(
  p_inputfile blob,
  p_message varchar2,
  p_x number,
  p_y number,
  p_font blob,
  p_fontSize number,
  p_color t_color default null,
  p_password varchar2 default null,
  p_startPage number default null,
  p_endPage number default null,
  p_rotation number default 0,
  p_opacity number default 1
  ) return blob;

/**
<h2>TextWaterMark</h2>
<type>Function</type>
<desc>
  Watermark content into a PDF document with a text.
</desc>
<version>v2.0.0</version>
<params>
 <param>
  <param-def>p_inputfile blob</param-def>
  <param-desc>source PDF file</param-desc>
 </param>
 <param>
  <param-def> p_message</param-def>
  <param-desc>watermark text (encoding: plpdf_toolkit_par.GetDefaultEncoding)</param-desc>
 </param>
 <param>
  <param-def> p_x number</param-def>
  <param-desc>X coordinate (top left) for the image, -1:center, -2: right
   (unit in plpdf_toolkit_par.GetDefaultUOM function)
  </param-desc>
 </param>
 <param>
  <param-def> p_y number</param-def>
  <param-desc>Y coordinate (top left) for the image, -1:center, -2: bottom
   (unit in plpdf_toolkit_par.GetDefaultUOM function)
  </param-desc>
 </param>
 <param>
  <param-def> p_font t_14font</param-def>
  <param-desc>Standard Base 14 Font</param-desc>
 </param>
 <param>
  <param-def> p_fontSize number</param-def>
  <param-desc>font size in point</param-desc>
 </param>
 <param>
  <param-def> p_color plpdf_type.t_color default null</param-def>
  <param-desc>color of text see <a>http://www.forret.com/tools/color_palette.asp</a></param-desc>
 </param>
 <param>
  <param-def> p_password varchar2 default null</param-def>
  <param-desc>the owner password of the PDF file (if encrypted).</param-desc>
 </param>
 <param>
  <param-def>p_rotation number default 0</param-def>
  <param-desc>Text rotation angle</param-desc>
 </param>
 <param>
  <param-def>p_opacity number default 1</param-def>
  <param-desc>the constant opacity value to be used for the transparent imaging model, opacity vary from 0.0 (no contribution) to 1.0 (maximum contribution)</param-desc>
 </param>
 <param>
  <param-def>p_reencrypt boolean default false</param-def>
  <param-desc>Reencrypt document if encrypted.</param-desc>
 </param>
 </params>
<return>
 <return-def>blob</return-def>
 <return-desc>result PDF document</return-desc>
</return>
*/
function TextWaterMark(
  p_inputfile blob,
  p_message varchar2,
  p_x number,
  p_y number,
  p_font t_14font,
  p_fontSize number,
  p_color t_color default null,
  p_password varchar2 default null,
  p_rotation number default 0,
  p_opacity number default 1,
  p_reencrypt boolean default false
  ) return blob;

/**
<h2>PDFWatermark</h2>
<type>Function</type>
<desc>
  Overlay on PDF document with another one PDF.
</desc>
<version>v2.0.0</version>
<params>
 <param>
  <param-def> p_inputfile blob</param-def>
  <param-desc>source PDF file</param-desc>
 </param>
 <param>
  <param-def> p_overlayinputfile blob</param-def>
  <param-desc>overlay PDF file</param-desc>
 </param>
 <param>
  <param-def> p_overlaypassword varchar2 default null</param-def>
  <param-desc>the owner password of the PDF file (if encrypted).</param-desc>
 </param>
 <param>
  <param-def> p_password varchar2 default null</param-def>
  <param-desc>the owner password of the PDF file (if encrypted).</param-desc>
 </param>
</params>
<return>
 <return-def>blob</return-def>
 <return-desc>result PDF document</return-desc>
</return>
*/

function PDFWatermark(
  p_inputfile blob,
  p_overlayinputfile blob,
  p_password varchar2 default null,
  p_overlaypassword varchar2 default null
  ) return blob;

/**
<name>ExportFDF</name>
<type>Function</type>
<desc>
  Export PDF Form data as an FDF file.
</desc>
<version>v2.0.0</version>
<params>
 <param>
  <param-def>p_inputfile blob</param-def>
  <param-desc>source PDF file</param-desc>
 </param>
 <param>
  <param-def>p_password varchar2</param-def>
  <param-desc>the owner password to the PDF file</param-desc>
 </param>
</params>
<return>
 <return-def>blob</return-def>
 <return-desc>FDF document</return-desc>
</return>
*/
function ExportFDF(
  p_inputfile blob,
  p_password varchar2 default null
  ) return blob;

/**
<name>ExportXFDF</name>
<type>Function</type>
<desc>
  Export PDF Form data as an XFDF file.
</desc>
<version>v2.0.0</version>
<params>
 <param>
  <param-def>p_inputfile blob</param-def>
  <param-desc>source PDF file</param-desc>
 </param>
 <param>
  <param-def>p_password varchar2</param-def>
  <param-desc>the owner password to the PDF file</param-desc>
 </param>
 <param>
  <param-def>p_exportnull boolean default true</param-def>
  <param-desc>export empty fields</param-desc>
 </param>
</params>
<return>
 <return-def>blob</return-def>
 <return-desc>XFDF document</return-desc>
</return>
*/
function ExportXFDF(
  p_inputfile blob,
  p_password varchar2 default null,
  p_exportnull boolean default true
  ) return blob;

/**
<name>ImportFDF</name>
<type>Function</type>
<desc>
  Import an FDF file into the PDF Form's fields.
</desc>
<version>v2.0.0</version>
<params>
 <param>
  <param-def>p_inputfile blob</param-def>
  <param-desc>source PDF file</param-desc>
 </param>
 <param>
  <param-def>p_fdffile blob</param-def>
  <param-desc>FDF file</param-desc>
 </param>
 <param>
  <param-def>p_password varchar2</param-def>
  <param-desc>the owner password to the PDF file</param-desc>
 </param>
</params>
<return>
 <return-def>blob</return-def>
 <return-desc>FDF document</return-desc>
</return>
*/
function ImportFDF(
  p_inputfile blob,
  p_fdffile blob,
  p_password varchar2 default null
  ) return blob;

/**
<name>ImportXFDF</name>
<type>Function</type>
<desc>
  Import an XFDF file into the PDF Form's fields.
</desc>
<version>v2.0.0</version>
<params>
 <param>
  <param-def>p_inputfile blob</param-def>
  <param-desc>source PDF file</param-desc>
 </param>
 <param>
  <param-def>p_xfdffile blob</param-def>
  <param-desc>XFDF file</param-desc>
 </param>
 <param>
  <param-def>p_password varchar2</param-def>
  <param-desc>the owner password to the PDF file</param-desc>
 </param>
 <param>
  <param-def>p_needapp boolean default true</param-def>
  <param-desc>Set NeedAppearances option in PDF source.
	When NeedAppearances is true then force compliant PDF readers to render the fields.
	If false then toolkit automatically generates the appearance streams for all form fields.
  </param-desc>
 </param>
</params>
<return>
 <return-def>blob</return-def>
 <return-desc>XFDF document</return-desc>
</return>
*/
function ImportXFDF(
  p_inputfile blob,
  p_xfdffile blob,
  p_password varchar2 default null,
  p_needapp boolean default true
  ) return blob;

/**
<h2>ExtractPages</h2>
<type>Function</type>
<desc>
  Extract pages of any PDF document into a new PDF.
</desc>
<version>v2.0.0</version>
<params>
 <param>
  <param-def>p_inputfile blob</param-def>
  <param-desc>source PDF file</param-desc>
 </param>
 <param>
  <param-def>p_startPage number default null</param-def>
  <param-desc>the start page to extract.</param-desc>
 </param>
 <param>
  <param-def>p_endPage number default null</param-def>
  <param-desc>the end page to extract.</param-desc>
 </param>
 <param>
  <param-def>p_password varchar2</param-def>
  <param-desc>the owner password of the PDF file (if encrypted).</param-desc>
 </param>
</params>
<return>
 <return-def>blob</return-def>
 <return-desc>result PDF document</return-desc>
</return>
*/
function ExtractPages(
  p_inputfile blob,
  p_startPage number default null,
  p_endPage number default null,
  p_password varchar2 default null
  ) return blob;

/**
<h2>GetNumberOfPages</h2>
<type>Function</type>
<desc>
  Get number of pages of any PDF document.
</desc>
<version>v2.0.0</version>
<params>
 <param>
  <param-def>p_inputfile blob</param-def>
  <param-desc>source PDF file</param-desc>
 </param>
 <param>
  <param-def>p_password varchar2</param-def>
  <param-desc>the owner password of the PDF file (if encrypted).</param-desc>
 </param>
</params>
<return>
 <return-def>number</return-def>
 <return-desc>number of pages</return-desc>
</return>
*/
function GetNumberOfPages(
  p_inputfile blob,
  p_password varchar2 default null
  ) return number;

/**
<h2>JPGCompress</h2>
<type>Function</type>
<desc>
  Sets the compression quality to a value between 0  and 1 of the JPG images in a PDF document.
</desc>
<version>v2.0.0</version>
<params>
 <param>
  <param-def>p_inputfile blob</param-def>
  <param-desc>source PDF file</param-desc>
 </param>
 <param>
  <param-def>p_compress number</param-def>
  <param-desc>compression quality to a value between 0  and 1</param-desc>
 </param>
 <param>
  <param-def>p_password varchar2 default null</param-def>
  <param-desc>the owner password of the PDF file (if encrypted).</param-desc>
 </param>
</params>
<return>
 <return-def>blob</return-def>
 <return-desc>result PDF document</return-desc>
</return>
*/
function JPGCompress(
   p_inputfile blob,
   p_compress number,
   p_password varchar2 default null
  ) return blob;


/**
<h2>Rebuild</h2>
<type>Function</type>
<desc>
  Rebuild any PDF document into a new PDF.
</desc>
<version>v2.0.0</version>
<params>
 <param>
  <param-def>p_inputfile blob</param-def>
  <param-desc>source PDF file</param-desc>
 </param>
 <param>
  <param-def>p_password varchar2</param-def>
  <param-desc>the owner password of the PDF file (if encrypted).</param-desc>
 </param>
</params>
<return>
 <return-def>blob</return-def>
 <return-desc>result PDF document</return-desc>
</return>
*/
function Rebuild(
  p_inputfile blob,
  p_password varchar2 default null
  ) return blob;

/**
<name>Get_Version</name>
<type>Function</type>
<desc>
  Get version number of the toolkit.
</desc>
<version>v2.0.0</version>
<return>
 <return-def>varchar2</return-def>
 <return-desc>version number</return-desc>
</return>
*/
function get_Version return varchar2;

/**
<name>Get_VersionDate</name>
<type>Function</type>
<desc>
  Get release date of the toolkit.
</desc>
<version>v2.0.0</version>
<return>
 <return-def>varchar2</return-def>
 <return-desc>release date</return-desc>
</return>
*/
function get_VersionDate return varchar2;


/**
<h2>setFieldsReadonly</h2>
<type>Function</type>
<desc>
  Set the PDF form fields to read-only.
</desc>
<version>v2.0.1</version>
<params>
 <param>
  <param-def>p_inputfile blob</param-def>
  <param-desc>source PDF file</param-desc>
 </param>
 <param>
  <param-def>p_password varchar2</param-def>
  <param-desc>the owner password of the PDF file (if encrypted).</param-desc>
 </param>
 <param>
  <param-def>p_raise_noform boolean default null</param-def>
  <param-desc>Raise no form error or not</param-desc>
 </param>
</params>
<return>
 <return-def>blob</return-def>
 <return-desc>result PDF document</return-desc>
</return>
*/
function setFieldsReadonly(
  p_inputfile blob,
  p_password varchar2 default null,
  p_raise_noform boolean default null
  ) return blob;


/**
<h2>getPDFACompliant</h2>
<type>Function</type>
<desc>
  Get the PDF/A Compliant information.
</desc>
<version>v2.0.1</version>
<params>
 <param>
  <param-def>p_inputfile blob</param-def>
  <param-desc>source PDF file</param-desc>
 </param>
 <param>
  <param-def>p_password varchar2</param-def>
  <param-desc>the owner password of the PDF file (if encrypted).</param-desc>
 </param>
</params>
<return>
 <return-def>The Compliant info</return-def>
 <return-desc>Format is part + conf, example 1B</return-desc>
</return>
*/
function getPDFACompliant(
  p_inputfile blob,
  p_password varchar2 default null
  ) return varchar2;


-- end of package

end;

/

CREATE OR REPLACE PACKAGE BODY plpdf_toolkit wrapped 
a000000
b2
abcd
abcd
abcd
abcd
abcd
abcd
abcd
abcd
abcd
abcd
abcd
abcd
abcd
abcd
abcd
b
11454 2676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/
