create or replace package plpdf
authid current_user as

/**
 <config>
  <pck-name>PL/PDF</pck-name>
  <pck-title>User's Guide</pck-title>
  <pck-version>v5.19</pck-version>
  <header-prc>plpdf_doc_xml.plpdf_header</header-prc>
  <header-size>10</header-size>
  <footer-prc>plpdf_doc_xml.plpdf_footer</footer-prc>
  <footer-size>10</footer-size>
 </config>
*/

/**
 <h1>Introduction</h1>
 <br/>
 <p>
 PL/PDF is a PL/SQL based PDF document generator from Oracle databases. Use PL/PDF to quickly and easily develop applications with dynamic content but also quality presentation and printing capabilities.
 </p>
 <br/>
 <h2>Installation</h2>
 <p>
 Before using PL/PDF, make sure that the steps described in the Installation Guide were completed.
 </p>
 <br/>
 <h2>Using PL/PDF</h2>
 <p>
 Essentially, PL/PDF is a series of calls to PL/PDF procedures and functions from a PL/SQL script, package, procedure or function. If you are not familiar with the PL/SQL programming language, study the Oracle documentation.
 </p>
 <br/>
 <h2>Documentation</h2>
 <p>
 The PL/PDF documentation is available at: <a>http://www.plpdf.com/downloads</a>
 </p>
 <br/>
 <h2>Code Examples</h2>
 <p>
 PL/PDF code examples can be found at: <a>http://plpdf.com/index.php/support/examples/</a>
 </p>
 <br/>
 <h2>Contact</h2>
 <p>
 Contact us at: <a>mailto:info@plpdf.com</a>
 </p>
 <newpage/>
*/

/**
 <h1>Certification</h1>

 <name>GetCertKey</name>
 <type>Function</type>
 <desc>
  Generates preliminary certification key.
  This function should be run once when the PL/PDF package is installed. The function generates a string
  that should be sent to <a>mailto:info@plpdf.com</a>. Our representatives will send a certificate key back that should
  be used as the return value in the pl_pdf_cert.SetCertKey function.
 </desc>
 <version>v5.0</version>
 <params>-</params>
 <return>
  <return-def>varchar2</return-def>
  <return-desc>Preliminary certificate key</return-desc>
 </return>
*/
function GetCertKey
  return varchar2;

/**
 <name>checkCertKey</name>
 <type>Function</type>
 <desc>
  Checks the certification key.<br/>
  Usage:<br/>
  declare                            <br/>
    l_ret boolean;                   <br/>
  begin                              <br/>
    l_ret := plpdf.checkCertKey;     <br/>
    if l_ret then                    <br/>
      dbms_output.put_line('true');  <br/>
    else                             <br/>
      dbms_output.put_line('false'); <br/>
    end if;                          <br/>
  end;                               <br/>
 </desc>
 <version>v3.0.0</version>
 <params>-</params>
 <return>
  <return-def>boolean</return-def>
  <return-desc>validity of the key</return-desc>
 </return>
*/
function checkCertKey
  return boolean;

/**
 <name>getLicencedProductsV5</name>
 <type>Function</type>
 <desc>
  This function returns with licence informations as a string.
 </desc>
 <version>v4.1.0</version>
 <params>-</params>
 <return>
  <return-def>varchar2</return-def>
  <return-desc>Licence</return-desc>
 </return>
*/
function getLicencedProductsV5
  return varchar2;

/**
 <newpage/>
 <h1>Administrative commands</h1>
 <br/>
*/

/**
 <name>AddCJKFont</name>
 <type>Procedure</type>
 <desc>
  Add CJK (Chinese/Japanese/Korean) font.
 </desc>
 <version>v3.0.0</version>
 <params>
  <param>
   <param-def>p_name varchar2</param-def>
   <param-desc>name of font</param-desc>
   <param-ul>
    <param-li>japanese: 'HeiseiKakuGo-W5'</param-li>
    <param-li>korean: 'HYSMyeongJoStd-Medium-Acro'</param-li>
    <param-li>chinese: 'MSungStd-Light-Acro', 'STSongStd-Light-Acro'</param-li>
   </param-ul>
  </param>
 </params>
 <return>-</return>
*/
procedure AddCJKFont(
  p_name varchar2
  );

/**
 <name>AddTTF</name>
 <type>Procedure</type>
 <desc>
  Imports a TTF (True Type Font). The imported font becomes available to the SetPrintFont procedure.
  The font is stored as binary data in the PDF file.
 </desc>
 <version>v3.0.0</version>
 <params>
  <param>
   <param-def>p_family varchar2</param-def>
   <param-desc>TTF name</param-desc>
  </param>
  <param>
   <param-def>p_style varchar2 default null</param-def>
   <param-desc>Font style</param-desc>
   <param-ul>
    <param-li>null: Regular</param-li>
    <param-li>B: Bold</param-li>
    <param-li>I: Italic</param-li>
    <param-li>BI or IB: Bold and Italic</param-li>
   </param-ul>
  </param>
  <param>
   <param-def>p_data t_addfont</param-def>
   <param-desc>Data structure necessary to insert the font. It contains the font definition binary file as well</param-desc>
  </param>
  <param>
   <param-def>p_embed boolean default true</param-def>
   <param-desc>PDF containts TTF binary file or not</param-desc>
  </param>
 </params>
 <return>-</return>
*/
procedure AddTTF(
  p_family varchar2,
  p_style varchar2 default null,
  p_data Plpdf_Type.t_addfont,
  p_embed boolean default true
  );

/**
 <name>init</name>
 <type>Procedure</type>
 <desc>
  Initializes generator program variables.
 </desc>
 <version>v3.0.0</version>
 <params>
  <param>
   <param-def>p_orientation varchar2 default 'P'</param-def>
   <param-desc>Orientation</param-desc>
   <param-ul>
    <param-li>P: portrait</param-li>
    <param-li>L: landscape</param-li>
   </param-ul>
  </param>
  <param>
   <param-def>p_unit varchar2 default 'mm'</param-def>
   <param-desc>Unit of measure</param-desc>
  </param>
  <param>
   <param-def>p_format varchar2 default 'A4'</param-def>
   <param-desc>Page format</param-desc>
  </param>
 </params>
 <return>-</return>
*/
procedure init(
  p_orientation varchar2 default plpdf_const.portrait,
  p_unit varchar2 default plpdf_const.mm,
  p_format varchar2 default 'A4'
  );

/**
 <name>init1</name>
 <type>Procedure</type>
 <desc>
  Initializes generator program variables.
 </desc>
 <version>v3.0.0</version>
 <params>
  <param>
   <param-def>p_orientation varchar2 default 'P'</param-def>
   <param-desc>Orientation</param-desc>
   <param-ul>
    <param-li>P: portrait</param-li>
    <param-li>L: landscape</param-li>
   </param-ul>
  </param>
  <param>
   <param-def>p_unit varchar2 default 'mm'</param-def>
   <param-desc>Unit of measure</param-desc>
  </param>
  <param>
   <param-def>p_format plpdf_type.t_pageformat</param-def>
   <param-desc>see plpdf_const package for predefined pageformats</param-desc>
  </param>
 </params>
 <return>-</return>
*/
procedure init1(
  p_orientation varchar2 default plpdf_const.portrait,
  p_unit varchar2 default plpdf_const.mm,
  p_format plpdf_type.t_pageformat
  );

/**
 <name>getVersion</name>
 <type>Function</type>
 <desc>
  Gets the version number of pl/pldf package.
 </desc>
 <version>v3.0.0</version>
 <params>-</params>
 <return>
  <return-def>varchar2</return-def>
  <return-desc>number of current version</return-desc>
 </return>
*/
function getVersion
  return varchar2;

/**
 <name>setRotatePages</name>
 <type>Procedure</type>
 <desc>
  Sets the degree of rotation for all pages. The value must be a multiple of 90.
 </desc>
 <version>v3.0.0</version>
 <params>
  <param>
   <param-def>p_angle number default 0</param-def>
   <param-desc>Angle/degree of rotation</param-desc>
  </param>
 </params>
 <return>-</return>
*/
procedure setRotatePages(
  p_angle number default 0
  );

/**
 <name>sendDoc</name>
 <type>Procedure</type>
 <desc>
  Returns the generated PDF document. The document is closed and then returned in the OUT parameter.
 </desc>
 <version>v4.8.3</version>
 <params>
 <param>
  <param-def>p_blob out nocopy blob</param-def>
  <param-desc>the generated document</param-desc>
 </param>
 <param>
  <param-def>p_freetemp boolean default true</param-def>
  <param-desc>frees the temporary BLOB variable of the PDF file in your default temporary tablespace</param-desc>
 </param>
 </params>
 <return>-</return>
*/
procedure sendDoc(
  p_blob out nocopy blob,
  p_freetemp boolean default true
  );

/**
 <name>setAllMargin</name>
 <type>Procedure</type>
 <desc>
 Sets the margin size.
 </desc>
 <version>v3.0.0</version>
 <params>
  <param>
   <param-def>p_left number</param-def>
   <param-desc>Left margin size</param-desc>
  </param>
  <param>
   <param-def>p_top number</param-def>
   <param-desc>Top margin size</param-desc>
  </param>
  <param>
   <param-def>p_right number  default null</param-def>
   <param-desc>Right margin size. If null then the right margin will be the same as the left margin</param-desc>
  </param>
 </params>
 <return>-</return>
*/
procedure setAllMargin(
  p_left number,
  p_top number,
  p_right number  default null
  );

/**
 <name>setAutoNewPage</name>
 <type>Procedure</type>
 <desc>
 Enables or disables automatic page breaks. The p_margin parameter is
 only relevant when enabling automatic page breaks. It sets the bottom
 margin size under which an automatic page break is triggered.
 </desc>
 <version>v3.0.0</version>
 <params>
  <param>
   <param-def>p_auto boolean default true</param-def>
   <param-desc>Automatic page break</param-desc>
   <param-ul>
    <param-li>true: Enable</param-li>
    <param-li>false: Disable</param-li>
   </param-ul>
  </param>
  <param>
   <param-def>p_margin number default 0</param-def>
   <param-desc>Bottom margin size, if not set, then 1 cm</param-desc>
  </param>
 </params>
 <return>-</return>
*/
procedure setAutoNewPage(
  p_auto boolean default true,
  p_margin number default 0
  );

/**
 <name>setNOCompress</name>
 <type>Procedure</type>
 <desc>
  Disables the compression of PDF documents. By default compression is enabled.
 </desc>
 <version>v3.0.0</version>
 <return>-</return>
*/
procedure setNOCompress;

/**
 <name>setDocAuthor</name>
 <type>Procedure</type>
 <desc>
  Sets the document author.
 </desc>
 <version>v3.0.0</version>
 <params>
  <param>
   <param-def>p_author varchar2</param-def>
   <param-desc>Document author</param-desc>
  </param>
 </params>
 <return>-</return>
*/
procedure setDocAuthor(
  p_author varchar2
  );

/**
 <name>setDocCreator</name>
 <type>Procedure</type>
 <desc>
  Sets the document creator.
 </desc>
 <version>v3.0.0</version>
 <params>
  <param>
   <param-def>p_creator varchar2</param-def>
   <param-desc>Document creator</param-desc>
  </param>
 </params>
 <return>-</return>
*/
procedure setDocCreator(
  p_creator varchar2
  );

/**
 <name>setDocDisplayMode</name>
 <type>Procedure</type>
 <desc>
  Sets how to display the document in a PDF viewer (e.g. Acrobat Reader).
 </desc>
 <version>v3.0.0</version>
 <params>
  <param>
   <param-def>p_zoom varchar2</param-def>
   <param-desc>Layout view</param-desc>
   <param-ul>
    <param-li>fullpage: fits as full page</param-li>
    <param-li>fullwidth: fits as full width</param-li>
    <param-li>real: fits in actual size</param-li>
    <param-li>default: displays according to the setting of the PDF viewer</param-li>
   </param-ul>
  </param>
  <param>
   <param-def>p_layout varchar2 default 'continuous'</param-def>
   <param-desc>Page layout</param-desc>
   <param-ul>
    <param-li>single: displays single pages</param-li>
    <param-li>continuous: displays pages continuously</param-li>
    <param-li>default: displays according to the setting of the PDF viewer</param-li>
   </param-ul>
  </param>
  <param>
   <param-def>p_hidemenubar boolean default false</param-def>
   <param-desc>
     A flag specifying whether to hide the viewer application's menu bar when the document is active.
   </param-desc>
  </param>
  <param>
   <param-def>p_hidetoolbar boolean default false</param-def>
   <param-desc>
     A flag specifying whether to hide the viewer application's tool bars when the document is active.
   </param-desc>
  </param>
  <param>
   <param-def>p_hidewindowui boolean default false</param-def>
   <param-desc> A flag specifying whether to hide
    user interface elements in the document's window (such as scroll
    bars and navigation controls), leaving only the document's contents displayed.
   </param-desc>
  </param>
  <param>
   <param-def>p_displaydoctitle boolean default false</param-def>
   <param-desc>
    A flag specifying whether
    the window's title bar should display the document title taken from the
    Title entry of the document information dictionary.
   </param-desc>
  </param>
  <param>
   <param-def>p_centerwindow boolean default false</param-def>
   <param-desc>
    A flag specifying whether to position the document's window in the center of the screen.
   </param-desc>
  </param>
  <param>
   <param-def>p_fitwindow boolean default false</param-def>
   <param-desc>
    A flag specifying whether to resize the document's window to fit the size of the first displayed page.
   </param-desc>
  </param>
 </params>
 <return>-</return>
*/
procedure setDocDisplayMode(
  p_zoom varchar2 default 'fullwidth',
  p_layout varchar2 default 'continuous',
  p_hidemenubar boolean default false,
  p_hidetoolbar boolean default false,
  p_hidewindowui boolean default false,
  p_displaydoctitle boolean default false,
  p_centerwindow boolean default false,
  p_fitwindow boolean default false
  );

/**
 <name>setDocKeywords</name>
 <type>Procedure</type>
 <desc>
  Sets the document keywords.
 </desc>
 <version>v3.0.0</version>
 <params>
  <param>
   <param-def>p_keywords varchar2</param-def>
   <param-desc>Keyword list</param-desc>
  </param>
 </params>
 <return>-</return>
*/
procedure setDocKeywords(
  p_keywords varchar2
  );

/**
 <name>setDocSubject</name>
 <type>Procedure</type>
 <desc>
  Sets the document subject.
 </desc>
 <version>v3.0.0</version>
 <params>
  <param>
   <param-def>p_subject varchar2</param-def>
   <param-desc>Document subject</param-desc>
  </param>
 </params>
 <return>-</return>
*/
procedure setDocSubject(
  p_subject varchar2
  );

/**
 <name>SetDocTitle</name>
 <type>Procedure</type>
 <desc>
 Sets the document title.
 </desc>
 <version>v3.0.0</version>
 <params>
  <param>
   <param-def>p_title varchar2</param-def>
   <param-desc>Document title</param-desc>
  </param>
 </params>
 <return>-</return>
*/
procedure setDocTitle(
  p_title varchar2
  );

/**
 <name>setEncoding</name>
 <type>Procedure</type>
 <desc>
  Sets Adobe Core character encoding (default code is cp1252).
 </desc>
 <version>v3.0.0</version>
 <params>
  <param>
   <param-def>p_enc varchar2</param-def>
   <param-desc>Character encoding code</param-desc>
   <param-ul>
    <param-li>cp1250</param-li>
    <param-li>cp1251</param-li>
    <param-li>cp1252: this is the default, no need to set </param-li>
    <param-li>cp1253 </param-li>
    <param-li>cp1254</param-li>
    <param-li>cp1255</param-li>
    <param-li>cp1256</param-li>
    <param-li>cp1257</param-li>
    <param-li>cp1258</param-li>
    <param-li>cp874</param-li>
    <param-li>iso-8859-1</param-li>
    <param-li>iso-8859-2</param-li>
    <param-li>iso-8859-4</param-li>
    <param-li>iso-8859-5</param-li>
    <param-li>iso-8859-7</param-li>
    <param-li>iso-8859-9</param-li>
    <param-li>iso-8859-15</param-li>
    <param-li>iso-8859-16</param-li>
    <param-li>utf16: AL32UTF8 database with unicode TTF embending</param-li>
   </param-ul>
  </param>
 </params>
 <return>-</return>
*/
procedure setEncoding(
  p_enc varchar2
  );

/**
 <name>setHeaderProcName</name>
 <type>Procedure</type>
 <desc>
  Sets the page header procedure name. The program name passed
  as a parameter executes when the page header is created. If
  SetHeaderProcName is not executed before a page header is created then
  the page header will be empty.
 </desc>
 <version>v3.0.0</version>
 <params>
  <param>
   <param-def>p_proc_name varchar2</param-def>
   <param-desc>Page header procedure name</param-desc>
  </param>
  <param>
   <param-def>p_height number</param-def>
   <param-desc>height of header section</param-desc>
  </param>
 </params>
 <return>-</return>
*/
procedure setHeaderProcName(
  p_proc_name varchar2,
  p_height number
  );

/**
 <name>setHeaderProcName5</name>
 <type>Procedure</type>
 <desc>
  Sets the page header procedure name with 5 parameters. The program name passed
  as a parameter executes when the page header is created. If
  SetHeaderProcName is not executed before a page header is created then
  the page header will be empty.
 </desc>
 <version>v3.0.0</version>
 <params>
  <param>
   <param-def>p_proc_name varchar2</param-def>
   <param-desc>Page header procedure name</param-desc>
  </param>
  <param>
   <param-def>p_height number</param-def>
   <param-desc>height of header section</param-desc>
  </param>
  <param>
   <param-def>p_1 varchar2 default null</param-def>
   <param-desc>parameter of header procedure</param-desc>
  </param>
  <param>
   <param-def>p_2 varchar2 default null</param-def>
   <param-desc>parameter of header procedure</param-desc>
  </param>
  <param>
   <param-def>p_3 varchar2 default null</param-def>
   <param-desc>parameter of header procedure</param-desc>
  </param>
  <param>
   <param-def>p_4 varchar2 default null</param-def>
   <param-desc>parameter of header procedure</param-desc>
  </param>
  <param>
   <param-def>p_5 varchar2 default null</param-def>
   <param-desc>parameter of header procedure</param-desc>
  </param>
 </params>
 <return>-</return>
*/
procedure setHeaderProcName5(
  p_proc_name varchar2,
  p_height number,
  p_1 varchar2 default null,
  p_2 varchar2 default null,
  p_3 varchar2 default null,
  p_4 varchar2 default null,
  p_5 varchar2 default null
  );

/**
 <name>set3HeaderProcName</name>
 <type>Procedure</type>
 <desc>
  Sets the page header procedure names. The program name passed as a parameter executes when the page header is created.
 </desc>
 <version>v3.0.1</version>
 <params>
  <param>
   <param-def>p_default_proc_name varchar2 default null</param-def>
   <param-desc>Specifies that this header shall appear on every page in this section which is not overridden with a specific even or first page header.</param-desc>
  </param>
  <param>
   <param-def>p_first_proc_name varchar2 default null</param-def>
   <param-desc>Specifies that this header shall appear on the first page in this document.</param-desc>
  </param>
  <param>
   <param-def>p_even_proc_name varchar2 default null</param-def>
   <param-desc>Specifies that this header shall appear on all even numbered pages in this document.</param-desc>
  </param>
  <param>
   <param-def>p_height number</param-def>
   <param-desc>height of header section</param-desc>
  </param>
 </params>
 <return>-</return>
*/
procedure set3HeaderProcName(
  p_default_proc_name varchar2 default null,
  p_first_proc_name varchar2 default null,
  p_even_proc_name varchar2 default null,
  p_height number
  );

/**
 <name>setFooterProcName</name>
 <type>Procedure</type>
 <desc>
  Sets the page footer procedure name. The program name passed as a parameter
  executes when the page footer is created. If SetFooterProcName is not executed
  before a page footer is created then the page footer will be empty.
 </desc>
 <version>v3.0.0</version>
 <params>
 <param>
  <param-def>p_proc_name varchar2</param-def>
  <param-desc>Page header procedure name</param-desc>
 </param>
 <param>
  <param-def>p_height number</param-def>
  <param-desc>height of header section</param-desc>
 </param>
 </params>
 <return>-</return>
*/
procedure setFooterProcName(
  p_proc_name varchar2,
  p_height number
  );

/**
 <name>setFooterProcName5</name>
 <type>Procedure</type>
 <desc>
  Sets the page footer procedure name with 5 parameters. The program name passed as a parameter
  executes when the page footer is created. If SetFooterProcName is not executed
  before a page footer is created then the page footer will be empty.
 </desc>
 <version>v3.0.0</version>
 <params>
 <param>
  <param-def>p_proc_name varchar2</param-def>
  <param-desc>Page header procedure name</param-desc>
 </param>
 <param>
  <param-def>p_height number</param-def>
  <param-desc>height of header section</param-desc>
 </param>
 <param>
   <param-def>p_1 varchar2 default null</param-def>
   <param-desc>parameter of footer procedure</param-desc>
  </param>
  <param>
   <param-def>p_2 varchar2 default null</param-def>
   <param-desc>parameter of footer procedure</param-desc>
  </param>
  <param>
   <param-def>p_3 varchar2 default null</param-def>
   <param-desc>parameter of footer procedure</param-desc>
  </param>
  <param>
   <param-def>p_4 varchar2 default null</param-def>
   <param-desc>parameter of footer procedure</param-desc>
  </param>
  <param>
   <param-def>p_5 varchar2 default null</param-def>
   <param-desc>parameter of footer procedure</param-desc>
  </param>
 </params>
 <return>-</return>
*/
procedure setFooterProcName5(
  p_proc_name varchar2,
  p_height number,
  p_1 varchar2 default null,
  p_2 varchar2 default null,
  p_3 varchar2 default null,
  p_4 varchar2 default null,
  p_5 varchar2 default null
  );

/**
 <name>set3FooterProcName</name>
 <type>Procedure</type>
 <desc>
  Sets the page footer procedure names. The program name passed as a parameter executes when the page footer is created.
 </desc>
 <version>v3.0.1</version>
 <params>
  <param>
   <param-def>p_default_proc_name varchar2 default null</param-def>
   <param-desc>Specifies that this footer shall appear on every page in this section which is not overridden with a specific even or first page footer.</param-desc>
  </param>
  <param>
   <param-def>p_first_proc_name varchar2 default null</param-def>
   <param-desc>Specifies that this footer shall appear on the first page in this document.</param-desc>
  </param>
  <param>
   <param-def>p_even_proc_name varchar2 default null</param-def>
   <param-desc>Specifies that this footer shall appear on all even numbered pages in this document.</param-desc>
  </param>
  <param>
   <param-def>p_height number</param-def>
   <param-desc>height of header section</param-desc>
  </param>
 </params>
 <return>-</return>
*/
procedure set3FooterProcName(
  p_default_proc_name varchar2 default null,
  p_first_proc_name varchar2 default null,
  p_even_proc_name varchar2 default null,
  p_height number
  );

/**
 <name>setBackgroundColor</name>
 <type>Procedure</type>
 <desc>
  Sets the background color of the pages
 </desc>
 <version>v3.0.0</version>
 <params>
 <param>
  <param-def>p_color plpdf_type.t_color</param-def>
  <param-desc>The color of the background</param-desc>
 </param>
 </params>
 <return>-</return>
*/
procedure setBackgroundColor(
  p_color plpdf_type.t_color
  );

/**
 <name>SetWatermarkProcName</name>
 <type>Procedure</type>
 <desc>
  Sets the name of the watermark creator procedure
 </desc>
 <version>v3.0.0</version>
 <params>
 <param>
  <param-def> p_proc_name varchar2</param-def>
  <param-desc>The name of the procedure</param-desc>
 </param>
 </params>
 <return>-</return>
*/
procedure SetWatermarkProcName(
  p_proc_name varchar2
  );

/**
 <name>SetStampProcName</name>
 <type>Procedure</type>
 <desc>
  Sets the name of the stamp creator procedure
 </desc>
 <version>v3.0.0</version>
 <params>
 <param>
  <param-def> p_proc_name varchar2</param-def>
  <param-desc>The name of the procedure</param-desc>
 </param>
 </params>
 <return>-</return>
*/
procedure SetStampProcName(
  p_proc_name varchar2
  );


/**
 <name>setNewPageProcName</name>
 <type>Procedure</type>
 <desc>
  Sets the new page procedure name. The program name passed
  as a parameter executes when the page is started.
 </desc>
 <version>v4.4.0</version>
 <params>
  <param>
   <param-def>p_proc_name varchar2</param-def>
   <param-desc>New Page procedure name</param-desc>
  </param>
  <param>
   <param-def>p_height number</param-def>
   <param-desc>height of this section</param-desc>
  </param>
 </params>
 <return>-</return>
*/
procedure setNewPageProcName(
  p_proc_name varchar2,
  p_height number
  );

/**
 <name>setLeftMargin</name>
 <type>Procedure</type>
 <desc>
  Sets the left margin size.
 </desc>
 <version>v3.0.0</version>
 <params>
  <param>
   <param-def>p_margin number</param-def>
   <param-desc>Margin size</param-desc>
  </param>
 </params>
 <return>-</return>
*/
procedure setLeftMargin(
  p_margin number
  );

/**
 <name>setTopMargin</name>
 <type>Procedure</type>
 <desc>
 Sets the top margin size.
 </desc>
 <version>v3.0.0</version>
 <params>
  <param>
   <param-def>p_margin number</param-def>
   <param-desc>Margin size</param-desc>
  </param>
 </params>
 <return>-</return>
*/
procedure setTopMargin(
  p_margin number
  );

/**
 <name>setRightMargin</name>
 <type>Procedure</type>
 <desc>
  Sets the right margin size.
 </desc>
 <version>v3.0.0</version>
 <params>
  <param>
   <param-def>p_margin number</param-def>
   <param-desc>Margin size</param-desc>
  </param>
 </params>
 <return>-</return>
*/
procedure setRightMargin(
  p_margin number
  );

/**
 <name>setBottomMargin</name>
 <type>Procedure</type>
 <desc>
  Sets the bottom margin size with autonewpage.
 </desc>
 <version>v3.6.0</version>
 <params>
  <param>
   <param-def>p_margin number</param-def>
   <param-desc>Margin size</param-desc>
  </param>
 </params>
 <return>-</return>
*/
procedure setBottomMargin(
  p_margin number
  );

/**
 <name>setNOLastpageProcName</name>
 <type>Procedure</type>
 <desc>
  Sets the not last page procedure name. The program name passed as a parameter
  executes when the not last page closed.
 </desc>
 <version>v3.0.0</version>
 <params>
  <param>
   <param-def>p_proc_name varchar2</param-def>
   <param-desc>Procedure name</param-desc>
  </param>
 </params>
 <return>-</return>
*/
procedure setNOLastpageProcName(
  p_proc_name varchar2
  );

/**
 <name>SetPageNoStart</name>
 <type>Procedure</type>
 <desc>
  Sets start of page numbering to the current page.
 </desc>
 <version>v3.2.0</version>
 <params>
 <param>
 <param-def>p_start_num pls_integer default null</param-def>
   <param-desc>The start of the numbering, null means 1</param-desc>
  </param>
 </params>
 <return>-</return>
*/
procedure setPageNoStart(
  p_start_num pls_integer default null
  );

/**
 <name>SetPageNoEnd</name>
 <type>Procedure</type>
 <desc>
  Sets end of page numbering to the current page, see <link>SetPageNoStart</link>.
 </desc>
 <version>v3.0.0</version>
 <params>-</params>
 <return>-</return>
*/
procedure setPageNoEnd;

/**
 <name>getPageNoAlter</name>
 <type>Function</type>
 <desc>
  Gets page altering flag, see <link>SetPageNoStart</link>.
 </desc>
 <version>v3.0.0</version>
 <params>-</params>
 <return>
 <return-def>boolean</return-def>
 <return-desc>Altered or not</return-desc>
 </return>
*/
function getPageNoAlter
  return boolean;

/**
 <name>SetProtection</name>
 <type>Procedure</type>
 <desc>
  Allows to protect the PDF document.
 </desc>
 <version>v3.0.0</version>
 <params>
  <param>
   <param-def>p_print_perm Boolean default false</param-def>
   <param-desc>print the PDF</param-desc>
  </param>
  <param>
   <param-def>p_modify_perm Boolean default false</param-def>
   <param-desc>modify the PDF</param-desc>
  </param>
  <param>
   <param-def>p_annot_forms_perm Boolean default false</param-def>
   <param-desc>add annotations and forms</param-desc>
  </param>
  <param>
   <param-def>p_user_pass varchar2 default null</param-def>
   <param-desc>user password</param-desc>
  </param>
  <param>
   <param-def>p_owner_pass varchar2 default null</param-def>
   <param-desc>owner password</param-desc>
  </param>
 </params>
 <return>-</return>

*/
procedure setProtection(
  p_print_perm boolean default false,
  p_modify_perm boolean default false,
  p_copy_perm boolean default false,
  p_annot_forms_perm boolean default false,
  p_user_pass varchar2 default null,
  p_owner_pass varchar2 default null
  );

/**
 <name>setPDFA1B</name>
 <type>Procedure</type>
 <desc>
  Set PDF/A-1B support.
 </desc>
 <version>v3.0.0</version>
 <params>-</params>
 <return>-</return>
*/
procedure setPDFA1B;

/**
 <name>setID</name>
 <type>Procedure</type>
 <desc>
  Sets the internal identifier of the PDF file. ID format is a 32 characters length hex string,
  possible generator is the "rawtohex(SYS_GUID())" expression.
 </desc>
 <version>v3.0.0</version>
 <params>
  <param>
   <param-def>p_id varchar2</param-def>
   <param-desc>ID value</param-desc>
  </param>
 </params>
 <return>-</return>
*/
procedure setID(
  p_id varchar2
  );
-------------------------------------------------------------------------------
/**
 <newpage/>
 <h1>Navigations and formatting commands</h1>
 <br/>
*/
------------------------------------------------------------------------------
/**
 <name>checkPageBreak</name>
 <type>Function</type>
 <desc>
  If the height would cause an overflow, add a new page immediately, and return true.
 </desc>
 <version>v3.0.0</version>
 <params>
  <param>
   <param-def>p_h number</param-def>
   <param-desc>height for check page overflow</param-desc>
  </param>
  <param>
   <param-def>p_newpage boolean default true</param-def>
   <param-desc>do NewPage procedure</param-desc>
  </param>
 </params>
 <return>
  <return-def>boolean</return-def>
  <param-ul>
    <param-li>true: overflow is reached</param-li>
    <param-li>false: don't reach overflow</param-li>
  </param-ul>
 </return>
*/
function checkPageBreak(
  p_h number,
  p_newpage boolean default true
  ) return boolean;

/**
 <name>getCurrPageNum</name>
 <type>Function</type>
 <desc>
  Returns the page number of the current page.
 </desc>
 <version>v3.0.0</version>
 <params>-</params>
 <return>
  <return-def>pls_integer</return-def>
  <return-desc>Page number of the current page</return-desc>
 </return>
*/
function getCurrPageNum
  return pls_integer;

/**
 <name>getNumOfPages</name>
 <type>Function</type>
 <desc>
  Returns the number of pages, this procedure works in header and footer procedures only.
 </desc>
 <version>v3.0.0</version>
 <params>-</params>
 <return>
  <return-def>pls_integer</return-def>
  <return-desc>Number of pages</return-desc>
 </return>
*/
function getNumOfPages
  return pls_integer;

/**
 <name>getLeftMargin</name>
 <type>Function</type>
 <desc>
  Gets the page left margin size.
 </desc>
 <version>v3.0.0</version>
 <params>-</params>
 <return>
  <return-def>number</return-def>
  <return-desc>Size of left margin</return-desc>
 </return>
*/
function getLeftMargin
  return number;

/**
 <name>getRightMargin</name>
 <type>Function</type>
 <desc>
  Gets the page right margin size.
 </desc>
 <version>v3.0.0</version>
 <params>-</params>
 <return>
  <return-def>number</return-def>
  <return-desc>Size of right margin</return-desc>
 </return>
*/
function getRightMargin
  return number;

/**
 <name>getTopMargin</name>
 <type>Function</type>
 <desc>
  Gets the page top margin size.
 </desc>
 <version>v3.0.0</version>
 <params>-</params>
 <return>
  <return-def>number</return-def>
  <return-desc>Size of top margin</return-desc>
 </return>
*/
function getTopMargin
  return number;

/**
 <name>getBottomMargin</name>
 <type>Function</type>
 <desc>
  Gets the page bottom margin size.
 </desc>
 <version>v3.0.0</version>
 <params>-</params>
 <return>
  <return-def>number</return-def>
  <return-desc>Size of bottom margin</return-desc>
 </return>
*/
function getBottomMargin
  return number;

/**
 <name>getHeaderHeight</name>
 <type>Function</type>
 <desc>
  Gets the page header size.
 </desc>
 <version>v3.0.0</version>
 <params>-</params>
 <return>
  <return-def>number</return-def>
  <return-desc>Size of header</return-desc>
 </return>
*/
function getHeaderHeight
  return number;

/**
 <name>getFooterHeight</name>
 <type>Function</type>
 <desc>
  Gets the page footer size.
 </desc>
 <version>v3.0.0</version>
 <params>-</params>
 <return>
  <return-def>number</return-def>
  <return-desc>Size of footer</return-desc>
 </return>
*/
function getFooterHeight
  return number;

/**
 <name>getCurrentX</name>
 <type>Function</type>
 <desc>
  Gets the X coordinate of the cursor.
 </desc>
 <version>v3.0.0</version>
 <params>-</params>
 <return>
  <return-def>number</return-def>
  <return-desc>X coordinate</return-desc>
 </return>
*/
function getCurrentX
  return number;

/**
 <name>getCurrentY</name>
 <type>Function</type>
 <desc>
  Gets the Y coordinate of the cursor.
 </desc>
 <version>v3.0.0</version>
 <params>-</params>
 <return>
  <return-def>number</return-def>
  <return-desc>Y coordinate</return-desc>
 </return>
*/
function getCurrentY
  return number;

/**
 <name>getPageAvailableHeight</name>
 <type>Function</type>
 <desc>
  Gets the available height (without margins) of actual page.
 </desc>
 <version>v3.0.0</version>
 <params>-</params>
 <return>
  <return-def>number</return-def>
  <return-desc>Available height</return-desc>
 </return>
*/
function getPageAvailableHeight
  return number;

/**
 <name>getPageAvailableWidth</name>
 <type>Function</type>
 <desc>
  Gets the available width (without margins) of actual page.
 </desc>
 <version>v3.0.0</version>
 <params>-</params>
 <return>
  <return-def>number</return-def>
  <return-desc>Available width</return-desc>
 </return>
*/
function getPageAvailableWidth
  return number;

/**
 <name>getPageHeight</name>
 <type>Function</type>
 <desc>
 Gets the height of actual page.
 </desc>
 <version>v3.0.0</version>
 <params>-</params>
 <return>
  <return-def>number</return-def>
  <return-desc>The height of the page</return-desc>
 </return>
*/
function getPageHeight
  return number;

/**
 <name>getPageRightSpace</name>
 <type>Function</type>
 <desc>
  Gets the available width (without margins) of actual page from current position.
 </desc>
 <version>v3.0.0</version>
 <params>-</params>
 <return>
  <return-def>number</return-def>
  <return-desc>Actual page available space</return-desc>
 </return>
*/
function getPageRightSpace
  return number;

/**
 <name>getPageLeftSpace</name>
 <type>Function</type>
 <desc>
  Gets the available width (without margins) of actual page from current position.
 </desc>
 <version>v3.0.0</version>
 <params>-</params>
 <return>
  <return-def>number</return-def>
  <return-desc>Actual page available space</return-desc>
 </return>
*/
function getPageLeftSpace
  return number;

/**
 <name>getPageSpace</name>
 <type>Function</type>
 <desc>
  Gets the available width (without margins) of actual page from current position.
 </desc>
 <version>v3.0.0</version>
 <params>-</params>
 <return>
  <return-def>number</return-def>
  <return-desc>Actual page available space</return-desc>
 </return>
*/
function getPageSpace
  return number;

/**
 <name>getPageWidth</name>
 <type>Function</type>
 <desc>
  Gets the width of actual page.
 </desc>
 <version>v3.0.0</version>
 <params>-</params>
 <return>
  <return-def>number</return-def>
  <return-desc>Actual page width</return-desc>
 </return>
*/
function getPageWidth
  return number;

/**
 <name>getPageBottomSpace</name>
 <type>Function</type>
 <desc>
  Gets the available height (without margins) of actual page from current position.
 </desc>
 <version>v3.0.0</version>
 <params>-</params>
 <return>
  <return-def>number</return-def>
  <return-desc>Actual page bottom space</return-desc>
 </return>
*/
function getPageBottomSpace
  return number;

/**
 <name>getPrintFontKey</name>
 <type>Function</type>
 <desc>
  Get the current font name and style. See <link>SetPrintFont</link>.
 </desc>
 <version>v3.0.0</version>
 <params>-</params>
 <return>
  <return-def>varchar2</return-def>
  <return-desc>Font type</return-desc>
 </return>
*/
function getPrintFontKey
  return varchar2;

/**
 <name>getPrintFontName</name>
 <type>Function</type>
 <desc>
  Get the current font type. See <link>SetPrintFont</link>.
 </desc>
 <version>v3.0.0</version>
 <params>-</params>
 <return>
  <return-def>varchar2</return-def>
  <return-desc>Font type</return-desc>
 </return>
*/
function getPrintFontName
  return varchar2;

/**
 <name>getPrintFontStyle</name>
 <type>Function</type>
 <desc>
  Get the current font style. See <link>SetPrintFont</link>
 </desc>
 <version>v3.0.0</version>
 <params>-</params>
 <return>
  <return-def>varchar2</return-def>
  <return-desc>Font style</return-desc>
 </return>
*/
function getPrintFontStyle
  return varchar2;

/**
 <name>getPrintFontSize</name>
 <type>Function</type>
 <desc>
  Get the current font size. See <link>SetPrintFont</link>
 </desc>
 <version>v3.0.0</version>
 <params>-</params>
 <return>
  <return-def>varchar2</return-def>
  <return-desc>Font size</return-desc>
 </return>
*/
function getPrintFontSize
  return number;

/**
 <name>getScaleFactor</name>
 <type>Function</type>
 <desc>
  Get the current scale factor.
 </desc>
 <version>v3.0.0</version>
 <params>-</params>
 <return>
  <return-def>number</return-def>
  <return-desc>scale factor</return-desc>
 </return>
*/
function getScaleFactor
  return number;

/**
 <name>getTextWidth</name>
 <type>Function</type>
 <desc>
 Returns the width of the parameter text (uses current font).
 </desc>
 <version>v3.0.0</version>
 <params>
  <param>
   <param-def>p_s varchar2</param-def>
   <param-desc>Text</param-desc>
  </param>
 </params>
 <return>
  <return-def>number</return-def>
  <return-desc>The width of the text in the unit of measure specified in Init</return-desc>
 </return>
*/
function getTextWidth(
  p_s varchar2
  ) return number;

/**
 <name>getTextwidthFont</name>
 <type>Function</type>
 <desc>
 Returns the width of the parameter text, font is a parameter too.
 </desc>
 <version>v3.0.0</version>
 <params>
  <param>
   <param-def>p_family varchar2 default null</param-def>
   <param-desc>Font type, null means current font family</param-desc>
   <param-ul>
    <param-li>Helvetica (Arial, Sans Serif)</param-li>
    <param-li>Courier</param-li>
    <param-li>Times (Serif)</param-li>
    <param-li>Symbol</param-li>
    <param-li>ZapfDingBats</param-li>
    <param-li>If TTF fonts were added using the AddTTF procedure then they may be used here as well.</param-li>
   </param-ul>
  </param>
  <param>
   <param-def>p_style varchar2 default null</param-def>
   <param-desc>Font style</param-desc>
   <param-ul>
    <param-li>null: Regular</param-li>
    <param-li>B: Bold</param-li>
    <param-li>I: Italic</param-li>
    <param-li>U: Underline</param-li>
    <param-li>The styles may be combined (i.e. BI will be Bold and Italic)</param-li>
   </param-ul>
  </param>
  <param>
   <param-def>p_size number default null</param-def>
   <param-desc>Font size, null means current font size</param-desc>
  </param>
  <param>
   <param-def>p_s varchar2</param-def>
   <param-desc>Text</param-desc>
  </param>
 </params>
 <return>
  <return-def>number</return-def>
  <return-desc>The width of the text in the unit of measure specified in Init</return-desc>
 </return>
*/
function getTextwidthFont(
  p_family varchar2 default null,
  p_style varchar2 default null,
  p_size number default null,
  p_s varchar2
  ) return number;

/**
 <name>removeMissingChars</name>
 <type>Function</type>
 <desc>Remove unknown characters from text (uses current font).</desc>
 <version>v3.0.0</version>
 <params>
  <param>
   <param-def>p_s varchar2</param-def>
   <param-desc>Text</param-desc>
  </param>
 </params>
 <return>
  <return-def>varchar2</return-def>
  <return-desc>The text without missing characters</return-desc>
 </return>
*/
function removeMissingChars(
  p_s varchar2
  ) return varchar2;

/**
 <name>insertMovePage</name>
 <type>Procedure</type>
 <desc>
  Insert a new page and move it by changing order of pages.
 </desc>
 <version>v3.0.0</version>
 <params>
  <param>
   <param-def>p_orig_pos number</param-def>
   <param-desc>origin page position</param-desc>
  </param>
  <param>
   <param-def>p_new_pos number</param-def>
   <param-desc>new page position</param-desc>
  </param>
 </params>
 <return>-</return>
*/
procedure insertMovePage(
  p_orig_pos number,
  p_new_pos number
  );

/**
 <name>NewPage</name>
 <type>Procedure</type>
 <desc>
  Creates a new page. If no orientation is given then the orientation specified in Init
  will be used. If a page was previously open, a page break is initiated and the
  cursor is placed at the top left corner of the new page.
 </desc>
 <version>v3.0.0</version>
 <params>
  <param>
   <param-def>p_orientation varchar2 default NULL</param-def>
   <param-desc>Page orientation</param-desc>
   <param-ul>
    <param-li>P: Portrait</param-li>
    <param-li>L: Landscape</param-li>
   </param-ul>
  </param>
 </params>
 <return>-</return>
*/
procedure NewPage(
  p_orientation varchar2 default null
  );

/**
 <name>SetNoPrintFooter</name>
 <type>Procedure</type>
 <desc>
  Disables the footer on the specified page.
 </desc>
 <version>v3.0.0</version>
 <params>
  <param>
   <param-def>p_page pls_integer default null</param-def>
   <param-desc>Page number</param-desc>
  </param>
 </params>
 <return>-</return>
*/
procedure setNoPrintFooter(
  p_page pls_integer default null
  );

/**
 <name>setNoPrintHeader</name>
 <type>Procedure</type>
 <desc>
  Disables the header on the specified page.
 </desc>
 <version>v4.9.6</version>
 <params>
  <param>
   <param-def>p_page pls_integer default null</param-def>
   <param-desc>Page number</param-desc>
  </param>
 </params>
 <return>-</return>
*/
procedure setNoPrintHeader(
  p_page pls_integer default null
  );

/**
 <name>setCurrentX</name>
 <type>Procedure</type>
 <desc>
  Places the cursor at the X coordinate.
 </desc>
 <version>v3.0.0</version>
 <params>
  <param>
   <param-def>p_x number default 0</param-def>
   <param-desc>X coordinate</param-desc>
  </param>
 </params>
 <return>-</return>
*/
procedure setCurrentX(
  p_x number default 0
  );

/**
 <name>setCurrentY</name>
 <type>Procedure</type>
 <desc>
  Places the cursor at the Y coordinate and reset X.
 </desc>
 <version>v3.0.0</version>
 <params>
  <param>
   <param-def>p_y number default 0</param-def>
   <param-desc>Y coordinate</param-desc>
  </param>
  <param>
   <param-def>p_cr boolean default true</param-def>
   <param-desc>X position returns to beginning of line</param-desc>
  </param>
 </params>
 <return>-</return>
*/
procedure setCurrentY(
  p_y number default 0,
  p_cr boolean default true
  );

/**
 <name>setCurrentXY</name>
 <type>Procedure</type>
 <desc>
  Places the cursor at the X and Y coordinate.
 </desc>
 <version>v3.0.0</version>
 <params>
  <param>
   <param-def>p_x number default 0</param-def>
   <param-desc>X coordinate</param-desc>
  </param>
  <param>
   <param-def>p_y number default 0</param-def>
   <param-desc>Y coordinate</param-desc>
  </param>
 </params>
 <return>-</return>

*/
procedure setCurrentXY(
  p_x number default 0,
  p_y number default 0
  );


/**
 <name>NewLine</name>
 <type>Procedure</type>
 <desc>
  Line break. Cursor is placed at the start of the next line.
 </desc>
 <version>v3.0.0</version>
 <params>
  <param>
   <param-def>p_h number default 0</param-def>
   <param-desc>Height of the line break.</param-desc>
  </param>
 </params>
 <return>-</return>
*/
procedure NewLine(
  p_h number default 0
  );

------------------------------------------------------------------------------
/**
 <newpage/>
 <h1>Content Related Commands</h1>
 <br/>
*/
------------------------------------------------------------------------------

/**
 <name>AttachFile</name>
 <type>Procedure</type>
 <desc>
  Attachs a file to the PDF document.
 </desc>
 <version>v3.0.0</version>
 <params>
  <param>
   <param-def>p_filename varchar2</param-def>
   <param-desc>name of file</param-desc>
  </param>
  <param>
   <param-def>p_fileblob blob</param-def>
   <param-desc>file</param-desc>
  </param>
  <param>
   <param-def>p_desc varchar2 default null</param-def>
   <param-desc>description of file</param-desc>
  </param>
 </params>
 <return>-</return>
*/
procedure AttachFile(
  p_filename varchar2,
  p_fileblob blob,
  p_desc varchar2 default null
  );

/**
 <name>crBookmark</name>
 <type>Procedure</type>
 <desc>
  Create a bookmark in the PDF document. Bookmarks are displayed in the PDF viewer program (e.g. Acrobat Reader).
 </desc>
 <version>v3.0.0</version>
 <params>
  <param>
   <param-def>p_txt varchar2</param-def>
   <param-desc>Text of the bookmark</param-desc>
  </param>
  <param>
   <param-def>p_level number default 0</param-def>
   <param-desc>Bookmark level, 0 is the highest level</param-desc>
  </param>
  <param>
   <param-def>p_y number default null</param-def>
   <param-desc>Y coordinate of the bookmark within the current page. Null denotes the cursor position.</param-desc>
  </param>
 </params>
 <return>-</return>
*/
procedure crBookmark(
  p_txt varchar2,
  p_level number default 0,
  p_y number default null
  );

/**
 <name>PrintoutText</name>
 <type>Procedure</type>
 <desc>
  Prints a text starting from the specified position.
 </desc>
 <version>v3.0.0</version>
 <params>
  <param>
   <param-def>p_x number</param-def>
   <param-desc>X coordinate for the start of the text</param-desc>
  </param>
  <param>
   <param-def>p_y number  default null</param-def>
   <param-desc>Y coordinate for the start of the text</param-desc>
  </param>
  <param>
   <param-def>p_txt varchar2</param-def>
   <param-desc>Text</param-desc>
  </param>
 </params>
 <return>-</return>
*/
procedure PrintoutText(
  p_x number,
  p_y number,
  p_txt varchar2
  );

/**
 <name>Circle</name>
 <type>Procedure</type>
 <desc>
  Draws a cirle on the current page.
 </desc>
 <version>v3.0.1</version>
 <params>
  <param>
   <param-def>p_x number</param-def>
   <param-desc>X coordinate of the center of the circle</param-desc>
  </param>
   <param>
   <param-def>p_y number</param-def>
  <param-desc>Y coordinate of the center of the circle</param-desc>
  </param>
  <param>
   <param-def>p_r number</param-def>
   <param-desc>the radius of the circle</param-desc>
  </param>
  <param>
   <param-def>p_draw boolean default true</param-def>
   <param-desc>frame drawing</param-desc>
  </param>
  <param>
   <param-def>p_fill boolean default false</param-def>
   <param-desc>filling</param-desc>
  </param>
  <param>
   <param-def>p_draw_color plpdf_type.t_color default null</param-def>
   <param-desc>frame drawing color</param-desc>
  </param>
  <param>
   <param-def>p_fill_color plpdf_type.t_color default null</param-def>
   <param-desc>filling color</param-desc>
  </param>
  <param>
   <param-def>p_linewidth number default null</param-def>
   <param-desc>The width of the frame line of the rectangle</param-desc>
  </param>
  <param>
   <param-def>p_fill_pattern varchar2 default null</param-def>
   <param-desc>The filling pattern</param-desc>
  </param>
 </params>
 <return>-</return>
*/
procedure Circle(
  p_x number,
  p_y number,
  p_r number,
  p_draw boolean default true,
  p_fill boolean default false,
  p_draw_color plpdf_type.t_color default null,
  p_fill_color plpdf_type.t_color default null,
  p_linewidth number default null,
  p_fill_pattern varchar2 default null
  );

/**
 <name>Ellipse</name>
 <type>Procedure</type>
 <desc>
  Draws an ellipse on the current page.
 </desc>
 <version>v3.0.1</version>
 <params>
  <param>
   <param-def>p_x number</param-def>
   <param-desc>X coordinate of the center of the ellipse</param-desc>
  </param>
  <param>
   <param-def>p_y number</param-def>
   <param-desc>Y coordinate of the center of the ellipse</param-desc>
  </param>
  <param>
   <param-def>p_rx number</param-def>
   <param-desc>horizontal radius of the ellipse</param-desc>
  </param>
  <param>
   <param-def>p_ry number</param-def>
   <param-desc>vertical radius of the ellipse</param-desc>
  </param>
  <param>
   <param-def>p_draw boolean default true</param-def>
   <param-desc>frame drawing</param-desc>
  </param>
  <param>
   <param-def>p_fill boolean default false</param-def>
   <param-desc>filling</param-desc>
  </param>
  <param>
   <param-def>p_draw_color plpdf_type.t_color default null</param-def>
   <param-desc>frame drawing color</param-desc>
  </param>
  <param>
   <param-def>p_fill_color plpdf_type.t_color default null</param-def>
   <param-desc>filling color</param-desc>
  </param>
  <param>
   <param-def>p_linewidth number default null</param-def>
   <param-desc>The width of the frame line of the rectangle</param-desc>
  </param>
  <param>
   <param-def>p_fill_pattern varchar2 default null</param-def>
   <param-desc>The filling pattern</param-desc>
  </param>
 </params>
 <return>-</return>
*/
procedure Ellipse(
  p_x number,
  p_y number,
  p_rx number,
  p_ry number,
  p_draw boolean default true,
  p_fill boolean default false,
  p_draw_color plpdf_type.t_color default null,
  p_fill_color plpdf_type.t_color default null,
  p_linewidth number default null,
  p_fill_pattern varchar2 default null
  );

/**
 <name>Line</name>
 <type>Procedure</type>
 <desc>
  Draws a line between the two positions on the page.
 </desc>
 <version>v3.0.0</version>
 <params>
  <param>
   <param-def>p_x1 number</param-def>
   <param-desc>X coordinate for the start of the line</param-desc>
  </param>
  <param>
   <param-def>p_y1 number</param-def>
   <param-desc>Y coordinate for the start of the line</param-desc>
  </param>
  <param>
   <param-def>p_x2 number</param-def>
   <param-desc>X coordinate for the end of the line</param-desc>
  </param>
  <param>
   <param-def>p_y2 number</param-def>
   <param-desc>Y coordinate for the end of the line</param-desc>
  </param>
  <param>
   <param-def>p_color plpdf_type.t_color default null</param-def>
   <param-desc>color of the line</param-desc>
  </param>
  <param>
   <param-def>p_width number default null</param-def>
   <param-desc>width of the line</param-desc>
  </param>
 </params>
 <return>-</return>
*/
procedure Line(
  p_x1 number,
  p_y1 number,
  p_x2 number,
  p_y2 number,
  p_color plpdf_type.t_color default null,
  p_width number default null
  );

/**
 <name>Polygon</name>
 <type>Procedure</type>
 <desc>
  Draws a lines (polygon) between the points on the page.
 </desc>
 <version>v3.0.1</version>
 <params>
  <param>
   <param-def>p_points plpdf_type.t_points</param-def>
   <param-desc>set of ordered points</param-desc>
  </param>
  <param>
   <param-def>p_draw boolean default true</param-def>
   <param-desc>frame drawing</param-desc>
  </param>
  <param>
   <param-def>p_fill boolean default false</param-def>
   <param-desc>filling</param-desc>
  </param>
  <param>
   <param-def>p_draw_color plpdf_type.t_color default null</param-def>
   <param-desc>frame drawing color</param-desc>
  </param>
  <param>
   <param-def>p_fill_color plpdf_type.t_color default null</param-def>
   <param-desc>filling color</param-desc>
  </param>
  <param>
   <param-def>p_linewidth number default null</param-def>
   <param-desc>The width of the frame line of the rectangle</param-desc>
  </param>
  <param>
   <param-def>p_fill_pattern varchar2 default null</param-def>
   <param-desc>The filling pattern</param-desc>
  </param>
 </params>
 <return>-</return>
*/
procedure Polygon(
  p_points plpdf_type.t_points,
  p_draw boolean default true,
  p_fill boolean default false,
  p_draw_color plpdf_type.t_color default null,
  p_fill_color plpdf_type.t_color default null,
  p_linewidth number default null,
  p_fill_pattern varchar2 default null
  );

--v3.6.0
/**
 <name>Path</name>
 <type>Procedure</type>
 <desc>
  Draws a lines between the points on the page.
 </desc>
 <version>v3.6.0</version>
 <params>
  <param>
   <param-def>p_operators plpdf_type.t_path_operators</param-def>
   <param-desc>set of ordered operators defined in the PDF Reference. Supported Operators: m,l,c,v,y,h,re</param-desc>
  </param>
  <param>
   <param-def>p_draw boolean default true</param-def>
   <param-desc>frame drawing</param-desc>
  </param>
  <param>
   <param-def>p_fill boolean default false</param-def>
   <param-desc>filling</param-desc>
  </param>
  <param>
   <param-def>p_draw_color plpdf_type.t_color default null</param-def>
   <param-desc>frame drawing color</param-desc>
  </param>
  <param>
   <param-def>p_fill_color plpdf_type.t_color default null</param-def>
   <param-desc>filling color</param-desc>
  </param>
  <param>
   <param-def>p_linewidth number default null</param-def>
   <param-desc>The width of the frame line of the rectangle</param-desc>
  </param>
  <param>
   <param-def>p_fill_pattern varchar2 default null</param-def>
   <param-desc>The filling pattern</param-desc>
  </param>
 </params>
 <return>-</return>
*/
procedure Path(
  p_operators plpdf_type.t_path_operators,
  p_draw boolean default true,
  p_fill boolean default false,
  p_draw_color plpdf_type.t_color default null,
  p_fill_color plpdf_type.t_color default null,
  p_linewidth number default null,
  p_fill_pattern varchar2 default null
  );

/**
 <name>Rect</name>
 <type>Procedure</type>
 <desc>
  Draws a rectangle on the page.
 </desc>
 <version>v3.0.0</version>
 <params>
  <param>
  <param-def>p_x number</param-def>
  <param-desc>X coordinate for the top left corner of the rectangle</param-desc>
  </param>
  <param>
   <param-def>p_y number</param-def>
   <param-desc>Y coordinate for the top left corner of the rectangle</param-desc>
  </param>
  <param>
   <param-def>p_w number</param-def>
   <param-desc>The width of the rectangle</param-desc>
  </param>
  <param>
   <param-def>p_h number</param-def>
   <param-desc>The height of the rectangle</param-desc>
  </param>
  <param>
   <param-def>p_draw boolean default true</param-def>
   <param-desc>frame drawing</param-desc>
  </param>
  <param>
   <param-def>p_fill boolean default false</param-def>
   <param-desc>filling</param-desc>
  </param>
  <param>
   <param-def>p_draw_color plpdf_type.t_color default null</param-def>
   <param-desc>frame drawing color</param-desc>
  </param>
  <param>
   <param-def>p_fill_color plpdf_type.t_color default null</param-def>
   <param-desc>filling color</param-desc>
  </param>
  <param>
   <param-def>p_linewidth number default null</param-def>
   <param-desc>The width of the frame line of the rectangle</param-desc>
  </param>
  <param>
   <param-def>p_fill_pattern varchar2 default null</param-def>
   <param-desc>The filling pattern</param-desc>
  </param>
 </params>
 <return>-</return>
*/
procedure Rect(
  p_x number,
  p_y number,
  p_w number,
  p_h number,
  p_draw boolean default true,
  p_fill boolean default false,
  p_draw_color plpdf_type.t_color default null,
  p_fill_color plpdf_type.t_color default null,
  p_linewidth number default null,
  p_fill_pattern varchar2 default null
  );

/**
 <name>RoundedRect</name>
 <type>Procedure</type>
 <desc>
  Draws a rectangle on the page.
 </desc>
 <version>v3.0.1</version>
 <params>
  <param>
  <param-def>p_x number</param-def>
  <param-desc>X coordinate for the top left corner of the rectangle</param-desc>
  </param>
  <param>
   <param-def>p_y number</param-def>
   <param-desc>Y coordinate for the top left corner of the rectangle</param-desc>
  </param>
  <param>
   <param-def>p_w number</param-def>
   <param-desc>The width of the rectangle</param-desc>
  </param>
  <param>
   <param-def>p_h number</param-def>
   <param-desc>The height of the rectangle</param-desc>
  </param>
   <param>
   <param-def>p_r number</param-def>
   <param-desc>The radius of the corners</param-desc>
  </param>
  <param>
   <param-def>p_draw boolean default true</param-def>
   <param-desc>frame drawing</param-desc>
  </param>
  <param>
   <param-def>p_fill boolean default false</param-def>
   <param-desc>filling</param-desc>
  </param>
  <param>
   <param-def>p_draw_color plpdf_type.t_color default null</param-def>
   <param-desc>frame drawing color</param-desc>
  </param>
  <param>
   <param-def>p_fill_color plpdf_type.t_color default null</param-def>
   <param-desc>filling color</param-desc>
  </param>
  <param>
   <param-def>p_linewidth number default null</param-def>
   <param-desc>The width of the frame line of the rectangle</param-desc>
  </param>
  <param>
   <param-def>p_fill_pattern varchar2 default null</param-def>
   <param-desc>The filling pattern</param-desc>
  </param>
 </params>
 <return>-</return>
*/
procedure RoundedRect(
  p_x number,
  p_y number,
  p_w number,
  p_h number,
  p_r number,
  p_draw boolean default true,
  p_fill boolean default false,
  p_draw_color plpdf_type.t_color default null,
  p_fill_color plpdf_type.t_color default null,
  p_linewidth number default null,
  p_fill_pattern varchar2 default null
  );

/**
 <name>Sector</name>
 <type>Procedure</type>
 <desc>
  Draws a sector on the current page.
 </desc>
 <version>v3.0.1</version>
 <params>
  <param>
   <param-def>p_xc number</param-def>
   <param-desc>X coordinate of the center of the sector</param-desc>
  </param>
  <param>
   <param-def>p_yc number</param-def>
   <param-desc>Y coordinate of the center of the sector</param-desc>
  </param>
  <param>
   <param-def>p_r number</param-def>
   <param-desc>Radius of the sector</param-desc>
  </param>
  <param>
   <param-def>p_a number</param-def>
   <param-desc>Starting point of the sector</param-desc>
  </param>
  <param>
   <param-def>p_b number</param-def>
   <param-desc>End point of the sector</param-desc>
  </param>
  <param>
   <param-def>p_draw boolean default true</param-def>
   <param-desc>frame drawing</param-desc>
  </param>
  <param>
   <param-def>p_fill boolean default false</param-def>
   <param-desc>filling</param-desc>
  </param>
  <param>
   <param-def>p_draw_color plpdf_type.t_color default null</param-def>
   <param-desc>frame drawing color</param-desc>
  </param>
  <param>
   <param-def>p_fill_color plpdf_type.t_color default null</param-def>
   <param-desc>filling color</param-desc>
  </param>
  <param>
   <param-def>p_linewidth number default null</param-def>
   <param-desc>The width of the frame line of the rectangle</param-desc>
  </param>
  <param>
   <param-def>p_fill_pattern varchar2 default null</param-def>
   <param-desc>The filling pattern</param-desc>
  </param>
  <param>
   <param-def>p_cw boolean default true</param-def>
   <param-desc>Draw direction</param-desc>
   <param-ul>
    <param-li>true: Clockwise</param-li>
    <param-li>false: Anti-clockwise</param-li>
   </param-ul>
  </param>
  <param>
   <param-def>p_o number default 90</param-def>
   <param-desc>origin of angles</param-desc>
  </param>
 </params>
 <return>-</return>
*/
procedure Sector(
  p_xc number,
  p_yc number,
  p_r number,
  p_a number,
  p_b number,
  p_draw boolean default true,
  p_fill boolean default false,
  p_draw_color plpdf_type.t_color default null,
  p_fill_color plpdf_type.t_color default null,
  p_linewidth number default null,
  p_fill_pattern varchar2 default null,
  p_cw boolean default true,
  p_o number default 90
  );

/**
 <name>EndOpacity</name>
 <type>Procedure</type>
 <desc>
  End a transparency.  See <link>StartOpacity</link>
 </desc>
 <version>v3.0.0</version>
 <params>-</params>
 <return>-</return>
*/
procedure EndOpacity;

/**
 <name>EndOptCont</name>
 <type>Procedure</type>
 <desc>
  End an optional content.  See <link>link StartOptCont</link>
 </desc>
 <version>v3.0.0</version>
 <params>-</params>
 <return>-</return>
*/
procedure EndOptCont;

/**
 <name>putImage</name>
 <type>Procedure</type>
 <desc>
  Inserts an image (JPG, BMP, PNG, etc.) from a BLOB variable into the current page.
 </desc>
 <version>v3.0.1</version>
 <params>
  <param>
   <param-def>p_name varchar2</param-def>
   <param-desc>Image name or ID, if the name is equal with an earlier used image name then procedure reuses the original blob</param-desc>
  </param>
  <param>
   <param-def>p_data blob</param-def>
   <param-desc>Variable containing the image as binary data</param-desc>
  </param>
  <param>
   <param-def>p_x number</param-def>
   <param-desc>X coordinate of the image</param-desc>
  </param>
  <param>
   <param-def>p_y number</param-def>
   <param-desc>Y coordinate of the image</param-desc>
  </param>
  <param>
   <param-def>p_w number default null</param-def>
   <param-desc>Image width, if null then the image is displayed in its actual width</param-desc>
  </param>
  <param>
   <param-def>p_h number default null</param-def>
   <param-desc>Image height, if null then the image is displayed in its actual height</param-desc>
  </param>
  <param>
   <param-def>p_link varchar2 default NULL</param-def>
   <param-desc>URL or internal link ID</param-desc>
  </param>
 </params>
 <return>-</return>
*/
procedure putImage(
  p_name varchar2,
  p_data in out nocopy blob,
  p_x number,
  p_y number,
  p_w number default null,
  p_h number default null,
  p_link varchar2 default null
  );

/**
 <name>putImage</name>
 <type>Procedure</type>
 <desc>
  Inserts an image (JPG, BMP, PNG, etc.) from a BLOB variable into the current page.
 </desc>
 <version>v3.0.1</version>
 <params>
  <param>
   <param-def>p_name varchar2</param-def>
   <param-desc>The name is an earlier used image name and procedure reuse this image blob.</param-desc>
  </param>
  <param>
   <param-def>p_x number</param-def>
   <param-desc>X coordinate of the image</param-desc>
  </param>
  <param>
   <param-def>p_y number</param-def>
   <param-desc>Y coordinate of the image</param-desc>
  </param>
  <param>
   <param-def>p_w number default null</param-def>
   <param-desc>Image width, if null then the image is displayed in its actual width</param-desc>
  </param>
  <param>
   <param-def>p_h number default null</param-def>
   <param-desc>Image height, if null then the image is displayed in its actual height</param-desc>
  </param>
  <param>
   <param-def>p_link varchar2 default NULL</param-def>
   <param-desc>URL or internal link ID</param-desc>
  </param>
 </params>
 <return>-</return>
*/
procedure PutImage(
  p_name varchar2,
  p_x number,
  p_y number,
  p_w number default null,
  p_h number default null,
  p_link varchar2 default null
  );

/**
 <name>putImage</name>
 <type>Procedure</type>
 <desc>
  Inserts an image (JPG, BMP, PNG, etc.) from a BLOB variable into the current page.
 </desc>
 <version>v3.0.1</version>
 <params>
  <param>
   <param-def>p_data blob</param-def>
   <param-desc>Variable containing the image as binary data</param-desc>
  </param>
  <param>
   <param-def>p_x number</param-def>
   <param-desc>X coordinate of the image</param-desc>
  </param>
  <param>
   <param-def>p_y number</param-def>
   <param-desc>Y coordinate of the image</param-desc>
  </param>
  <param>
   <param-def>p_w number default null</param-def>
   <param-desc>Image width, if null then the image is displayed in its actual width</param-desc>
  </param>
  <param>
   <param-def>p_h number default null</param-def>
   <param-desc>Image height, if null then the image is displayed in its actual height</param-desc>
  </param>
  <param>
   <param-def>p_link varchar2 default NULL</param-def>
   <param-desc>URL or internal link ID</param-desc>
  </param>
 </params>
 <return>-</return>
*/
procedure PutImage(
  p_data in out nocopy blob,
  p_x number,
  p_y number,
  p_w number default null,
  p_h number default null,
  p_link varchar2 default null
  );

/**
 <name>putInternalLink</name>
 <type>Procedure</type>
 <desc>
  Puts a link area on the page that point to an internal link.
 </desc>
 <version>v3.0.0</version>
 <params>
  <param>
   <param-def>p_x number</param-def>
   <param-desc>X coordinate of the top left corner of the area</param-desc>
  </param>
  <param>
   <param-def>p_y number</param-def>
   <param-desc>Y coordinate of the top left corner of the area</param-desc>
  </param>
  <param>
   <param-def>p_w number</param-def>
   <param-desc>Link area width</param-desc>
  </param>
  <param>
   <param-def>p_h number</param-def>
   <param-desc>Link area height</param-desc>
  </param>
  <param>
   <param-def>p_link pls_integer</param-def>
   <param-desc>Internal link ID returned by crInternalLinkDest</param-desc>
  </param>
 </params>
 <return>-</return>
*/
procedure putInternalLink(
  p_x number,
  p_y number,
  p_w number,
  p_h number,
  p_link pls_integer
  );

/**
 <name>putExternalLink</name>
 <type>Procedure</type>
 <desc>
  Puts a link area on the page that point to an external link.
 </desc>
 <version>v3.0.0</version>
 <params>
  <param>
   <param-def>p_x number</param-def>
   <param-desc>X coordinate of the top left corner of the area</param-desc>
  </param>
  <param>
   <param-def>p_y number</param-def>
   <param-desc>Y coordinate of the top left corner of the area</param-desc>
  </param>
  <param>
   <param-def>p_w number</param-def>
   <param-desc>Link area width</param-desc>
  </param>
  <param>
   <param-def>p_h number</param-def>
   <param-desc>Link area height</param-desc>
  </param>
  <param>
   <param-def>p_link varchar2</param-def>
   <param-desc>External link</param-desc>
  </param>
 </params>
 <return>-</return>
*/
procedure putExternalLink(
  p_x number,
  p_y number,
  p_w number,
  p_h number,
  p_link varchar2
  );

/**
 <name>putLink</name>
 <type>Procedure</type>
 <desc>
  Puts a link area on the page that point to an external link. If p_link is a number then internal if string then external link.
 </desc>
 <version>v3.0.0</version>
 <params>
  <param>
   <param-def>p_x number</param-def>
   <param-desc>X coordinate of the top left corner of the area</param-desc>
  </param>
  <param>
   <param-def>p_y number</param-def>
   <param-desc>Y coordinate of the top left corner of the area</param-desc>
  </param>
  <param>
   <param-def>p_w number</param-def>
   <param-desc>Link area width</param-desc>
  </param>
  <param>
   <param-def>p_h number</param-def>
   <param-desc>Link area height</param-desc>
  </param>
  <param>
   <param-def>p_link varchar2</param-def>
   <param-desc>Identification of the link</param-desc>
  </param>
 </params>
 <return>-</return>
*/
procedure putLink(
  p_x number,
  p_y number,
  p_w number,
  p_h number,
  p_link varchar2
  );

/**
 <name>setStrokingColor</name>
 <type>Procedure</type>
 <desc>
  Sets the color for stroking operation. Colors must be specified according to the RGB pallet.
 </desc>
 <version>v3.0.0</version>
 <params>
  <param>
   <param-def>p_color plpdf_type.t_color</param-def>
   <param-desc>color</param-desc>
  </param>
 </params>
 <return>-</return>
*/
procedure setStrokingColor(
  p_color plpdf_type.t_color
  );

/**
 <name>setNonStrokingColor</name>
 <type>Procedure</type>
 <desc>
  Sets the color for non stroking operation. Colors must be specified according to the RGB pallet.
 </desc>
 <version>v3.0.0</version>
 <params>
  <param>
   <param-def>p_color plpdf_type.t_color</param-def>
   <param-desc>color</param-desc>
  </param>
 </params>
 <return>-</return>
*/
procedure setNonStrokingColor(
  p_color plpdf_type.t_color
  );

/**
 <name>addJavaScript</name>
 <type>Procedure</type>
 <desc>
  Add JavaScript inside the PDF. See Acrobat JavaScript Reference.
 </desc>
 <version>v3.0.0</version>
 <params>
 <param>
   <param-def>p_name varchar2</param-def>
   <param-desc>Name of the script</param-desc>
  </param>
  <param>
   <param-def>p_text varchar2</param-def>
   <param-desc>The script</param-desc>
  </param>
 </params>
 <return>-</return>
*/
procedure addJavaScript(
  p_name varchar2,
  p_text varchar2
  );

/**
 <name>LineWidth</name>
 <type>Procedure</type>
 <desc>
  Sets the thickness of lines drawn after this statement.
 </desc>
 <version>v3.0.0</version>
 <params>
  <param>
   <param-def>p_width number</param-def>
   <param-desc>Line thickness</param-desc>
  </param>
 </params>
 <return>-</return>
*/
procedure LineWidth(
  p_width number
  );

/**
 <name>crInternalLink</name>
 <type>Function</type>
 <desc>
  Creates an internal link, and returns its ID. An internal link can be used
  to take the cursor to another place within the document by clicking over it.
 </desc>
 <version>v3.0.0</version>
 <params>-</params>
 <return>
  <return-def>number</return-def>
  <return-desc>The ID of the internal link</return-desc>
 </return>
*/
function crInternalLink
  return pls_integer;

/**
 <name>setLinkDest</name>
 <type>Procedure</type>
 <desc>
  Sets the destination page and position for the internal link.
 </desc>
 <version>v3.0.0</version>
 <params>
  <param>
   <param-def>p_link number</param-def>
   <param-desc>Internal link ID returned by CrInternalLink</param-desc>
  </param>
  <param>
   <param-def>p_y number default 0</param-def>
   <param-desc>Y position on the page, if null then the current Y coordinate, if 0 then the top of the page</param-desc>
  </param>
  <param>
   <param-def>p_page number default null</param-def>
   <param-desc>Page number, if null then the current page</param-desc>
  </param>
 </params>
 <return>-</return>
*/
procedure setLinkDest(
  p_link pls_integer,
  p_page pls_integer default null,
  p_y number default null
  );

/**
 <name>crInternalLinkDest</name>
 <type>Function</type>
 <desc>
  Create the destination page and position for the internal link.
 </desc>
 <version>v3.0.0</version>
 <params>
  <param>
   <param-def>p_page pls_integer default null</param-def>
   <param-desc>Page number, if null then the current page</param-desc>
  </param>
  <param>
   <param-def>p_y number default null</param-def>
   <param-desc>Y position on the page, if null then the current Y coordinate, if 0 then the top of the page</param-desc>
  </param>
 </params>
 <return>Internal link ID</return>
*/
function crInternalLinkDest(
  p_page pls_integer default null,
  p_y number default null
  ) return pls_integer;

/**
 <name>setPrintFont</name>
 <type>Procedure</type>
 <desc>
  Sets the font and its properties.
 </desc>
 <version>v3.0.0</version>
 <params>
  <param>
   <param-def>p_family varchar2 default NULL</param-def>
   <param-desc>Font type</param-desc>
   <param-ul>
    <param-li>Helvetica (Arial, Sans Serif)</param-li>
    <param-li>Courier</param-li>
    <param-li>Times (Serif)</param-li>
    <param-li>Symbol</param-li>
    <param-li>ZapfDingBats</param-li>
    <param-li>If TTF fonts were added using the AddTTF procedure then they may be used here as well.</param-li>
   </param-ul>
  </param>
  <param>
   <param-def>p_style varchar2 default NULL</param-def>
   <param-desc>Font style</param-desc>
   <param-ul>
    <param-li>null: Regular</param-li>
    <param-li>B: Bold</param-li>
    <param-li>I: Italic</param-li>
    <param-li>U: Underline</param-li>
    <param-li>The styles may be combined (i.e. BI will be Bold and Italic)</param-li>
   </param-ul>
  </param>
  <param>
   <param-def>p_size number default null</param-def>
   <param-desc>Font size, null means the current size. If no size has been specified then value is 12. </param-desc>
  </param>
 </params>
 <return>-</return>
*/
procedure setPrintFont(
  p_family varchar2 default null,
  p_style varchar2 default null,
  p_size number default null
  );

/**
 <name>setPrintFontSize</name>
 <type>Procedure</type>
 <desc>
  Sets the font size. All other font properties remain unchanged.
 </desc>
 <version>v3.0.0</version>
 <params>
  <param>
   <param-def>p_size number</param-def>
   <param-desc>Font size</param-desc>
  </param>
 </params>
 <return>-</return>
*/
procedure setPrintFontSize(
  p_size number
  );

/**
 <name>setRotate</name>
 <type>Procedure</type>
 <desc>
  Sets the degree of rotation. Text and images can both be rotated.
 </desc>
 <version>v3.0.0</version>
 <params>
  <param>
   <param-def>p_angle number</param-def>
   <param-desc>Angle/degree of rotation</param-desc>
  </param>
  <param>
   <param-def>p_x number default null</param-def>
   <param-desc>X coordinate of the rotation point, if null then the X coordinate of the cursor</param-desc>
  </param>
  <param>
   <param-def>p_y number default null</param-def>
   <param-desc>Y coordinate of the rotation point, if null then the Y coordinate of the cursor</param-desc>
  </param>
 </params>
 <return>-</return>
*/
procedure setRotate(
  p_angle number,
  p_x number default null,
  p_y number default null
  );

/**
 <name>startClipping</name>
 <type>Procedure</type>
 <desc>
  Starts a clipping frame. The clipping limits the region that can be paint. The clipping affects for
  objects wich located between StartClipping and EndClipping.
 </desc>
 <version>v3.0.0</version>
 <params>
  <param>
   <param-def>p_x number</param-def>
   <param-desc>X coordinate for the start of the clipping frame</param-desc>
  </param>
  <param>
   <param-def>p_y number</param-def>
   <param-desc>Y coordinate for the start of the clipping frame</param-desc>
  </param>
  <param>
   <param-def>p_w number</param-def>
   <param-desc>height of clipping frame</param-desc>
  </param>
  <param>
   <param-def>p_h number</param-def>
   <param-desc>width of clipping frame</param-desc>
  </param>
 </params>
 <return>-</return>
*/
procedure startClipping(
  p_x number,
  p_y number,
  p_w number,
  p_h number
  );

/**
 <name>EndClipping</name>
 <type>Procedure</type>
 <desc>
  End a cliping frame.  See <link>StartClipping</link>
 </desc>
 <version>v3.0.0</version>
 <params>-</params>
 <return>-</return>
*/
procedure EndClipping;

/**
 <name>isInClipping</name>
 <type>Function</type>
 <desc>
  Get the state of clipping.
 </desc>
 <version>v3.0.0</version>
 <params>-</params>
 <return>
  <return-def>boolean</return-def>
  <return-desc>in clipping or not</return-desc>
 </return>
*/
function isInClipping
  return boolean;

/**
 <name>startOpacity</name>
 <type>Procedure</type>
 <desc>
  Starts a transparency. The transparency affects for
  objects wich located between StartOpacity and EndOpacity.
 </desc>
 <version>v3.0.0</version>
 <params>
  <param>
   <param-def>p_val number</param-def>
   <param-desc>can be a value from 0.0 - 1.0. A lower value makes the element more transparent.</param-desc>
  </param>
 </params>
 <return>-</return>
*/
procedure startOpacity(
  p_val number
  );

/**
 <name>startOptCont</name>
 <type>Procedure</type>
 <desc>
  Start an optional content.
 </desc>
 <version>v3.0.0</version>
 <params>
  <param>
   <param-def>p_name varchar2</param-def>
   <param-desc>name of optional content</param-desc>
  </param>
  <param>
   <param-def>p_state boolean default true</param-def>
   <param-desc>dafault state is visible or invisible</param-desc>
  </param>
  <param>
   <param-def>p_ui_order boolean default true</param-def>
   <param-desc>insert into the state array</param-desc>
  </param>
  <param>
   <param-def>p_print boolean default true</param-def>
   <param-desc>printed out</param-desc>
  </param>
 </params>
 <return>-</return>
*/
procedure startOptCont(
  p_name varchar2,
  p_state boolean default true,
  p_ui_order boolean default true,
  p_print boolean default true
  );

--v4.10
procedure startScaling(
  p_w number,
  p_h number
  );

--v4.10
procedure endScaling;

------------------------------------------------------------------------------
/**
 <newpage/>
 <h1>Template Related Commands</h1>
 <br/>
*/
------------------------------------------------------------------------------

/**
 <name>InsTemplate</name>
 <type>Function</type>
 <desc>
  Insert a template into the pdf.
 </desc>
 <version>v3.0.0</version>
 <params>
  <param>
   <param-def>p_tpl plpdf_type.tr_tpl_data</param-def>
   <param-desc>template data</param-desc>
  </param>
 </params>
 <return>
  <return-def>pls_integer</return-def>
  <return-desc>template ID</return-desc>
 </return>
*/
function InsTemplate(
  p_tpl plpdf_type.tr_tpl_data
  ) return pls_integer;

/**
 <name>setPageTemplate</name>
 <type>Procedure</type>
 <desc>
  Use a template.
 </desc>
 <version>v4.9.2</version>
 <params>
  <param>
   <param-def>p_page pls_integer default null</param-def>
   <param-desc>page, Null means current page</param-desc>
  </param>
  <param>
   <param-def>p_tplidx pls_integer</param-def>
   <param-desc>template ID</param-desc>
  </param>
  <param>
   <param-def>p_fittopage boolean default true</param-def>
   <param-desc>fit template to page size</param-desc>
  </param>
  <param>
   <param-def>p_rotate90 boolean default false</param-def>
   <param-desc>rotate template right with 90</param-desc>
  </param>
 </params>
 <return>-</return>
*/
procedure setPageTemplate(
  p_page pls_integer default null,
  p_tplidx pls_integer,
  p_fittopage boolean default true,
  p_rotate90 boolean default false
  );

/**
 <name>setPageDefaultTemplate</name>
 <type>Procedure</type>
 <desc>
  Set template for all new page. Remove template when p_tplidx is null.
 </desc>
 <version>v3.0.0</version>
 <params>
  <param>
   <param-def>p_tplidx pls_integer</param-def>
   <param-desc>template ID</param-desc>
  </param>
  <param>
   <param-def>p_fittopage boolean default true</param-def>
   <param-desc>fit template to page size</param-desc>
  </param>
 </params>
 <return>-</return>
*/
procedure setPageDefaultTemplate(
  p_tplidx pls_integer,
  p_fittopage boolean default true
  );


/**
 <name>UseTemplate</name>
 <type>Procedure</type>
 <desc>
  Use a template in the page content.
 </desc>
 <version>v4.9.3</version>
 <params>
  <param>
   <param-def>p_tplidx pls_integer</param-def>
   <param-desc>template ID</param-desc>
  </param>
  <param>
   <param-def>p_fittopage boolean default true</param-def>
   <param-desc>fit template to page size</param-desc>
  </param>
 </params>
 <return>-</return>
*/
procedure UseTemplate(
  p_tplidx pls_integer,
  p_fittopage boolean default true
  );

------------------------------------------------------------------------------
/**
 <newpage/>
 <h1>Digital Signature Related Commands</h1>
 <br/>
*/
------------------------------------------------------------------------------

/**
 <name>setDigSig</name>
 <type>Procedure</type>
 <desc>
  Set digital signature appearances.
 </desc>
 <version>v3.0.0</version>
 <params>
  <param>
   <param-def>p_Name varchar2 default null</param-def>
   <param-desc>The name of the person or authority signing the document.</param-desc>
  </param>
  <param>
   <param-def>p_Location varchar2 default null</param-def>
   <param-desc>The CPU host name or physical location of the signing.</param-desc>
  </param>
  <param>
   <param-def>p_Reason varchar2 default null</param-def>
   <param-desc>The reason for the signing, such as (I agree?).</param-desc>
  </param>
  <param>
   <param-def>p_ContactInfo varchar2 default null</param-def>
   <param-desc>Information provided by the signer to allow a recipient to contact the signer to verify the signature; for example, a phone number.</param-desc>
  </param>
  <param>
   <param-def>p_access_perms</param-def>
   <param-desc>The access permissions granted for this document. Defined values are:</param-desc>
   <param-ul>
     <param-li>1: No changes to the document are permitted; any change to the document invalidates the signature.</param-li>
     <param-li>2: Permitted changes are filling in forms, instantiating page templates, and signing; other changes invalidate the signature.</param-li>
     <param-li>3: Permitted changes are those allowed by 2, as well as annotation creation, deletion, and modification; other changes invalidate the signature.</param-li>
    </param-ul>
  </param>
 </params>
 <return>-</return>
*/
procedure setDigSig(
  p_id pls_integer,
  p_Name varchar2 default null,
  p_Location varchar2 default null,
  p_Reason varchar2 default null,
  p_ContactInfo varchar2 default null,
  p_access_perms number default 1
  );

/**
 <name>setNeedAppearances</name>
 <type>Procedure</type>
 <desc>
  Set Need Appearances property for Acroform fields.
 </desc>
 <version>v3.0.0</version>
 <params>-</params>
 <return>-</return>
*/
procedure setNeedAppearances;

/**
 <name>getCalcCurrPageNum</name>
 <type>Function</type>
 <desc>
  Gets the calculated number of the page.
 </desc>
 <version>v3.0.0</version>
 <params>
  <param>
   <param-def>p_page pls_integer</param-def>
   <param-desc>The page number</param-desc>
  </param>
 </params>
 <return>
  <return-def>pls_integer</return-def>
  <return-desc>page number</return-desc>
 </return>
*/
function getCalcCurrPageNum(
  p_page pls_integer
  ) return pls_integer;

/**
 <name>getCalcCurrPageNum</name>
 <type>Function</type>
 <desc>
  Gets the calculated amount number of the pages.
 </desc>
 <version>v3.0.0</version>
 <params>
  <param>
   <param-def>p_page pls_integer</param-def>
   <param-desc>The page number</param-desc>
  </param>
 </params>
 <return>
  <return-def>pls_integer</return-def>
  <return-desc>number of pages</return-desc>
 </return>
*/
function getCalcNumberOfPages(
  p_page pls_integer
  ) return pls_integer;

------------------------------------------------------------------------------
--INTERNAL PROCEDURES and FUNCTIONS
function getFontInfo(
  p_fontkey varchar2
  ) return plpdf_type.t_font_info;
--v4.2.0
function getFontCW(
  p_family varchar2,
  p_style varchar2
  ) return plpdf_type.t_cw;
--
procedure font(
  p_family varchar2 default null,
  p_style varchar2 default null,
  p_size number default null,
  p_check boolean default true
  );
--
procedure font(
  p_fk plpdf_type.t_fk,
  p_size number default null,
  p_underline boolean default null,
  p_check boolean default true
  );
--
function GetTextWidthFontKey(
  p_fk plpdf_type.t_fk,
  p_size number default null,
  p_s varchar2
  ) return number;
--
function GetPrintFontStyle1
  return varchar2;
--
function GetFontInfoCurr
  return plpdf_type.t_font_info;
--
function getEncrypted
  return boolean;
--
function getEncrytionKey
  return raw;
--
function getcurrobjnum
  return pls_integer;
--
function inHeader
  return boolean;
--
function inFooter
  return boolean;
--
function inHeaderFooter
  return boolean;
--
function getPageNoCp(
  p_index pls_integer
  ) return pls_integer;
--
function getPrintFontIndex
  return number;
--
procedure dasharray2(
  p_on number default null,
  p_off number default null
  );
--
procedure dasharray(
  p_array plpdf_type.t_numbers
  );
--
procedure setCSnonstroking(
  p_name varchar2
  );
--
procedure tillingpattern(
  p_n varchar2,
  p_color plpdf_type.t_color  default null
  );
--
procedure strokingcolor(
  p_r number,
  p_g number,
  p_b number
  );
--
procedure nonstrokingcolor(
  p_r number,
  p_g number,
  p_b number
  );
--
procedure saveGS;
--
procedure restoreGS;
--
procedure text(
  p_x number,
  p_y number,
  p_txt varchar2
  );
--
function getU
  return boolean;
--
procedure underline(
  p_x number,
  p_y number,
  p_w number
  );
--
procedure underline(
  p_x number,
  p_y number,
  p_txt varchar2
  );
--
procedure wordspace(
  p_ws number
  );
--
procedure setWordspace(
  p_ws number
  );
--
procedure setTextScalePct(
  p_scale number
  );
--
procedure setCharSpace(
  p_cs number
  );
--
function getFontUPCurr --underlinePosition
  return number;
--
function getFontUTCurr --underlineThickness
  return number;
--
function getFontIdx(
  p_fontkey varchar2
  ) return pls_integer;
--
function getnextobjnum
  return pls_integer;
--
procedure putstreamobj(
  p_objnum plpdf_type.t_objnum,
  p_stream in out nocopy blob,
  p_needcomp boolean default true
  );
--
-------------------------------------------------------------------------------
/**
 <newpage/>
 <h1>PL/PDF v2 Compatible procedures and functions</h1>
 <br/>
*/
------------------------------------------------------------------------------
/**
 <name>LineBreak</name>
 <type>Procedure</type>
 <desc>
  Line break. Cursor is placed at the start of the next line.
 </desc>
 <version>v3.0.0</version>
 <params>
  <param>
   <param-def>p_h number default NULL</param-def>
   <param-desc>Height of the line break. If not set, then the last printed cells height is the height by default.</param-desc>
  </param>
 </params>
 <return>-</return>
*/
procedure LineBreak(
  p_h number default null
  );

/**
 <name>setLineWidth</name>
 <type>Procedure</type>
 <desc>
  Sets the thickness of lines drawn after this statement.
 </desc>
 <version>v3.0.0</version>
 <params>
  <param>
   <param-def>p_width number</param-def>
   <param-desc>Line thickness</param-desc>
  </param>
 </params>
 <return>-</return>
*/
procedure setLineWidth(
  p_width number
  );

/**
 <name>getLineWidth</name>
 <type>Function</type>
 <desc>
  Gets the current line size.
 </desc>
 <version>v3.0.0</version>
 <params>-</params>
 <return>
  <return-def>number</return-def>
  <return-desc>Line size</return-desc>
 </return>
*/
function getLineWidth
  return number;

/**
 <name>DrawLine</name>
 <type>Procedure</type>
 <desc>
  Draws a line between the two positions on the page.
 </desc>
 <version>v3.0.0</version>
 <params>
  <param>
   <param-def>p_x1 number</param-def>
   <param-desc>X coordinate for the start of the line</param-desc>
  </param>
  <param>
   <param-def>p_y1 number</param-def>
   <param-desc>Y coordinate for the start of the line</param-desc>
  </param>
  <param>
   <param-def>p_x2 number</param-def>
   <param-desc>X coordinate for the end of the line</param-desc>
  </param>
  <param>
   <param-def>p_y2 number</param-def>
   <param-desc>Y coordinate for the end of the line</param-desc>
  </param>
 </params>
 <return>-</return>
*/
procedure DrawLine(
  p_x1 number,
  p_y1 number,
  p_x2 number,
  p_y2 number
  );

/**
 <name>DrawRect</name>
 <type>Procedure</type>
 <desc>
  Draws a rectangle on the page.
 </desc>
 <version>v3.0.0</version>
 <params>
  <param>
  <param-def>p_x number</param-def>
  <param-desc>X coordinate for the top left corner of the rectangle</param-desc>
  </param>
  <param>
   <param-def>p_y number</param-def>
   <param-desc>Y coordinate for the top left corner of the rectangle</param-desc>
  </param>
  <param>
   <param-def>p_w number</param-def>
   <param-desc>The width of the rectangle</param-desc>
  </param>
  <param>
   <param-def>p_h number</param-def>
   <param-desc>The height of the rectangle</param-desc>
  </param>
  <param>
   <param-def>p_style varchar2 default NULL</param-def>
   <param-desc>Style</param-desc>
   <param-ul>
    <param-li>D: No fill, border line</param-li>
    <param-li>F: Fill, no border line</param-li>
    <param-li>DF or FD: Fill and border line</param-li>
    </param-ul>
  </param>
 </params>
 <return>-</return>
*/
procedure DrawRect(
  p_x number,
  p_y number,
  p_w number,
  p_h number,
  p_style varchar2 default null
  );

/**
 <name>drawRoundedRect</name>
 <type>Procedure</type>
 <desc>
  Draws a rectangle on the page.
 </desc>
 <version>v3.0.0</version>
 <params>
  <param>
  <param-def>p_x number</param-def>
  <param-desc>X coordinate for the top left corner of the rectangle</param-desc>
  </param>
  <param>
   <param-def>p_y number</param-def>
   <param-desc>Y coordinate for the top left corner of the rectangle</param-desc>
  </param>
  <param>
   <param-def>p_w number</param-def>
   <param-desc>The width of the rectangle</param-desc>
  </param>
  <param>
   <param-def>p_h number</param-def>
   <param-desc>The height of the rectangle</param-desc>
  </param>
   <param>
   <param-def>p_r number</param-def>
   <param-desc>The radius of the corners</param-desc>
  </param>
  <param>
   <param-def>p_style varchar2 default NULL</param-def>
   <param-desc>Style</param-desc>
   <param-ul>
    <param-li>D: No fill, border line</param-li>
    <param-li>F: Fill, no border line</param-li>
    <param-li>DF or FD: Fill and border line</param-li>
    </param-ul>
  </param>
 </params>
 <return>-</return>
*/
procedure drawRoundedRect(
  p_x number,
  p_y number,
  p_w number,
  p_h number,
  p_r number,
  p_style varchar2 default null
  );

/**
 <name>DrawEllipse</name>
 <type>Procedure</type>
 <desc>
  Draws an ellipse on the current page.
 </desc>
 <version>v3.0.0</version>
 <params>
  <param>
   <param-def>p_x number</param-def>
   <param-desc>X coordinate of the center of the ellipse</param-desc>
  </param>
  <param>
   <param-def>p_y number</param-def>
   <param-desc>Y coordinate of the center of the ellipse</param-desc>
  </param>
  <param>
   <param-def>p_rx number</param-def>
   <param-desc>horizontal radius of the ellipse</param-desc>
  </param>
  <param>
   <param-def>p_ry number</param-def>
   <param-desc>vertical radius of the ellipse</param-desc>
  </param>
  <param>
   <param-def>p_style varchar2 default 'D'</param-def>
   <param-desc>Style</param-desc>
   <param-ul>
    <param-li>D: No fill, border line</param-li>
    <param-li>F: Fill, no border line</param-li>
    <param-li>DF or FD: Fill and border line</param-li>
   </param-ul>
  </param>
 </params>
 <return>-</return>
*/
procedure DrawEllipse(
  p_x number,
  p_y number,
  p_rx number,
  p_ry number,
  p_style varchar2 default 'D'
  );

/**
 <name>DrawCircle</name>
 <type>Procedure</type>
 <desc>
  Draws a cirle on the current page.
 </desc>
 <version>v3.0.0</version>
 <params>
  <param>
   <param-def>p_x number</param-def>
   <param-desc>X coordinate of the center of the circle</param-desc>
  </param>
   <param>
   <param-def>p_y number</param-def>
  <param-desc>Y coordinate of the center of the circle</param-desc>
  </param>
  <param>
   <param-def>p_r number</param-def>
   <param-desc>the radius of the circle</param-desc>
  </param>
  <param>
   <param-def>p_style varchar2 default NULL</param-def>
   <param-desc>Style</param-desc>
   <param-ul>
    <param-li>D: No fill, border line</param-li>
    <param-li>F: Fill, no border line</param-li>
    <param-li>DF or FD: Fill and border line</param-li>
   </param-ul>
  </param>
 </params>
 <return>-</return>
*/
procedure DrawCircle(
  p_x number,
  p_y number,
  p_r number,
  p_style varchar2 default 'D'
  );

/**
 <name>drawSector</name>
 <type>Procedure</type>
 <desc>
  Draws a sector on the current page.
 </desc>
 <version>v3.0.0</version>
 <params>
  <param>
   <param-def>p_xc number</param-def>
   <param-desc>X coordinate of the center of the sector</param-desc>
  </param>
  <param>
   <param-def>p_yc number</param-def>
   <param-desc>Y coordinate of the center of the sector</param-desc>
  </param>
  <param>
   <param-def>p_r number</param-def>
   <param-desc>Radius of the sector</param-desc>
  </param>
  <param>
   <param-def>p_a number</param-def>
   <param-desc>Starting point of the sector</param-desc>
  </param>
  <param>
   <param-def>p_b number</param-def>
   <param-desc>End point of the sector</param-desc>
  </param>
  <param>
   <param-def> p_style varchar2 default 'FD'</param-def>
   <param-desc>Style</param-desc>
   <param-ul>
    <param-li>D: No fill, border line</param-li>
    <param-li>F: Fill, no border line</param-li>
    <param-li>DF vagy FD: Fill and border line</param-li>
   </param-ul>
  </param>
  <param>
   <param-def>p_cw boolean default true</param-def>
   <param-desc>Draw direction</param-desc>
   <param-ul>
    <param-li>true: Clockwise</param-li>
    <param-li>false: Anti-clockwise</param-li>
   </param-ul>
  </param>
  <param>
   <param-def>p_o number default 90</param-def>
   <param-desc>origin of angles</param-desc>
  </param>
 </params>
 <return>-</return>
*/
procedure drawSector(
  p_xc number,
  p_yc number,
  p_r number,
  p_a number,
  p_b number,
  p_style varchar2 default 'FD',
  p_cw boolean default true,
  p_o number default 90
  );

/**
 <name>DrawPolygon</name>
 <type>Procedure</type>
 <desc>
  Draws a lines (polygon) between the points on the page.
 </desc>
 <version>v3.0.0</version>
 <params>
  <param>
   <param-def>p_points plpdf_type.t_points</param-def>
   <param-desc>set of ordered points</param-desc>
  </param>
  <param>
   <param-def>p_style varchar2 default 'D'</param-def>
   <param-desc>Style</param-desc>
   <param-ul>
    <param-li>D: No fill, border line</param-li>
    <param-li>F: Fill, no border line</param-li>
    <param-li>DF or FD: Fill and border line</param-li>
   </param-ul>
  </param>
 </params>
 <return>-</return>
*/
procedure drawPolygon(
  p_points plpdf_type.t_points,
  p_style varchar2 default 'D'
  );

/**
 <name>PrintText</name>
 <type>Procedure</type>
 <desc>
  Prints a text starting from the specified position.
 </desc>
 <version>v3.0.0</version>
 <params>
  <param>
   <param-def>p_x number</param-def>
   <param-desc>X coordinate for the start of the text</param-desc>
  </param>
  <param>
   <param-def>p_y number</param-def>
   <param-desc>Y coordinate for the start of the text</param-desc>
  </param>
  <param>
   <param-def>p_txt varchar2</param-def>
   <param-desc>Text</param-desc>
  </param>
 </params>
 <return>-</return>
*/
procedure PrintText(
  p_x number,
  p_y number,
  p_txt varchar2
  );

/**
 <name>setCellMargin</name>
 <type>Procedure</type>
 <desc>
  Sets the cell margin size.
 </desc>
 <version>v3.0.0</version>
 <params>
  <param>
   <param-def>p_margin number</param-def>
   <param-desc>Margin size</param-desc>
  </param>
 </params>
 <return>-</return>
*/
procedure setCellMargin(
  p_margin number
  );

/**
 <name>getCellMargin</name>
 <type>Function</type>
 <desc>
  Gets the cell margin size.
 </desc>
 <version>v3.0.0</version>
 <params>-</params>
 <return>
  <return-def>number</return-def>
  <return-desc>Margin size</return-desc>
 </return>
*/
function getCellMargin
  return number;

/**
 <name>setCellBottomMargin</name>
 <type>Procedure</type>
 <desc>
  Sets the cell bottom margin size.
 </desc>
 <version>v3.0.0</version>
 <params>
  <param>
   <param-def>p_margin number</param-def>
   <param-desc>Margin size</param-desc>
  </param>
 </params>
 <return>-</return>
*/
procedure setCellBottomMargin(
  p_margin number
  );

/**
 <name>setCellLeftMargin</name>
 <type>Procedure</type>
 <desc>
  Sets the cell left margin size.
 </desc>
 <version>v3.0.0</version>
 <params>
  <param>
   <param-def>p_margin number</param-def>
   <param-desc>Margin size</param-desc>
  </param>
 </params>
 <return>-</return>
*/
procedure setCellLeftMargin(
  p_margin number
  );

/**
 <name>setCellRightMargin</name>
 <type>Procedure</type>
 <desc>
  Sets the cell right margin size.
 </desc>
 <version>v3.0.0</version>
 <params>
  <param>
   <param-def>p_margin number</param-def>
   <param-desc>Margin size</param-desc>
  </param>
 </params>
 <return>-</return>
*/
procedure setCellRightMargin(
  p_margin number
  );

/**
 <name>setCellTopMargin</name>
 <type>Procedure</type>
 <desc>
  Sets the cell top margin size.
 </desc>
 <version>v3.0.0</version>
 <params>
  <param>
   <param-def>p_margin number</param-def>
   <param-desc>Margin size</param-desc>
  </param>
 </params>
 <return>-</return>
*/
procedure setCellTopMargin(
  p_margin number
  );

/**
 <name>getCellBottomMargin</name>
 <type>Function</type>
 <desc>
  Gets the cell bottom margin size.
 </desc>
 <version>v3.0.0</version>
 <params>-</params>
 <return>
  <return-def>number</return-def>
  <return-desc>Size of bottom margin</return-desc>
 </return>
*/
function getCellBottomMargin
  return number;

/**
 <name>getCellLeftMargin</name>
 <type>Function</type>
 <desc>
  Gets the cell left margin size.
 </desc>
 <version>v3.0.0</version>
 <params>-</params>
 <return>
  <return-def>number</return-def>
  <return-desc>Size of left margin</return-desc>
 </return>
*/
function getCellLeftMargin
  return number;

/**
 <name>getCellRightMargin</name>
 <type>Function</type>
 <desc>
  Gets the cell right margin size.
 </desc>
 <version>v3.0.0</version>
 <params>-</params>
 <return>
  <return-def>number</return-def>
  <return-desc>Size of right margin</return-desc>
 </return>
*/
function getCellRightMargin
  return number;

/**
 <name>getCellTopMargin</name>
 <type>Function</type>
 <desc>
  Gets the cell top margin size.
 </desc>
 <version>v3.0.0</version>
 <params>-</params>
 <return>
  <return-def>number</return-def>
  <return-desc>Size of top margin</return-desc>
 </return>
*/
function getCellTopMargin
  return number;

/**
 <name>getTextHeight</name>
 <type>Function</type>
 <desc>
  Gets the height of the actual font.
 </desc>
 <version>v3.4.0</version>
 <params>-</params>
 <return>
  <return-def>number</return-def>
  <return-desc>The size of the actual font</return-desc>
 </return>
*/
function getTextHeight
  return number;

/**
 <name>getCellMinHeight</name>
 <type>Function</type>
 <desc>
  Gets the minimum cell height (uses current font).
 </desc>
 <version>v3.0.0</version>
 <params>- </params>
 <return>
  <return-def>number</return-def>
 </return>
*/
function getCellMinHeight
  return number;


/**
 <name>getCellMinHeightFont</name>
 <type>Function</type>
 <desc>
  Gets the minimum cell height, font is parameter.
 </desc>
 <version>v3.0.0</version>
 <params>
 <param>
   <param-def>p_family varchar2 default null</param-def>
   <param-desc>Font type, null means current font family</param-desc>
   <param-ul>
    <param-li>Helvetica (Arial, Sans Serif)</param-li>
    <param-li>Courier</param-li>
    <param-li>Times (Serif)</param-li>
    <param-li>Symbol</param-li>
    <param-li>ZapfDingBats</param-li>
    <param-li>If TTF fonts were added using the AddTTF procedure then they may be used here as well.</param-li>
   </param-ul>
  </param>
  <param>
   <param-def>p_style varchar2 default null</param-def>
   <param-desc>Font style</param-desc>
   <param-ul>
    <param-li>null: Regular</param-li>
    <param-li>B: Bold</param-li>
    <param-li>I: Italic</param-li>
    <param-li>U: Underline</param-li>
    <param-li>The styles may be combined (i.e. BI will be Bold and Italic)</param-li>
   </param-ul>
  </param>
  <param>
   <param-def>p_size number default null</param-def>
   <param-desc>Font size, null means current font size</param-desc>
  </param>
 </params>
 <return>
  <return-def>number</return-def>
 </return>
*/
function getCellMinHeightFont(
  p_family varchar2 default null,
  p_style varchar2 default null,
  p_size number default null
  ) return number;

/**
 <name>getCellMinWidth</name>
 <type>Function</type>
 <desc>
  Gets the minimum cell width (uses current font).
 </desc>
 <version>v3.0.0</version>
 <params>
 <param>
   <param-def>p_txt varchar2</param-def>
   <param-desc>Text for calculating width</param-desc>
  </param>
 </params>
 <return>
  <return-def>number</return-def>
 </return>
*/
function getCellMinWidth(
  p_txt varchar2
  ) return number;

/**
 <name>getCellMinWidthFont</name>
 <type>Function</type>
 <desc>
  Gets the minimum cell width, font is parameter.
 </desc>
 <version>v3.0.0</version>
 <params>
 <param>
   <param-def>p_family varchar2 default null</param-def>
   <param-desc>Font type, null means current font family</param-desc>
   <param-ul>
    <param-li>Helvetica (Arial, Sans Serif)</param-li>
    <param-li>Courier</param-li>
    <param-li>Times (Serif)</param-li>
    <param-li>Symbol</param-li>
    <param-li>ZapfDingBats</param-li>
    <param-li>If TTF fonts were added using the AddTTF procedure then they may be used here as well.</param-li>
   </param-ul>
  </param>
  <param>
   <param-def>p_style varchar2 default null</param-def>
   <param-desc>Font style</param-desc>
   <param-ul>
    <param-li>null: Regular</param-li>
    <param-li>B: Bold</param-li>
    <param-li>I: Italic</param-li>
    <param-li>U: Underline</param-li>
    <param-li>The styles may be combined (i.e. BI will be Bold and Italic)</param-li>
   </param-ul>
  </param>
  <param>
   <param-def>p_size number default null</param-def>
   <param-desc>Font size, null means current font size</param-desc>
  </param>
 <param>
   <param-def>p_txt varchar2</param-def>
   <param-desc>Text for calculating width</param-desc>
  </param>
 </params>
 <return>
  <return-def>number</return-def>
 </return>
*/
function getCellMinWidthFont(
  p_family varchar2 default null,
  p_style varchar2 default null,
  p_size number default null,
  p_txt varchar2
  ) return number;

--v4.1.0
function getLastCellHeight
  return number;

 /**
 <name>PrintCell</name>
 <type>Procedure</type>
 <desc>
  Draws a rectangle cell with text inside. The rectangle may have a border and fill color specified.
 </desc>
 <version>v4.8.0</version>
 <params>
  <param>
   <param-def>p_w number default 0</param-def>
   <param-desc>Rectangle width</param-desc>
   <param-ul>
    <param-li>0: the rectangle will extend till the right margin</param-li>
    <param-li>null: uses minimum cell width</param-li>
   </param-ul>
  </param>
  <param>
   <param-def>p_h number default 0</param-def>
   <param-desc>Rectangle height</param-desc>
   <param-ul>
    <param-li>null: uses minimum cell height</param-li>
    <param-li>-1: uses minimum cell height</param-li>
    <param-li>0: uses last cell height</param-li>
   </param-ul>
  </param>
  <param>
   <param-def>p_txt varchar2 default NULL</param-def>
   <param-desc>Text</param-desc>
  </param>
  <param>
   <param-def>p_border varchar2 default '0'</param-def>
   <param-desc>Rectangle border</param-desc>
   <param-ul>
    <param-li>0: no border</param-li>
    <param-li>1: outline border</param-li>
    <param-li>L: left side border</param-li>
    <param-li>T: top side border</param-li>
    <param-li>R: right side border</param-li>
    <param-li>B: bottom side border</param-li>
    <param-li>or a combination of L, T, R and B</param-li>
   </param-ul>
  </param>
  <param>
   <param-def>p_ln number default plpdf_const.beside</param-def>
   <param-desc>Cursor position after the cell is printed</param-desc>
   <param-ul>
    <param-li>0: Beside the cell</param-li>
    <param-li>1: New line</param-li>
    <param-li>2: Under the cell</param-li>
   </param-ul>
  </param>
  <param>
   <param-def>p_align varchar2 default NULL</param-def>
   <param-desc>Text alignment</param-desc>
   <param-ul>
    <param-li>L: left</param-li>
    <param-li>R: right</param-li>
    <param-li>C: </param-li>
    <param-li>J: Force Justification</param-li>
   </param-ul>
  </param>
  <param>
   <param-def>p_fill boolean default false</param-def>
   <param-desc>Fill with current fill color</param-desc>
   <param-ul>
    <param-li>false: no fill</param-li>
    <param-li>true: fill with current fill color</param-li>
   </param-ul>
  </param>
  <param>
   <param-def>p_link varchar2 default null</param-def>
   <param-desc>Link related to the text, can be a URL or an internal link ID</param-desc>
  </param>
  <param>
   <param-def>p_clipping boolean default true</param-def>
   <param-desc>Clipping the cell</param-desc>
   <param-ul>
    <param-li>false: no clipping</param-li>
    <param-li>true: clipping</param-li>
   </param-ul>
  </param>
  <param>
   <param-def>p_vert_align varchar2 default null</param-def>
   <param-desc>Text vertical alignment in cell</param-desc>
   <param-ul>
    <param-li>null: normal</param-li>
    <param-li>T: top</param-li>
    <param-li>B: bottom</param-li>
   </param-ul>
  </param>
 </params>
 <return>-</return>
*/
procedure PrintCell(
  p_w number default 0, --null,
  p_h number  default 0, --null,
  p_txt varchar2 default null,
  p_border varchar2 default '0',
  p_ln number default plpdf_const.beside,
  p_align varchar2 default null,
  p_fill boolean default false,
  p_link varchar2 default null,
  p_clipping boolean default true,
  p_vert_align varchar2 default null
  );

--v2 compatible version
procedure PrintCell2(
  p_w number default 0,
  p_h number  default 0,
  p_txt varchar2 default null,
  p_border varchar2 default '0',
  p_ln number default 0,
  p_align varchar2 default null,
  p_fill number default 0,
  p_link varchar2 default null,
  p_clipping number default 1,
  p_vert_align varchar2 default null
  );

/**
 <name>PrintMultiLineCell</name>
 <type>Procedure</type>
 <desc>
  Draws a multi line cell.
 </desc>
 <version>v3.0.0</version>
 <params>
  <param>
   <param-def>p_w number default 0</param-def>
   <param-desc>Rectangle width, if 0 then the rectangle will extend till the right margin</param-desc>
  </param>
  <param>
   <param-def>p_h number default 0</param-def>
   <param-desc>Rectangle height</param-desc>
    <param-ul>
    <param-li>null: uses minimum cell height</param-li>
    <param-li>-1: uses minimum cell height</param-li>
    <param-li>0: uses last cell height</param-li>
   </param-ul>
  </param>
  <param>
   <param-def>p_txt varchar2 default null</param-def>
   <param-desc>Text</param-desc>
  </param>
  <param>
   <param-def>p_border varchar2 default '0'</param-def>
   <param-desc>Rectangle border</param-desc>
   <param-ul>
    <param-li>0: no border</param-li>
    <param-li>1: outline border</param-li>
    <param-li>L: left side border</param-li>
    <param-li>T: top side border</param-li>
    <param-li>R: right side border</param-li>
    <param-li>B: bottom side border</param-li>
    <param-li>Or a combination of L, T, R and B</param-li>
   </param-ul>
  </param>
  <param>
   <param-def>p_align varchar2 default 'J'</param-def>
   <param-desc>Text alignment</param-desc>
   <param-ul>
    <param-li>L: left</param-li>
    <param-li>R: right</param-li>
    <param-li>C: center</param-li>
    <param-li>J: force justification</param-li>
   </param-ul>
  </param>
  <param>
   <param-def>p_fill boolean default false</param-def>
   <param-desc>Fill with current fill color</param-desc>
   <param-ul>
    <param-li>false: no fill</param-li>
    <param-li>true: fill with current fill color</param-li>
   </param-ul>
  </param>
  <param>
   <param-def>p_maxline number default 0</param-def>
   <param-desc>Maximum number of lines displayed. If 0 then all lines will be displayed.</param-desc>
  </param>
  <param>
   <param-def>p_link varchar2 default NULL</param-def>
   <param-desc>Link related to the text, can be a URL or an internal link ID</param-desc>
  </param>
  <param>
   <param-def>p_clipping boolean default true</param-def>
   <param-desc>Clipping cell</param-desc>
   <param-ul>
    <param-li>false: no clipping</param-li>
    <param-li>true: clipping</param-li>
   </param-ul>
  </param>
  <param>
   <param-def>p_indent number default 0</param-def>
   <param-desc>First line Indent</param-desc>
  </param>
  <param>
   <param-def>p_ln number default plpdf_const.newline</param-def>
   <param-desc>Cursor position after the cell is printed</param-desc>
   <param-ul>
    <param-li>0: Beside the cell</param-li>
    <param-li>1: New line</param-li>
    <param-li>2: Under the cell</param-li>
   </param-ul>
  </param>
 </params>
 <return>-</return>
*/
procedure PrintMultiLineCell(
  p_w number default 0,
  p_h number default 0,
  p_txt varchar2 default null,
  p_border varchar2 default '0',
  p_align varchar2 default 'J',
  p_fill boolean default false,
  p_maxline number default 0,
  p_link varchar2 default null,
  p_clipping boolean default true,
  p_indent number default 0,
  p_ln number default plpdf_const.newline
  );

--v4.7.0
procedure PrintMultiLineCellClob(
  p_w number default 0,
  p_h number default 0,
  p_txt clob,
  p_border varchar2 default '0',
  p_align varchar2 default 'J',
  p_fill boolean default false,
  p_maxline number default 0,
  p_link varchar2 default null,
  p_clipping boolean default true,
  p_indent number default 0,
  p_ln number default plpdf_const.newline --v2.0.1
  );

--v2 compatible version
procedure PrintMultiLineCell2(
  p_w number default 0,
  p_h number default 0,
  p_txt varchar2 default null,
  p_border varchar2 default '0',
  p_align varchar2 default 'J',
  p_fill number default 0,
  p_maxline number default 0,
  p_link varchar2 default null,
  p_clipping number default 1,
  p_indent number default 0,
  p_ln number default 1
  );

/**
 <name>PrintFlowingText</name>
 <type>Procedure</type>
 <desc>
  Prints text continuously. Text is wrapped according to the right margin.
 </desc>
 <version>v3.0.0</version>
 <params>
   <param>
   <param-def>p_h number default 0</param-def>
   <param-desc>Rectangle height</param-desc>
    <param-ul>
    <param-li>null: uses minimum cell height</param-li>
    <param-li>-1: uses minimum cell height</param-li>
    <param-li>0: uses last cell height</param-li>
   </param-ul>
  </param>
  <param>
   <param-def>p_txt varchar2 default null</param-def>
   <param-desc>Text</param-desc>
  </param>
  <param>
   <param-def>p_link varchar2 default null</param-def>
   <param-desc>URL or internal link ID</param-desc>
  </param>
  <param>
   <param-def>p_align varchar2 default NULL</param-def>
   <param-desc>Text alignment</param-desc>
   <param-ul>
    <param-li>L: left</param-li>
    <param-li>R: right</param-li>
    <param-li>C: </param-li>
    <param-li>J: Force Justification</param-li>
   </param-ul>
  </param>
  <param>
   <param-def>p_clipping boolean default true</param-def>
   <param-desc>Clipping text</param-desc>
    <param-ul>
     <param-li>false: no clipping</param-li>
     <param-li>false: clipping</param-li>
   </param-ul>
  </param>
  <param>
   <param-def>p_lastline_j varchar2 default 'L'</param-def>
   <param-desc>justified text last line alignment</param-desc>
   <param-ul>
    <param-li>L: left</param-li>
    <param-li>R: right</param-li>
    <param-li>C: center</param-li>
    <param-li>J: Force Justification</param-li>
   </param-ul>
  </param>
 </params>
 <return>-</return>
*/
procedure PrintFlowingText(
  p_h number default 0,
  p_txt varchar2 default null,
  p_link varchar2 default null,
  p_align varchar2 default null,
  p_clipping boolean default true,
  p_lastline_j varchar2 default 'L'
  );

/**
 <name>PrintFlowingTextLimit</name>
 <type>Procedure</type>
 <desc>
  Prints text continuously. Text is wrapped according to the p_min_x and p_max_x.
 </desc>
 <version>v3.0.0</version>
 <params>
   <param>
   <param-def>p_h number default 0</param-def>
   <param-desc>Rectangle height</param-desc>
    <param-ul>
    <param-li>null: uses minimum cell height</param-li>
    <param-li>-1: uses minimum cell height</param-li>
    <param-li>0: uses last cell height</param-li>
   </param-ul>
  </param>
  <param>
   <param-def>p_txt varchar2 default null</param-def>
   <param-desc>Text</param-desc>
  </param>
  <param>
   <param-def>p_link varchar2 default null</param-def>
   <param-desc>URL or internal link ID</param-desc>
  </param>
  <param>
   <param-def>p_min_x number default 0</param-def>
   <param-desc>min X position, 0 means the limit is the left margin</param-desc>
  </param>
  <param>
   <param-def>p_max_x number default 0</param-def>
   <param-desc>max X position, 0 means the limit is the right margin</param-desc>
  </param>
  <param>
   <param-def>p_align varchar2 default NULL</param-def>
   <param-desc>Text alignment</param-desc>
   <param-ul>
    <param-li>L: left</param-li>
    <param-li>R: right</param-li>
    <param-li>C: </param-li>
    <param-li>J: Force Justification</param-li>
   </param-ul>
  </param>
  <param>
   <param-def>p_clipping boolean default true</param-def>
   <param-desc>Clipping text</param-desc>
    <param-ul>
     <param-li>false: no clipping</param-li>
     <param-li>true: clipping</param-li>
   </param-ul>
  </param>
  <param>
   <param-def>p_lastline_j varchar2 default 'L'</param-def>
   <param-desc>justified text last line alignment</param-desc>
   <param-ul>
    <param-li>L: left</param-li>
    <param-li>R: right</param-li>
    <param-li>C: </param-li>
    <param-li>J: Force Justification</param-li>
   </param-ul>
  </param>
 </params>
 <return>-</return>
*/
procedure PrintFlowingTextLimit(
  p_h number default 0,
  p_txt varchar2 default null,
  p_link varchar2 default null,
  p_min_x number default 0,
  p_max_x number default 0,
  p_align varchar2 default null,
  p_clipping boolean default true,
  p_lastline_j varchar2 default 'L'
  );

/**
 <name>PrintImageCell</name>
 <type>Procedure</type>
 <desc>
  Print an image with frame.
 </desc>
 <version>v3.0.0</version>
 <params>
  <param>
   <param-def>p_w number</param-def>
   <param-desc>Width of frame</param-desc>
  </param>
  <param>
   <param-def>p_h number</param-def>
   <param-desc>Height of frame</param-desc>
  </param>
  <param>
   <param-def>p_name varchar2</param-def>
   <param-desc>name of image</param-desc>
  </param>
  <param>
   <param-def>p_data blob</param-def>
   <param-desc>image</param-desc>
  </param>
  <param>
   <param-def>p_margin number default 0</param-def>
   <param-desc>margin size of frame</param-desc>
  </param>
  <param>
   <param-def>p_border varchar2 default '0'</param-def>
   <param-desc>Rectangle border</param-desc>
   <param-ul>
    <param-li>0: no border</param-li>
    <param-li>1: outline border</param-li>
    <param-li>L: left side border</param-li>
    <param-li>T: top side border</param-li>
    <param-li>R: right side border</param-li>
    <param-li>B: bottom side border</param-li>
    <param-li>Or a combination of L, T, R and B</param-li>
   </param-ul>
  </param>
  <param>
   <param-def>p_ln number default plpdf_const.beside</param-def>
   <param-desc>Cursor position after the cell is printed</param-desc>
   <param-ul>
    <param-li>0: Beside the cell</param-li>
    <param-li>1: New line</param-li>
    <param-li>2: Under the cell</param-li>
   </param-ul>
  </param>
  <param>
   <param-def>p_fill boolean default false</param-def>
   <param-desc>Fill with current fill color</param-desc>
   <param-ul>
    <param-li>false: no fill</param-li>
    <param-li>true: fill with current fill color</param-li>
   </param-ul>
  </param>
  <param>
   <param-def>p_link varchar2 default NULL</param-def>
   <param-desc>URL or internal link ID</param-desc>
  </param>
  <param>
   <param-def>p_type varchar2 default null</param-def>
   <param-desc>when use Oracle InterMedia then ignore this param,
    when does not use Oracle InterMedia then JPG or PNG, null means JPG</param-desc>
  </param>
 </params>
 <return>-</return>
*/
procedure PrintImageCell(
  p_w number,
  p_h number,
  p_name varchar2,
  p_data in out nocopy blob,
  p_margin number default 0,
  p_border varchar2 default '0',
  p_ln number default plpdf_const.beside,
  p_fill boolean default false,
  p_link varchar2 default null,
  p_type varchar2 default null -- JPG, PNG
  );

/**
 <name>setColor4Drawing</name>
 <type>Procedure</type>
 <desc>
  Sets the line color for objects (line, rectangle, circle etc.) inserted after this statement.
  Colors must be specified according to the RGB pallet.
 </desc>
 <version>v3.0.0</version>
 <params>
  <param>
   <param-def>p_r number</param-def>
   <param-desc>Red component code, can be between 0 and 255</param-desc>
  </param>
  <param>
   <param-def>p_g number</param-def>
   <param-desc>Green component code, can be between 0 and 255</param-desc>
  </param>
  <param>
   <param-def>p_b number</param-def>
   <param-desc>Blue component code, can be between 0 and 255</param-desc>
  </param>
 </params>
 <return>-</return>
*/
procedure setColor4Drawing(
  p_r number,
  p_g number,
  p_b number
  );

/**
 <name>setColor4Drawing</name>
 <type>Procedure</type>
 <desc>
  Sets the line color for objects (line, rectangle, circle etc.) inserted after this statement.
  Colors must be specified according to the RGB pallet.
 </desc>
 <version>v3.0.0</version>
 <params>
  <param>
   <param-def>p_color plpdf_type.t_color</param-def>
   <param-desc>color</param-desc>
  </param>
 </params>
 <return>-</return>
*/
procedure setColor4Drawing(
  p_color plpdf_type.t_color
  );

/**
 <name>setColor4Filling</name>
 <type>Procedure</type>
 <desc>
  Sets the fill color for objects inserted after this statement.
  Colors must be specified according to the RGB pallet.
 </desc>
 <version>v3.0.0</version>
 <params>
  <param>
   <param-def>p_r number</param-def>
   <param-desc>Red component code, can be between 0 and 255</param-desc>
  </param>
  <param>
   <param-def>p_g number</param-def>
   <param-desc>Green component code, can be between 0 and 255</param-desc>
  </param>
  <param>
   <param-def>p_b number</param-def>
   <param-desc>Blue component code, can be between 0 and 255</param-desc>
  </param>
 </params>
 <return>-</return>
*/
procedure setColor4Filling(
  p_r number,
  p_g number,
  p_b number
  );

/**
 <name>setColor4Filling</name>
 <type>Procedure</type>
 <desc>
  Sets the fill color for objects inserted after this statement. Colors must be specified according to the RGB pallet.
 </desc>
 <version>v3.0.0</version>
 <params>
  <param>
   <param-def>p_color plpdf_type.t_color</param-def>
   <param-desc>color</param-desc>
  </param>
 </params>
 <return>-</return>
*/
procedure setColor4Filling(
  p_color plpdf_type.t_color
  );

/**
 <name>setColor4Text</name>
 <type>Procedure</type>
 <desc>
  Sets the text color for objects inserted after this statement.
  Colors must be specified according to the RGB pallet.
 </desc>
 <version>v3.0.0</version>
 <params>
  <param>
   <param-def>p_r number</param-def>
   <param-desc>Red component code, can be between 0 and 255</param-desc>
  </param>
  <param>
   <param-def>p_g number</param-def>
   <param-desc>Green component code, can be between 0 and 255</param-desc>
  </param>
  <param>
   <param-def>p_b number</param-def>
   <param-desc>Blue component code, can be between 0 and 255</param-desc>
  </param>
 </params>
 <return>-</return>
*/
procedure setColor4Text(
  p_r number,
  p_g number default -1,
  p_b number default -1
  );

/**
 <name>setColor4Text</name>
 <type>Procedure</type>
 <desc>
  Sets the text color for objects inserted after this statement. Colors must be specified according to the RGB pallet.
 </desc>
 <version>v3.0.0</version>
 <params>
  <param>
   <param-def>p_color plpdf_type.t_color</param-def>
   <param-desc>color</param-desc>
  </param>
 </params>
 <return>-</return>
*/
procedure setColor4Text(
  p_color plpdf_type.t_color
  );

/**
 <name>getColor4Drawing</name>
 <type>Function</type>
 <desc>
  Gets the line color for objects.
 </desc>
 <version>v3.0.0</version>
 <params>-</params>
 <return>
  <return-def>plpdf_type.t_color</return-def>
  <return-desc>Drawing color</return-desc>
 </return>
*/
function getColor4Drawing
  return plpdf_type.t_color;

/**
 <name>getColor4Filling</name>
 <type>Function</type>
 <desc>
  Gets the filling color for objects.
 </desc>
 <version>v3.0.0</version>
 <params>-</params>
 <return>
  <return-def>plpdf_type.t_color</return-def>
  <return-desc>Filling color</return-desc>
 </return>
*/
function getColor4Filling
  return plpdf_type.t_color;

/**
 <name>getColor4Text</name>
 <type>Function</type>
 <desc>
  Gets the color of text.
 </desc>
 <version>v3.0.0</version>
 <params>-</params>
 <return>
  <return-def>plpdf_type.t_color</return-def>
  <return-desc>Color of text</return-desc>
 </return>
*/
function getColor4Text
  return plpdf_type.t_color;

--INTERNAL
--v4.2.0
procedure setRTOL;

--v4.2.0
function getRTOL
  return boolean;

--v4.8.0
procedure checkExecuteError(
  p_proc_name varchar2
  );

--v3.0.0
procedure setFCP(
  p_val varchar2
  );
--v3.0.0
function getFCP
  return varchar2;
--v3.0.0
procedure SetC4F(
  p_val plpdf_type.t_color
  );
--v3.0.0
function getC4F
 return plpdf_type.t_color;
--v3.0.0
procedure setCF(
  p_val boolean
  );
--v3.0.0
function getCF
  return boolean;
--v3.6.0
procedure linejoin(
  p_style pls_integer
  );
--v3.6.0
function getCompress
  return boolean;
--v3.6.0
function getcontentstream
  return blob;
--
procedure resetcontentstream;
--v3.6.0
function getState
  return pls_integer;
--v3.6.0
procedure setState(
  p_state pls_integer
  );
--v3.6.0
procedure storepagetemp;
--v3.6.0
procedure setPageW(
  p_w number
  );
--v3.6.0
procedure setPageH(
  p_h number
  );
--v4.1.0
function ck2
  return boolean;
--v4.1.0
function ck3
  return boolean;
--v4.1.0
procedure vRun(
  p_key varchar2,
  p_pos pls_integer,
  p_run_enabled in out boolean,
  p_exec_num in out pls_integer
  );
--v4.3.0
procedure DrawSector2(
  p_xc number,
  p_yc number,
  p_rx number,
  p_ry number,
  p_a number,
  p_b number,
  p_style varchar2 default 'FD',
  p_cw boolean default true,
  p_o number default 90
  );
--v4.9.6
function getLPsV4
  return varchar2;
-------------------------------------------------------
end plpdf;
/

