create or replace package plpdf_chart
authid current_user is
--v5.9
--
/*
Diagram/Chart creator package
The purpose of the package is to insert diagrams or charts into PDFs generated by PL/PDF. The procedures of plpdf_chart create PL/SQL code that will create and arrange diagrams or charts according to the specifications supplied by the user.
Currently the following diagrams/charts can be created:
- Bar
- Line
- Pie
To use plpdf_chart, the user has to set a diagram area and then arrange diagram objects on this area. The objects can be:
- Axis (X,Y)
- Axis label
- Grid
- Title
- Legend
- Bar
- Point
- Line
- Sector
The diagram can be created by putting these objects in the specified order on the diagram area.
Step by step this is what the user needs to do:
- Init: This procedure signifies the creation of a diagram to PL/PDF. A diagram/chart area is created where the coordinates of the objects placed on the area are counted from the left bottom of the area (0,0). The X (horizontal) axis goes from left to right; the Y (vertical) axis goes from bottom to up. The origo (0,0) can be reset using the SetArea procedure.
- The arrangement of objects: to place an object on the diagram area, call the appropriate procedure. The placement of the object is specified by setting the X and Y coordinates. The order of placing objects is important as objects placed later may cover objects already on the diagram area (the previously placed object may become hidden).
- EndDiagram: completes the diagram.
*/
---------------------------------------------------------------------------------------------------
 type t_diagram_value is record (
   x number,
   y number,
   value number,
   label plpdf_type.v2avg,
   color plpdf_type.t_color,
   pattern plpdf_type.v2avg
   );
 type t_diagram is table of t_diagram_value index by pls_integer;
---------------------------------------------------------------------------------------------------
/*
Initializes the global parameters of the diagram.
The objects created on this diagram area will be placed within the area according to the coordinates given. By default the origo (0,0) is at the left bottom corner of the diagram area.
*/
procedure Init(
  p_x number, -- X coordinate of the left upper corner of the diagram
  p_y number, -- Y coordinate of the left upper corner of the diagram
  p_w number, -- the height of the diagram
  p_h number, -- the width of the diagram
  p_margin number default null, -- the width of the margins
  p_gap number default null, -- the default size of the gap between objects (e.g. text and line)
  p_frame boolean default false, -- specifies whether a frame will be created around the diagram area
  p_frame_lw number default null, -- the thickness of the frame
  p_frame_lc plpdf_type.t_color default null, -- the color of the frame
  p_bgc plpdf_type.t_color default null, -- the background color of the diagram
  p_text_bgc plpdf_type.t_color default null, -- the default background color of the objects on the diagram
  p_title_position varchar2 default null, -- the title position: T(op), B(ottom)
  p_def_Color4Drawing plpdf_type.t_color default null, -- the default color of the frame of objects (or the text)
  p_def_Color4Filling plpdf_type.t_color default null, -- the default fill color of the objects
  p_def_LineWidth number default null -- the default thickness of lines
  );
---------------------------------------------------------------------------------------------------
/*
Signifies the end of the diagram creation process. All global variables are reset to the values before the diagram process.
*/
procedure EndDiagram(
  p_position varchar2 default 'O' -- After the end of the diagram the current position should be: O(riginal), E(nd of chart)
  );
---------------------------------------------------------------------------------------------------
/*
To reset the origo of the diagram area.
*/
procedure SetArea(
  p_x0 number, -- the offset (to the right) from the left bottom corner of the diagram area
  p_y0 number, -- the offset (upwards) from the left bottom corner of the diagram area
  p_x_plus_length number default null,
  p_x_minus_length number default null,
  p_y_plus_length number default null,
  p_y_minus_length number default null
  );
---------------------------------------------------------------------------------------------------
/*
If an object that is not a predefined object (e.g. picture) will be placed on the diagram area, then the PL/PDF X coordinate must be converted to the diagram's X coordinate.
e.g.. plpdf.PutImage('img_name', l_blob, plpdf_diagram.ConvertX(20,... , where 20 is the coordinate according to the diagram area.
*/
function ConvertX(
  p_value number
  ) return number;
---------------------------------------------------------------------------------------------------
/*
If an object that is not a predefined object (e.g. picture) will be placed on the diagram area, then the PL/PDF Y coordinate must be converted to the diagram's Y coordinate.
*/
function ConvertY(
  p_value number
  ) return number;
---------------------------------------------------------------------------------------------------
/*
To print text on the diagram the PrintCell command must be used. This procedure sets the default cell height for the PrintCell command.
*/
procedure SetDefCellHeight(
  p_h  number
  );
---------------------------------------------------------------------------------------------------
/*
Sets the gap between objects (e.g. text and line).
*/
procedure SetGap(
  p_gap  number
  );
---------------------------------------------------------------------------------------------------
/*
Sets the diagram's t_dg type variables
*/

procedure addDGValue(
  p_var in out t_diagram, -- the variable
  p_x number default null, -- X coordinate
  p_y number default null, -- Y coordinate
  p_value number default null, -- value
  p_label varchar2 default null, -- label
  p_color plpdf_type.t_color default null, -- color
  p_pattern varchar2 -- pattern
  );
---------------------------------------------------------------------------------------------------
/*
Adds a title text to the diagram. Can only be used if a header was specified in the INIT procedure.
*/
procedure PrintTitle(
  p_text varchar2, -- title text
  p_align varchar2 default 'L', -- alignment: L(eft), R(ight), C(enter)
  p_left varchar2 default 'M', -- align to: M(argin), Y(axis)
  p_bgc plpdf_type.t_color default null -- the background color
  );
---------------------------------------------------------------------------------------------------
/*
Draws a line between the header and the diagram
*/
procedure DrawTitleSeparator(
  p_lw number default null, -- line thickness
  p_lc plpdf_type.t_color default null -- line color
  );
---------------------------------------------------------------------------------------------------
/*
Draws a vertical legend. The legend may contain a title and a number of colored boxes with the explanation of the color on the diagram
*/
procedure PrintLegend1( -- vertical legend
  p_title varchar2 default null, -- title
  p_title_position varchar2 default 'T', -- title position within the legend: T(op), B(ottom)
  p_frame boolean default false, -- frame around the legend
  p_x_position varchar2 default 'R', -- horizontal position of the legend: L(eft), R(ight), C(enter)
  p_y_position varchar2 default 'T', -- vertical position of the legend: T(op), B(ottom), C(enter)
  p_data t_diagram, -- objects within the legend: color: the color that is explained, label: the explanation
  p_lw number default null, -- thickness of the lines
  p_lc plpdf_type.t_color default null, -- color of the lines
  p_bgc plpdf_type.t_color default null -- the background color
  );
---------------------------------------------------------------------------------------------------
/*
Draws a horizontal axis passing through the origo (0,0).
*/
procedure DrawXAxis(
  p_plus_length number default null, -- the length of the axis right of the origo, if null then the axis is drawn to the end of the diagram area
  p_minus_length number default null, -- the length of the axis left of the origo, if null then the axis is drawn to the end of the diagram area
  p_label varchar2 default null, -- label of the axis
  p_label_position varchar2 default 'O', -- position of the label according to the axis: T(op), O(n axis), B(ottom)
  p_lw number default null, -- thickness of the lines
  p_lc plpdf_type.t_color default null -- color of the lines
  );
---------------------------------------------------------------------------------------------------
/*
Draws a horizontal legend.
*/
procedure DrawXLegend(
  p_pointer_length number default 0, -- length of the pointer line
  p_legends t_diagram, -- X coordinates of the labels
  p_lw number default null, -- thickness of the lines
  p_lc plpdf_type.t_color default null -- color of the lines
  );
---------------------------------------------------------------------------------------------------
procedure DrawXLegend(
  p_pointer_length number default 0, -- length of the pointer line
  p_plus_length number default null,
  p_minus_length number default null,
  p_gap number default 5,
  p_inc number default 1,
  p_lw number default null, -- thickness of the lines
  p_lc plpdf_type.t_color default null, -- color of the lines
  p_format varchar2 default null
  );
---------------------------------------------------------------------------------------------------
/*
Draws a vertical axis passing through the origo (0,0).
*/
procedure DrawYAxis(
  p_plus_length number default null, -- the length of the axis upwards from the origo, if null then the axis is drawn to the end of the diagram area
  p_minus_length number default null, -- the length of the axis downward from the origo, if null then the axis is drawn to the end of the diagram area
  p_label varchar2 default null,
  p_label_position varchar2 default 'O', -- L(eft), O(n axis), R(ight)
  p_lw number default null, -- thickness of the lines
  p_lc plpdf_type.t_color default null -- color of the lines
  );
---------------------------------------------------------------------------------------------------
/*
Draws a vertical legend.
*/
procedure DrawYLegend(
  p_pointer_length number default 0, -- length of the pointer line
  p_legends t_diagram, -- Y coordinates of the labels
  p_lw number default null, -- thickness of the lines
  p_lc plpdf_type.t_color default null -- color of the lines
  );
---------------------------------------------------------------------------------------------------
procedure DrawYLegend(
  p_pointer_length number default 0, -- length of pointer line
  p_plus_length number default null,
  p_minus_length number default null,
  p_gap number default 5, -- gap between the labels
  p_inc number default 1, -- the value of the increments
  p_lw number default null, -- thickness of the lines
  p_lc plpdf_type.t_color default null, -- color of the lines
  p_format varchar2 default null
  );
---------------------------------------------------------------------------------------------------
/*
Draws grid lines along the X axis, parallel to the Y axis.
*/
procedure DrawGridX(
  p_plus_length number default null, -- height of grid upwards from the origo
  p_minus_length number default null, -- height of grid downwards from the origo
  p_points t_diagram, -- X coordinates of the grid lines
  p_lw number default null, -- thickness of the lines
  p_lc plpdf_type.t_color default null, -- color of the lines
  p_dash number default null -- length of the line dashes
  );
---------------------------------------------------------------------------------------------------
/*
Draws grid lines along the X axis, parallel to the Y axis according to the gap specified.
*/
procedure DrawGridX(
  p_x_plus_length number default null, -- length of grid right of the origo
  p_x_minus_length number default null, -- length of grid left of the origo
  p_y_plus_length number default null, -- height of grid upwards from the origo
  p_y_minus_length number default null, -- height of grid downwards from the origo
  p_gap number default 5, -- gap between the lines
  p_lw number default null, -- thickness of the lines
  p_lc plpdf_type.t_color default null, -- color of lines
  p_dash number default null -- length of the line dashes
  );
---------------------------------------------------------------------------------------------------
/*
Draws a grid along the Y axis, parallel to the X axis.
*/
procedure DrawGridY(
  p_plus_length number default null, -- length of grid right of the origo
  p_minus_length number default null, -- length of grid left of the origo
  p_points t_diagram, -- Y coordinates of the grid lines
  p_lw number default null, -- thickness of the lines
  p_lc plpdf_type.t_color default null, -- color of lines
  p_dash number default null -- length of the line dashes
  );
---------------------------------------------------------------------------------------------------
/*
Draws grid lines along the Y axis, parallel to the X axis according to the gap specified.
*/
procedure DrawGridY(
  p_x_plus_length number default null, -- length of grid right of the origo
  p_x_minus_length number default null, -- length of grid left of the origo
  p_y_plus_length number default null, -- height of grid upwards from the origo
  p_y_minus_length number default null, -- height of grid downwards from the origo
  p_gap number default 5, -- gap between the lines
  p_lw number default null, -- thickness of the lines
  p_lc plpdf_type.t_color default null, -- color of lines
  p_dash number default null -- length of the line dashes
  );
---------------------------------------------------------------------------------------------------
/*
Draws a full grid according to the gap specified.
*/
procedure DrawGrid(
  p_x_plus_length number default null, -- length of grid right of the origo
  p_x_minus_length number default null, -- length of grid left of the origo
  p_y_plus_length number default null, -- height of grid upwards from the origo
  p_y_minus_length number default null, -- height of grid downwards from the origo
  p_gap number default 5, -- gap between the lines
  p_lw number default null, -- thickness of the lines
  p_lc plpdf_type.t_color default null, -- color of lines
  p_dash number default null -- length of the line dashes
  );
---------------------------------------------------------------------------------------------------
/*
Draws a bar.
*/
procedure DrawBar(
  p_x_offset number default 0, -- the X coordinate of the middle of the bar
  p_y_offset number default 0, -- the Y coordinate of the bottom of the bar
  p_w number, -- width
  p_h number, -- height
  p_style varchar2 default null, -- style, see DrawRectangle
  p_legend_position varchar2 default null, -- label position: L(eft), R(ight), T(op), B(ottom), (C)enter
  p_legend varchar2 default null, -- label
  p_lw number default null, -- thickness of the lines
  p_lc plpdf_type.t_color default null, -- color of lines
  p_fc plpdf_type.t_color default null, -- fill color of lines
  p_pattern varchar2 default null,  -- pattern of the bar
  p_legend_rotate90 boolean default false --Sets the rotate (90 degree) of the legend
  );
---------------------------------------------------------------------------------------------------
/*
Draws a point. The point appears as a circle with a radius.
*/
procedure DrawPoint(
  p_x_offset number default 0, -- the X coordinate of the middle of the point (circle)
  p_y_offset number default 0, -- the Y coordinate of the bottom of the point (circle)
  p_r number default null, -- radius
  p_style varchar2 default 'D', -- style, see plpdf.DrawCircle
  p_legend_position varchar2 default null, -- label position: L(eft), R(ight), T(op), B(ottom), (C)enter, (X) axis, (Y) axis
  p_legend varchar2 default null, -- label text
  p_lw number default null, -- thickness of the circle line
  p_lc plpdf_type.t_color default null, -- color of the circle line
  p_fc plpdf_type.t_color default null,  -- fill color
  p_pattern varchar2 default null -- pattern
  );
---------------------------------------------------------------------------------------------------
/*
Draws a line.
*/
procedure DrawLine(
  p_points plpdf_type.t_points, -- points through which the line passes: x: X coordinate, y: Y coordinate
  p_lw number default null, -- thickness of the line
  p_lc plpdf_type.t_color default null, -- color of the line
  p_fc plpdf_type.t_color default null,  -- fill color
  p_pattern varchar2 default null, -- pattern
  p_style varchar2 default 'D'
  );
---------------------------------------------------------------------------------------------------
/*
Draws a sector.
*/
procedure DrawSector(
  p_xc NUMBER default 0, -- the X coordinate of the middle of the sector
  p_yc NUMBER default 0, -- the Y coordinate of the middle of the sector
  p_r NUMBER, -- radius
  p_a NUMBER, -- start point of the sector (from 12 o'clock clockwise)
  p_b NUMBER, -- end point of the sector (from 12 o'clock clockwise)
  p_style VARCHAR2 DEFAULT 'FD', -- style, see plpdf.DrawCircle
  p_legend_position varchar2 default 'I', -- label position: (O)uter, (I)nner
  p_legend varchar2 default null, -- label text
  p_lw number default null, -- thickness of the line
  p_lc plpdf_type.t_color default null, -- color of the line
  p_fc plpdf_type.t_color default null, -- fill color
  p_pattern varchar2 default null -- pattern
  );
---------------------------------------------------------------------------------------------------
end;
/

