create or replace package ploffx_xlsx
authid current_user is

/**
 <config>
  <pck-name>PL/OFFX_XLSX</pck-name>
  <pck-title>XLSX processor and generator</pck-title>
  <pck-version>v5.20</pck-version>
  <header-prc>plpdf_doc_xml.plpdf_header</header-prc>
  <header-size>10</header-size>
  <footer-prc>plpdf_doc_xml.plpdf_footer</footer-prc>
  <footer-size>10</footer-size>

 </config>
*/

/**
 <h1>Introduction</h1>
 <br/>
 <p>
 PL/OFFX_XLSX is a PL/SQL based XLSX Spreadsheet document generator from Oracle databases.
 Use PL/OFFX_XLSX to quickly and easily develop applications with dynamic content but also quality presentation and printing capabilities.
 </p>
 <br/>
 <h2>Links</h2>
 <p>Overview of the XML file formats in Office 2010: <a>http://technet.microsoft.com/en-us/library/cc179190.aspx</a></p>
 <p>XML Spreadsheet document structrue: <a>http://msdn.microsoft.com/en-us/library/office/gg278316(v=office.15).aspx</a></p>
 <br/>
 <h2>Installation</h2>
 <p>
 System requirements: Oracle 11g RDBMS Release 2 or higher. Oracle 11g Express Edition is supported.
 </p>
 <p>
 PL/OFFX_XLSX is a collection of PL/SQL packages. Before using PL/OFFX_XLSX, make sure that the steps described in the Installation
  Guide were completed.
 </p>
 <br/>
 <h2>Using PL/OFFX_XLSX</h2>
 <p>
 Essentially, PL/OFFX_XLSX is a series of calls to PL/OFFX_XLSX procedures and functions from a PL/SQL script, package, procedure or function.
 If you are not familiar with the PL/SQL programming language, study the Oracle documentation.
 </p>
 <br/>
 <h2>Contact</h2>
 <p>
 Contact us at: <a>mailto:info@plpdf.com</a>
 </p>
 <newpage/>
*/
----------------------------------------------------------------------------
-- http://www.microsoft.com/typography/fonts/product.aspx?PID=148
c_default_font_name constant plx_sc.normalv2 := 'Calibri';
c_default_font_size constant number := 11;
c_default_date_format constant plx_sc.normalv2 := 'd/m/yyyy\ hh/mm/ss';
----------------------------------------------------------------------------
type t_media_1 is record (
  id   pls_integer,
  name plx_sc.normalv2,
  image blob
  );

type t_media is table of t_media_1;

type t_xlsx is record(
  content_types      ploffx_content_types.CT_Types,
  sheets             ploffx_sheet.t_sheets,
  sharedstrings      ploffx_sharedstrings.CT_Sst,
  styles             ploffx_xlsx_styles.CT_Stylesheet,
  props              ploffx_core.CT_CoreProperties,
  workbook           ploffx_workbook.CT_Workbook,
  workbook_rels      ploffx_prels.CT_Relationships,
  calcchain          ploffx_calcchain.CT_CalcChain, --v4.9.1
  media              t_media,
  orig_content_files ploffx_util.t_files --4.8.3
  );

--v4.9.0
type t_cell_arr_ref is record (
  ws pls_integer,
  from_row  pls_integer,
  from_cell  plpdf_type.v2avg,
  to_row  pls_integer,
  to_cell  plpdf_type.v2avg
  );

--v4.9.0
type t_cell_ref is record (
  ws pls_integer,
  row  pls_integer,
  cell  plpdf_type.v2avg
  );

--v4.9.0
type t_datatype_formatstring is record (
  datatype   plpdf_type.v2avg,
  formatstring plpdf_type.v2avg
  );

--v4.9.1
type t_formula_cell_ref is record (
  pos pls_integer,
  cell_ref plpdf_type.v2avg,
  row  plpdf_type.v2avg, --v5.4
  cell  plpdf_type.v2avg
  );

--v4.9.1
type t_formula_cell_refs is table of t_formula_cell_ref index by pls_integer;


/**
 <name>setTableMode</name>
 <type>procedure</type>
 <desc>
 Sets table storage for sharedstrings and sheet data in runtime.
 </desc>
 <version>v3.3.0</version>
 <params>-</params>
 <return>-</return>
*/
procedure setTableMode;

/**
 <name>isMemoryMode</name>
 <type>function</type>
 <desc>
 Checks the runtime storage setting.
 </desc>
 <version>v3.3.0</version>
 <params>-</params>
 <return>
   <return-def>boolean</return-def>
   <return-desc>boolean value of the check</return-desc>
 </return>
*/
function isMemoryMode
  return boolean;

/**
 <name>setClobMode</name>
 <type>procedure</type>
 <desc>
 Sets clob storage for sharedstrings and sheet data in memory instead of structures.
 </desc>
 <version>v3.6.0</version>
 <params>-</params>
 <return>-</return>
*/
procedure setClobMode;

/**
 <name>setSharedstringUnique</name>
 <type>procedure</type>
 <desc>
 Sets unique checking execution for string table storage.
 </desc>
 <version>v4.9.0</version>
 <params>-</params>
 <return>-</return>
*/
procedure setSharedstringUnique;

/**
 <name>isSharedstringUnique</name>
 <type>function</type>
 <desc>
 Checks the shared strings storage setting.
 </desc>
 <version>v4.9.0</version>
 <params>-</params>
 <return>
 <return-def>boolean</return-def>
  <return-desc>boolean value of the check</return-desc>
 </return>
*/
function isSharedstringUnique
  return boolean;

/**
 <name>Init</name>
 <type>procedure</type>
 <desc>
 Initializes generator program variables.
 </desc>
 <version>v5.2</version>
 <params>
 <param>
   <param-def>p_date1904 boolean default false</param-def>
   <param-desc>date system is 1904 (true) or 1900 (false)</param-desc>
  </param>
 </params>
 <return>-</return>
*/
procedure init(
  p_date1904 boolean default false
  );

/**
 <name>setProducer</name>
 <type>procedure</type>
 <desc>
 Set the Producer property of the XLSX file.
 </desc>
 <version>v4.7.0</version>
 <params>
 <param>
   <param-def>p_producer varchar2</param-def>
   <param-desc>The name of the producer</param-desc>
  </param>
 </params>
 <return>-</return>
*/
procedure setProducer(
  p_producer varchar2
  );

/**
 <name>setCurrentFormat</name>
 <type>procedure</type>
 <desc>
 Sets the format to actual format.
 </desc>
 <version>v1.0.0</version>
 <params>
   <param>
   <param-def>p_idx pls_integer</param-def>
   <param-desc>Index of format</param-desc>
  </param>
 </params>
 <return>-</return>
*/
procedure setCurrentFormat(
  p_idx pls_integer
  );

/**
 <name>setCurrentWorksheet</name>
 <type>procedure</type>
 <desc>
 Sets the Worksheet to actual worksheet.
 </desc>
 <version>v1.0.0</version>
 <params>
   <param>
   <param-def>p_idx pls_integer</param-def>
   <param-desc>Index of Worksheet</param-desc>
  </param>
 </params>
 <return>-</return>
*/
procedure setCurrentWorksheet(
  p_idx pls_integer
  );

/**
 <name>setCurrentRow</name>
 <type>procedure</type>
 <desc>
 Sets the Row to actual row.
 </desc>
 <version>v1.0.0</version>
 <params>
   <param>
   <param-def>p_idx pls_integer</param-def>
   <param-desc>Index of Row</param-desc>
  </param>
 </params>
 <return>-</return>
*/
procedure setCurrentRow(
  p_idx pls_integer
  );

/**
 <name>setCurrentCell</name>
 <type>procedure</type>
 <desc>
 Sets the Cell to actual cell.
 </desc>
 <version>v1.0.0</version>
 <params>
   <param>
   <param-def>p_idx pls_integer</param-def>
   <param-desc>Index of Cell</param-desc>
  </param>
 </params>
 <return>-</return>
*/
procedure setCurrentCell(
  p_idx pls_integer
  );

/**
 <name>getCurrentFormat</name>
 <type>function</type>
 <desc>
 Returns the index of current format.
 </desc>
 <version>v1.0.0</version>
 <params>-</params>
 <return>
 <return-def>pls_integer</return-def>
  <return-desc>Index of current format</return-desc>
 </return>
*/
function getCurrentFormat
  return pls_integer;

/**
 <name>getCurrentWorksheet</name>
 <type>function</type>
 <desc>
 Returns the index of current worksheet.
 </desc>
 <version>v1.0.0</version>
 <params>-</params>
 <return>
 <return-def>pls_integer</return-def>
  <return-desc>Index of current worksheet</return-desc>
 </return>
*/
function getCurrentWorksheet
  return pls_integer;

/**
 <name>getCurrentRow</name>
 <type>function</type>
 <desc>
 Returns the index of current row.
 </desc>
 <version>v1.0.0</version>
 <params>-</params>
 <return>
 <return-def>pls_integer</return-def>
  <return-desc>Index of current row</return-desc>
 </return>
*/
function getCurrentRow
  return pls_integer;

/**
 <name>getCurrentCell</name>
 <type>function</type>
 <desc>
 Returns the index of current cell.
 </desc>
 <version>v1.0.0</version>
 <params>-</params>
 <return>
 <return-def>pls_integer</return-def>
  <return-desc>Index of current cell</return-desc>
 </return>
*/
function getCurrentCell
  return pls_integer;

/**
 <name>isFormatMask</name>
 <type>function</type>
 <desc>
 Checks the index is a Format Mask.
 </desc>
 <version>v1.0.0</version>
 <params>
   <param>
     <param-def>p_format_mask_idx pls_integer</param-def>
     <param-desc>index of the format</param-desc>
   </param>
 </params>
 <return>
 <return-def>boolean</return-def>
  <param-ul>
    <param-li>true: index is a format</param-li>
    <param-li>false: index is not a format</param-li>
  </param-ul>
 </return>
*/
function isFormatMask(
  p_format_mask_idx pls_integer
  ) return boolean;

/**
 <name>isWorksheet</name>
 <type>function</type>
 <desc>
 Checks the index is a Worksheet.
 </desc>
 <version>v1.0.0</version>
 <params>
   <param>
     <param-def>p_ws_idx pls_integer</param-def>
     <param-desc>index of the worksheet</param-desc>
   </param>
 </params>
 <return>
 <return-def>isWorksheetboolean</return-def>
  <param-ul>
    <param-li>true: index is a worksheet</param-li>
    <param-li>false: index is not a worksheet</param-li>
  </param-ul>
 </return>
*/
function isWorksheet(
  p_ws_idx pls_integer
  ) return boolean;

/**
 <name>isRow</name>
 <type>function</type>
 <desc>
 Checks the index is a Row.
 </desc>
 <version>v1.0.0</version>
 <params>
   <param>
     <param-def>p_ws_idx pls_integer</param-def>
     <param-desc>index of the worksheet</param-desc>
   </param>
   <param>
     <param-def>p_Row_idx pls_integer</param-def>
     <param-desc>index of the row</param-desc>
   </param>
 </params>
 <return>
 <return-def>boolean</return-def>
  <param-ul>
    <param-li>true: index is a row</param-li>
    <param-li>false: index is not a row</param-li>
  </param-ul>
 </return>
*/
function isRow(
  p_ws_idx pls_integer,
  p_row_idx pls_integer
  ) return boolean;

/**
 <name>isCell</name>
 <type>function</type>
 <desc>
 Checks the index is a Cell.
 </desc>
 <version>v1.0.0</version>
 <params>
   <param>
     <param-def>p_ws_idx pls_integer</param-def>
     <param-desc>index of the worksheet</param-desc>
   </param>
   <param>
     <param-def>p_Row_idx pls_integer</param-def>
     <param-desc>index of the row</param-desc>
   </param>
   <param>
     <param-def>p_Cell_idx pls_integer</param-def>
     <param-desc>index of the cell</param-desc>
   </param>
 </params>
 <return>
 <return-def>boolean</return-def>
  <param-ul>
    <param-li>true: index is a cell</param-li>
    <param-li>false: index is not a cell</param-li>
  </param-ul>
 </return>
*/
function isCell(
  p_ws_idx   pls_integer,
  p_Row_idx  pls_integer,
  p_Cell_idx pls_integer
  ) return boolean;

/**
<name>setDocumentProperties</name>
 <type>procedure</type>
 <desc>
 Excel documents store metadata related to the document for example, the title, subject, keywords, etc.
 </desc>
 <version>v1.0.0</version>
 <params>
   <param>
     <param-def>p_title varchar2 default null</param-def>
     <param-desc>Title</param-desc>
   </param>
   <param>
     <param-def>p_subject varchar2 default null</param-def>
     <param-desc>Subject</param-desc>
   </param>
   <param>
     <param-def>p_keywords varchar2 default null</param-def>
     <param-desc>Keywords</param-desc>
   </param>
   <param>
     <param-def>p_description varchar2 default null</param-def>
     <param-desc>Description</param-desc>
   </param>
   <param>
     <param-def>p_category varchar2 default null</param-def>
     <param-desc>Category</param-desc>
   </param>
   <param>
     <param-def>p_contentStatus varchar2 default null</param-def>
     <param-desc>ContentStatus</param-desc>
   </param>
 </params>
 <return>-</return>
*/
procedure setDocumentProperties(
  p_title          varchar2 default null,
  p_subject        varchar2 default null,
  p_keywords       varchar2 default null,
  p_description    varchar2 default null,
  p_category       varchar2 default null,
  p_contentStatus  varchar2 default null
  );

/**
 <name>GetVersion</name>
 <type>function</type>
 <desc>
  Gets the version number of PL/XLS package.
 </desc>
 <version>v1.0.0</version>
 <params>-</params>
 <return>
  <return-def>varchar2</return-def>
  <return-desc>number of current version</return-desc>
 </return>
*/
function getVersion return varchar2;


/**
 <name>GetProduct</name>
 <type>function</type>
 <desc>
  Gets the name of product.
 </desc>
 <version>v1.0.0</version>
 <params>-</params>
 <return>
  <return-def>varchar2</return-def>
  <return-desc>name of current product</return-desc>
 </return>
*/
function getProduct
  return varchar2;

/**
<name>addWorksheet</name>
 <type>function</type>
 <desc>
 Defines a worksheet within the current workbook.
 Sets index to the current worksheet.
 </desc>
 <version>v1.0.0</version>
 <params>
   <param>
     <param-def>p_Name varchar2</param-def>
     <param-desc>
     This tag specifies the name of this worksheet.
     This value must be unique within the list of sheet names in this workbook.
     </param-desc>
   </param>
 </params>
 <return>
  <return-def>pls_integer</return-def>
  <return-desc>index of the new worksheet</return-desc>
 </return>
*/
function addWorksheet(
    p_Name varchar2
    ) return pls_integer;

/**
<name>addWorksheet</name>
 <type>procedure</type>
 <desc>
 Defines a worksheet within the current workbook.
 Sets index to the current worksheet.
 </desc>
 <version>v1.0.0</version>
 <params>
   <param>
     <param-def>p_Name varchar2</param-def>
     <param-desc>
     This tag specifies the name of this worksheet.
     This value must be unique within the list of sheet names in this workbook.
     </param-desc>
   </param>
 </params>
 <return>-</return>
*/
procedure addWorksheet(
    p_Name varchar2
    );

/**
<name>getWorksheetName</name>
 <type>function</type>
 <desc>Get the selected worksheet name.
 </desc>
 <version>v1.3.0</version>
 <params>
   <param>
     <param-def>p_ws pls_integer default null</param-def>
     <param-desc>Index of the worksheet. null means the current Worksheet.</param-desc>
   </param>
 </params>
 <return>
  <return-def>varchar2</return-def>
  <return-desc>Name of the worksheet.</return-desc>
 </return>
*/
function getWorksheetName(
  p_ws pls_integer
  ) return varchar2;

/**
<name>getWorksheetId</name>
 <type>function</type>
 <desc>Get the selected worksheet id.
 </desc>
 <version>v4.9.0</version>
 <params>
   <param>
     <param-def>p_Name varchar2</param-def>
     <param-desc>Name of the worksheet.</param-desc>
   </param>
 </params>
 <return>
  <return-def>pls_integer</return-def>
  <return-desc>Id of the worksheet.</return-desc>
 </return>
*/
function getWorksheetId(
  p_Name varchar2
  ) return pls_integer;


/**
<name>getLastWorksheetNum</name>
 <type>function</type>
 <desc>Get the last worksheet index for looping in sheets.
 </desc>
 <version>v4.9.0</version>
 <params>-</params>
 <return>
  <return-def>pls_integer</return-def>
  <return-desc>the index of the last worksheet.</return-desc>
 </return>
*/
function getLastWorksheetNum
  return pls_integer;

--v4.9.1
procedure setWorksheetState(
  p_ws pls_integer default null,
  p_state varchar2 default null
  );

--v4.9.1
function getWorksheetState(
  p_ws pls_integer default null
  ) return varchar2;

--v4.9.1
procedure setWorksheetHidden(
  p_ws pls_integer default null
  );

/**
<name>addRow</name>
 <type>function</type>
 <desc>
 Defines the formatting and data for one or more adjacent rows.
 If this element contains cell data, it must not contain the Span attribute.
 Each Row element may contain one or more Cell elements defining the table's data.
 Sets index to the current row.
 </desc>
 <version>v1.0.0</version>
 <params>
   <param>
     <param-def>p_ws pls_integer default null</param-def>
     <param-desc>Index of the worksheet. null means the current Worksheet.</param-desc>
   </param>
 </params>
 <return>
  <return-def>pls_integer</return-def>
  <return-desc>index of the new row</return-desc>
 </return>
*/
function addRow(
    p_ws pls_integer default null
    ) return pls_integer;

/**
<name>addRow</name>
 <type>procedure</type>
 <desc>
 Adds row to the worksheet and sets index to the added row.
 </desc>
 <version>v1.0.0</version>
 <params>
   <param>
     <param-def>p_ws pls_integer default null</param-def>
     <param-desc>Index of the worksheet. null means the current Worksheet.</param-desc>
   </param>
 </params>
 <return>-</return>
*/
procedure addRow(
    p_ws pls_integer default null
    );

/**
<name>addRows</name>
 <type>function</type>
 <desc>
 Adds rows to the worksheet and sets index to the last added row.
 </desc>
 <version>v4.9.0</version>
 <params>
   <param>
     <param-def>p_ws pls_integer default null</param-def>
     <param-desc>Index of the worksheet. null means the current Worksheet.</param-desc>
   </param>
   <param>
     <param-def>p_num pls_integer default 1</param-def>
     <param-desc>Number of new rows</param-desc>
   </param>
 </params>
 <return>
  <return-def>pls_integer</return-def>
  <return-desc>index of the last new row</return-desc>
 </return>
*/
function addRows(
  p_ws pls_integer default null,
  p_num pls_integer default 1
  ) return pls_integer;

/**
<name>addRow</name>
 <type>procedure</type>
 <desc>
 Adds rows to the worksheet and sets index to the added row.
 </desc>
 <version>v4.9.0</version>
 <params>
   <param>
     <param-def>p_ws pls_integer default null</param-def>
     <param-desc>Index of the worksheet. null means the current Worksheet.</param-desc>
   </param>
   <param>
     <param-def>p_num pls_integer default 1</param-def>
     <param-desc>Number of new rows</param-desc>
   </param>
 </params>
 <return>-</return>
*/
procedure addRows(
  p_ws pls_integer default null,
  p_num pls_integer default 1
  );

/**
<name>getLastRowNum</name>
 <type>function</type>
 <desc>Get last row index of the the worksheet.</desc>
 <version>v4.9.0</version>
 <params>
   <param>
     <param-def>p_ws pls_integer default null</param-def>
     <param-desc>Index of the worksheet. null means the current Worksheet.</param-desc>
   </param>
 </params>
 <return>
  <return-def>pls_integer</return-def>
  <return-desc>index of the last row</return-desc>
 </return>
*/
function getLastRowNum(
  p_ws pls_integer default null
  ) return pls_integer;

/**
<name>Row_Height</name>
 <type>procedure</type>
 <desc>
 Sets property of an existing row.
 </desc>
 <version>v1.0.0</version>
 <params>
   <param>
     <param-def>p_ws pls_integer default null</param-def>
     <param-desc>Index of the worksheet. null means the current Worksheet.</param-desc>
   </param>
   <param>
     <param-def>p_row pls_integer default null</param-def>
     <param-desc>Index of the row. null means the current Row.</param-desc>
   </param>
   <param>
     <param-def>p_Height number</param-def>
     <param-desc>Specifies the height of a row in points. This value must be greater than or equal to 0.</param-desc>
   </param>
 </params>
 <return>-</return>
*/
procedure Row_Height(
    p_ws     pls_integer default null,
    p_row    pls_integer default null,
    p_Height number
    );

/**
<name>Row_Format</name>
 <type>procedure</type>
 <desc>
 Sets property of an existing row.
 </desc>
 <version>v1.0.0</version>
 <params>
   <param>
     <param-def>p_ws pls_integer default null</param-def>
     <param-desc>Index of the worksheet. null means the current Worksheet.</param-desc>
   </param>
   <param>
     <param-def>p_row pls_integer default null</param-def>
     <param-desc>Index of the row. null means the current Row.</param-desc>
   </param>
   <param>
     <param-def>p_format_id pls_integer</param-def>
     <param-desc>Specifies a reference to a previously defined format.</param-desc>
   </param>
 </params>
 <return>-</return>
*/
procedure Row_Format(
    p_ws        pls_integer default null,
    p_row       pls_integer default null,
    p_format_id pls_integer
    );

/**
<name>Row_Freeze</name>
 <type>procedure</type>
 <desc>
 Sets property of an existing row.
 </desc>
 <version>v1.0.0</version>
 <params>
   <param>
     <param-def>p_ws pls_integer default null</param-def>
     <param-desc>Index of the worksheet. null means the current Worksheet.</param-desc>
   </param>
   <param>
     <param-def>p_number_rows pls_integer default null</param-def>
     <param-desc>Numbers of rows what you freeze. The default value is 1.</param-desc>
   </param>
 </params>
 <return>-</return>
*/
procedure Row_Freeze(
    p_ws          pls_integer default null,
    p_number_rows pls_integer default 1
    );

/**
<name>Row_Hide</name>
 <type>procedure</type>
 <desc>
 Flag indicating if the affected row is hidden on this worksheet.
 </desc>
 <version>v4.3.0</version>
 <params>
   <param>
     <param-def>p_ws pls_integer default null</param-def>
     <param-desc>Index of the worksheet. null means the current Worksheet.</param-desc>
   </param>
   <param>
     <param-def>p_number_rows pls_integer default null</param-def>
     <param-desc>Numbers of rows what you freeze. The default value is 1.</param-desc>
   </param>
   <param>
     <param-def>p_hide boolean default true</param-def>
     <param-desc>Affected row is hidden or not on this worksheet.</param-desc>
   </param>
 </params>
 <return>-</return>
*/
procedure Row_Hide(
  p_ws   pls_integer default null,
  p_row  pls_integer default null,
  p_hide boolean default true
  );

/**
<name>Row_Collapse</name>
 <type>procedure</type>
 <desc>
 Flag indicating if the outlining of the affected row is in the collapsed state.
 </desc>
 <version>v4.3.0</version>
 <params>
   <param>
     <param-def>p_ws pls_integer default null</param-def>
     <param-desc>Index of the worksheet. null means the current Worksheet.</param-desc>
   </param>
   <param>
     <param-def>p_number_rows pls_integer default null</param-def>
     <param-desc>Numbers of rows what you freeze. The default value is 1.</param-desc>
   </param>
   <param>
     <param-def>p_collapse boolean default true</param-def>
     <param-desc>Affected row is collapsed or not on this worksheet.</param-desc>
   </param>
 </params>
 <return>-</return>
*/
procedure Row_Collapse(
  p_ws   pls_integer default null,
  p_row  pls_integer default null,
  p_collapse boolean default true
  );

/**
<name>Row_outlineLevel</name>
 <type>procedure</type>
 <desc>
 Outline level of affected row. Range is 0 to 7.
 </desc>
 <version>v4.3.0</version>
 <params>
   <param>
     <param-def>p_ws pls_integer default null</param-def>
     <param-desc>Index of the worksheet. null means the current Worksheet.</param-desc>
   </param>
   <param>
     <param-def>p_number_rows pls_integer default null</param-def>
     <param-desc>Numbers of rows what you freeze. The default value is 1.</param-desc>
   </param>
   <param>
     <param-def>p_outlineLevel pls_integer default null</param-def>
     <param-desc>Affected row's outline level.</param-desc>
   </param>
 </params>
 <return>-</return>
*/
procedure Row_outlineLevel(
  p_ws   pls_integer default null,
  p_row  pls_integer default null,
  p_outlineLevel pls_integer default null
  );

/**
<name>Column_Width</name>
 <type>procedure</type>
 <desc>
 Sets property of an column.
 </desc>
 <version>v1.0.0</version>
 <params>
   <param>
     <param-def>p_ws pls_integer default null</param-def>
     <param-desc>Index of the worksheet. null means the current Worksheet.</param-desc>
   </param>
   <param>
     <param-def>p_column pls_integer default null</param-def>
     <param-desc>Index of the column. null means the current Columns.</param-desc>
   </param>
   <param>
     <param-def>p_Width number</param-def>
     <param-desc>Specifies the width of a column in characters. This value must be greater than or equal to 0.</param-desc>
   </param>
 </params>
 <return>-</return>
*/
procedure Column_Width(
    p_ws pls_integer default null,
    p_column varchar2,
    p_width    number
    );

/**
<name>Column_Freeze</name>
 <type>procedure</type>
 <desc>
 Sets property of an existing column.
 </desc>
 <version>v1.0.0</version>
 <params>
   <param>
     <param-def>p_ws pls_integer default null</param-def>
     <param-desc>Index of the worksheet. null means the current Worksheet.</param-desc>
   </param>
   <param>
     <param-def>p_number_columns pls_integer default null</param-def>
     <param-desc>Numbers of columns what you freeze. The default value is 1.</param-desc>
   </param>
 </params>
 <return>-</return>
*/
procedure Column_Freeze(
    p_ws          pls_integer default null,
    p_number_columns pls_integer default 1
    );

/**
<name>Column_Hide</name>
 <type>procedure</type>
 <desc>
 Flag indicating if the affected column is hidden on this worksheet.
 </desc>
 <version>v4.3.0</version>
 <params>
   <param>
     <param-def>p_ws pls_integer default null</param-def>
     <param-desc>Index of the worksheet. null means the current Worksheet.</param-desc>
   </param>
   <param>
     <param-def>p_column pls_integer default null</param-def>
     <param-desc>Index of the column. null means the current Columns.</param-desc>
   </param>
   <param>
     <param-def>p_hide boolean default true</param-def>
     <param-desc>Affected column is hidden or not on this worksheet.</param-desc>
   </param>
 </params>
 <return>-</return>
*/
procedure Column_Hide(
  p_ws pls_integer default null,
  p_column varchar2,
  p_hide boolean default true
  );

/**
<name>Column_collapse</name>
 <type>procedure</type>
 <desc>
 Flag indicating if the outlining of the affected column is in the collapsed state.
 </desc>
 <version>v4.3.0</version>
 <params>
   <param>
     <param-def>p_ws pls_integer default null</param-def>
     <param-desc>Index of the worksheet. null means the current Worksheet.</param-desc>
   </param>
   <param>
     <param-def>p_column pls_integer default null</param-def>
     <param-desc>Index of the column. null means the current Columns.</param-desc>
   </param>
   <param>
     <param-def>p_collapse boolean default true</param-def>
     <param-desc>Affected column is collapsed or not on this worksheet.</param-desc>
   </param>
 </params>
 <return>-</return>
*/
procedure Column_collapse(
  p_ws pls_integer default null,
  p_column varchar2,
  p_collapse boolean default true
  );

/**
<name>Column_outlineLevel</name>
 <type>procedure</type>
 <desc>
 Outline level of affected column(s). Range is 0 to 7.
 </desc>
 <version>v4.3.0</version>
 <params>
   <param>
     <param-def>p_ws pls_integer default null</param-def>
     <param-desc>Index of the worksheet. null means the current Worksheet.</param-desc>
   </param>
   <param>
     <param-def>p_column pls_integer default null</param-def>
     <param-desc>Index of the column. null means the current Columns.</param-desc>
   </param>
   <param>
     <param-def>p_outlineLevel pls_integer default null</param-def>
     <param-desc>Affected column's outline level.</param-desc>
   </param>
 </params>
 <return>-</return>
*/
procedure Column_outlineLevel(
  p_ws pls_integer default null,
  p_column varchar2,
  p_outlineLevel pls_integer default null
  );

/**
<name>RowColumn_Freeze</name>
 <type>procedure</type>
 <desc>
 Sets property of an existing column and row.
 </desc>
 <version>v5.13</version>
 <params>
   <param>
     <param-def>p_ws pls_integer default null</param-def>
     <param-desc>Index of the worksheet. null means the current Worksheet.</param-desc>
   </param>
   <param>
     <param-def>p_number_rows pls_integer default 1</param-def>
     <param-desc>Numbers of rows what you freeze. The default value is 1.</param-desc>
   </param>
   <param>
     <param-def>p_number_columns pls_integer default 1</param-def>
     <param-desc>Numbers of columns what you freeze. The default value is 1.</param-desc>
   </param>
 </params>
 <return>-</return>
*/
procedure RowColumn_Freeze(
  p_ws          pls_integer default null,
  p_number_rows pls_integer default 1,
  p_number_columns pls_integer default 1
  );

/**
<name>addValidation_List</name>
 <type>function</type>
 <desc>Set cell validation list</desc>
 <version>v1.3.0</version>
 <params>
   <param>
     <param-def>p_ws pls_integer default null</param-def>
     <param-desc>Index of the worksheet. null means the current Worksheet.</param-desc>
   </param>
    <param>
     <param-def>p_row pls_integer default null</param-def>
     <param-desc>Index of the row. null means the current Row.</param-desc>
   </param>
    <param>
     <param-def>p_cell varchar2</param-def>
     <param-desc>Specifies a reference to a previously defined cell.</param-desc>
   </param>
   <param>
     <param-def>p_values varchar2</param-def>
     <param-desc>Sets the values of list, separator comma or semicolon</param-desc>
   </param>
   <param>
     <param-def>p_allow_blank boolean default true</param-def>
     <param-desc>The data validation treats empty or blank entries as valid.</param-desc>
   </param>
 </params>
 <return>
 <return-def>pls_integer</return-def>
  <return-desc>index of the new validation</return-desc>
 </return>
*/
function addValidation_List(
  p_ws  pls_integer default null,
  p_row pls_integer default null,
  p_cell varchar2,
  p_values varchar2,
  p_allow_blank boolean default true
  ) return pls_integer;

/**
<name>Validation_addApplied</name>
 <type>procedure</type>
 <desc>
 Set cell validation list
 </desc>
 <version>v1.3.0</version>
 <params>
   <param>
     <param-def>p_ws pls_integer default null</param-def>
     <param-desc>Index of the worksheet. null means the current Worksheet.</param-desc>
   </param>
    <param>
     <param-def>p_row pls_integer default null</param-def>
     <param-desc>Index of the row. null means the current Row.</param-desc>
   </param>
    <param>
     <param-def>p_cell varchar2</param-def>
     <param-desc>Specifies a reference to a previously defined cell.</param-desc>
   </param>
   <param>
     <param-def>p_values varchar2</param-def>
     <param-desc>Sets the values of list, separator comma or semicolon</param-desc>
   </param>
   <param>
     <param-def>p_allow_blank boolean default true</param-def>
     <param-desc>The data validation treats empty or blank entries as valid.</param-desc>
   </param>
 </params>
 <return>-</return>
*/
procedure Validation_addApplied(
  p_ws  pls_integer default null,
  p_row pls_integer default null,
  p_cell varchar2,
  p_validation pls_integer
  );

/**
<name>setDefaultFont</name>
 <type>procedure</type>
 <desc>
 Sets the default font properties
 </desc>
 <version>v1.0.0</version>
 <params>
   <param>
     <param-def>p_FontName varchar2 default c_default_font_name</param-def>
     <param-desc>Name of the font-family. Default value is Calibri</param-desc>
   </param>
   <param>
     <param-def>p_Size pls_integer default c_default_font_size</param-def>
     <param-desc>Size of the font. The default value is 11.</param-desc>
   </param>
 </params>
 <return>-</return>
*/
procedure setDefaultFont(
  p_FontName varchar2 default c_default_font_name,
  p_Size pls_integer default c_default_font_size
  );

/**
<name>addFill</name>
 <type>function</type>
 <desc>
 Defines the formatting and data for one or more adjacent rows.
 If this element contains cell data, it must not contain the Span attribute.
 Each Row element may contain one or more Cell elements defining the table's data.
 Sets index to the current row.
 </desc>
 <version>v1.0.0</version>
 <params>
   <param>
     <param-def>p_pattern_type varchar2 default null</param-def>
     <param-desc>Sets the pattern type.</param-desc>
   </param>
   <param>
     <param-def>p_color varchar2 default null</param-def>
     <param-desc>The color of the pattern.</param-desc>
   </param>
 </params>
 <return>
  <return-def>pls_integer</return-def>
  <return-desc>index of the new filling pattern</return-desc>
 </return>
*/
function addFill(
  p_pattern_type varchar2 default null,
  p_color        varchar2 default null
  ) return pls_integer;

/**
<name>addBorder</name>
 <type>function</type>
 <desc>
 Defines the cell borders.
 </desc>
 <version>v1.0.0</version>
 <params>
   <param>
     <param-def>p_border_top_style varchar2 default null</param-def>
     <param-desc>Style of the top side border</param-desc>
   </param>
   <param>
     <param-def>p_border_top_color varchar2 default null</param-def>
     <param-desc>Color of the top side border</param-desc>
   </param>
   <param>
     <param-def>p_border_bottom_style varchar2 default null</param-def>
     <param-desc>Style of the bottom side border</param-desc>
   </param>
   <param>
     <param-def>p_border_bottom_color varchar2 default null</param-def>
     <param-desc>Color of the bottom side border</param-desc>
   </param>
   <param>
     <param-def>p_border_left_style varchar2 default null</param-def>
     <param-desc>Style of the left side border</param-desc>
   </param>
   <param>
     <param-def>p_border_top_left varchar2 default null</param-def>
     <param-desc>Color of the left side border</param-desc>
   </param>
   <param>
     <param-def>p_border_right_style varchar2 default null</param-def>
     <param-desc>Style of the right side border</param-desc>
   </param>
   <param>
     <param-def>p_border_right_color varchar2 default null</param-def>
     <param-desc>Color of the right side border</param-desc>
   </param>
 </params>
 <return>
   <return-def>pls_integer</return-def>
   <return-desc>Index of the new border</return-desc>
 </return>
*/
function addBorder(
  p_border_top_style    varchar2 default null,
  p_border_top_color    varchar2 default null,
  p_border_bottom_style varchar2 default null,
  p_border_bottom_color varchar2 default null,
  p_border_left_style   varchar2 default null,
  p_border_left_color   varchar2 default null,
  p_border_right_style  varchar2 default null,
  p_border_right_color  varchar2 default null
  ) return pls_integer;

/**
<name>addFormatMask</name>
 <type>function</type>
 <desc>
 Defines a format mask.
 </desc>
 <version>v1.0.0</version>
 <params>
   <param>
     <param-def>p_format_mask varchar2</param-def>
     <param-desc>Implementation of the format mask</param-desc>
   </param>
 </params>
 <return>
   <return-def>pls_integer</return-def>
   <return-desc>Index of the new mask.</return-desc>
 </return>
*/
function addFormatMask(
  p_format_mask varchar2
  ) return pls_integer;

/**
<name>addFont</name>
 <type>function</type>
 <desc>
 Defines a font.
 </desc>
 <version>v5.20</version>
 <params>
   <param>
     <param-def>p_font_name varchar2 default null</param-def>
     <param-desc>Font-family name</param-desc>
   </param>
   <param>
     <param-def>p_font_size pls_integer default null</param-def>
     <param-desc>Font size.</param-desc>
   </param>
   <param>
     <param-def>p_italic boolean default false</param-def>
     <param-desc>Set to italic</param-desc>
   </param>
   <param>
     <param-def>p_underline boolean default false</param-def>
     <param-desc>Set to underline</param-desc>
   </param>
   <param>
     <param-def>p_bold boolean default false</param-def>
     <param-desc>Set to bold</param-desc>
   </param>
   <param>
     <param-def>p_color varchar2 default null</param-def>
     <param-desc>Sets the font color.</param-desc>
   </param>
   <param>
     <param-def>p_strike    boolean default false</param-def>
     <param-desc>Set to strike</param-desc>
   </param>
 </params>
 <return>
   <return-def>pls_integer</return-def>
   <return-desc>Index of the new font.</return-desc>
 </return>
*/
function addFont(
  p_font_name varchar2 default null,
  p_font_size pls_integer default null,
  p_italic    boolean default false,
  p_underline boolean default false,
  p_bold      boolean default false,
  p_color     varchar2 default null,
  p_strike    boolean default false
  ) return pls_integer;

/**
<name>addFormat</name>
 <type>function</type>
 <desc>
 Defines a format style.
 </desc>
 <version>v1.0.0</version>
 <params>
   <param>
     <param-def>p_numFmtId pls_integer default null</param-def>
     <param-desc>Index of numeric format, predefined IDs:</param-desc>
     <param-ul>
        <param-li>0 General</param-li>
        <param-li>1 0</param-li>
        <param-li>2 0.00</param-li>
        <param-li>3 #,##0</param-li>
        <param-li>4 #,##0.00</param-li>
        <param-li>9 0%</param-li>
        <param-li>10 0.00%</param-li>
        <param-li>11 0.00E+00</param-li>
        <param-li>12 # ?/?</param-li>
        <param-li>13 # ??/??</param-li>
        <param-li>14 mm-dd-yy</param-li>
        <param-li>15 d-mmm-yy</param-li>
        <param-li>16 d-mmm</param-li>
        <param-li>17 mmm-yy</param-li>
        <param-li>18 h:mm AM/PM</param-li>
        <param-li>19 h:mm:ss AM/PM</param-li>
        <param-li>20 h:mm</param-li>
        <param-li>21 h:mm:ss</param-li>
        <param-li>22 m/d/yy h:mm</param-li>
        <param-li>37 #,##0 ;(#,##0)</param-li>
        <param-li>38 #,##0 ;[Red](#,##0)</param-li>
        <param-li>39 #,##0.00;(#,##0.00)</param-li>
        <param-li>40 #,##0.00;[Red](#,##0.00)</param-li>
        <param-li>45 mm:ss</param-li>
        <param-li>46 [h]:mm:ss</param-li>
        <param-li>47 mmss.0</param-li>
        <param-li>48 ##0.0E+0</param-li>
        <param-li>49 @</param-li>
   </param-ul>
   </param>
   <param>
     <param-def>p_fontId pls_integer default null</param-def>
     <param-desc>Index of font</param-desc>
   </param>
   <param>
     <param-def>p_fillId pls_integer default null</param-def>
     <param-desc>Index of filling style</param-desc>
   </param>
   <param>
     <param-def>p_borderId pls_integer default null</param-def>
     <param-desc>Index of border style</param-desc>
   </param>
   <param>
     <param-def>p_horizontal_align varchar2 default null</param-def>
     <param-desc>Horizontal alignment</param-desc>
   </param>
   <param>
     <param-def>p_vertical_align   varchar2 default null</param-def>
     <param-desc>Vertical alignment</param-desc>
   </param>
   <param>
     <param-def>p_textRotation pls_integer default null</param-def>
     <param-desc>Text rotation in cells. Expressed in degrees. Values range from 0 to 180.</param-desc>
   </param>
   <param>
     <param-def>p_wrap_text boolean default false</param-def>
     <param-desc>Wrap text</param-desc>
   </param>
 </params>
 <return>
   <return-def>pls_integer</return-def>
   <return-desc>Index of the new format.</return-desc>
 </return>
*/
function addFormat(
  p_numFmtId         pls_integer default null,
  p_fontId           pls_integer default null,
  p_fillId           pls_integer default null,
  p_borderId         pls_integer default null,
  p_horizontal_align varchar2 default null,
  p_vertical_align   varchar2 default null,
  p_textRotation     pls_integer default null,
  p_wrap_text        boolean default false
  ) return pls_integer;

/**
<name>addFormat1</name>
 <type>function</type>
 <desc>
 Defines a format style.
 </desc>
 <version>v1.0.0</version>
 <params>
   <param>
     <param-def>p_font_name varchar2 default null</param-def>
     <param-desc>Font name</param-desc>
   </param>
   <param>
     <param-def>p_font_size pls_integer default null</param-def>
     <param-desc>Font size.</param-desc>
   </param>
   <param>
     <param-def>p_italic boolean default false</param-def>
     <param-desc>Set to italic</param-desc>
   </param>
   <param>
     <param-def>p_underline boolean default false</param-def>
     <param-desc>Set to underline</param-desc>
   </param>
   <param>
     <param-def>p_bold boolean default false</param-def>
     <param-desc>Set to bold</param-desc>
   </param>
   <param>
     <param-def>p_color varchar2 default null</param-def>
     <param-desc>Sets the font color.</param-desc>
   </param>
   <param>
     <param-def>p_pattern_type varchar2 default null</param-def>
     <param-desc>Sets the pattern type.</param-desc>
   </param>
   <param>
     <param-def>p_pattern_color varchar2 default null</param-def>
     <param-desc>Sets the pattern color.</param-desc>
   </param>
   <param>
     <param-def>p_border_top_style varchar2 default null</param-def>
     <param-desc>Style of the top side border</param-desc>
   </param>
   <param>
     <param-def>p_border_top_color varchar2 default null</param-def>
     <param-desc>Color of the top side border</param-desc>
   </param>
   <param>
     <param-def>p_border_bottom_style varchar2 default null</param-def>
     <param-desc>Style of the bottom side border</param-desc>
   </param>
   <param>
     <param-def>p_border_bottom_color varchar2 default null</param-def>
     <param-desc>Color of the bottom side border</param-desc>
   </param>
   <param>
     <param-def>p_border_left_style varchar2 default null</param-def>
     <param-desc>Style of the left side border</param-desc>
   </param>
   <param>
     <param-def>p_border_top_left varchar2 default null</param-def>
     <param-desc>Color of the left side border</param-desc>
   </param>
   <param>
     <param-def>p_border_right_style varchar2 default null</param-def>
     <param-desc>Style of the right side border</param-desc>
   </param>
   <param>
     <param-def>p_border_right_color varchar2 default null</param-def>
     <param-desc>Color of the right side border</param-desc>
   </param>
   <param>
     <param-def>p_horizontal_align varchar2 default null</param-def>
     <param-desc>Horizontal alignment</param-desc>
   </param>
   <param>
     <param-def>p_vertical_align   varchar2 default null</param-def>
     <param-desc>Vertical alignment</param-desc>
   </param>
   <param>
     <param-def>p_textRotation pls_integer default null</param-def>
     <param-desc>Text rotation in cells. Expressed in degrees. Values range from 0 to 180.</param-desc>
   </param>
   <param>
     <param-def>p_wrap_text boolean default false</param-def>
     <param-desc>Wrap text</param-desc>
   </param>
   <param>
     <param-def>p_format_mask varchar2 default null</param-def>
     <param-desc>Format mask</param-desc>
   </param>
 </params>
 <return>
   <return-def>pls_integer</return-def>
   <return-desc>Index of the new format.</return-desc>
 </return>
*/
function addFormat1(
   p_font_name           varchar2 default null,
   p_font_size           pls_integer default null,
   p_font_italic         boolean default false,
   p_font_underline      boolean default false,
   p_font_bold           boolean default false,
   p_font_color          varchar2 default null,
   p_pattern_type        varchar2 default null,
   p_pattern_color       varchar2 default null,
   p_border_top_style    varchar2 default null,
   p_border_top_color    varchar2 default null,
   p_border_bottom_style varchar2 default null,
   p_border_bottom_color varchar2 default null,
   p_border_left_style   varchar2 default null,
   p_border_left_color   varchar2 default null,
   p_border_right_style  varchar2 default null,
   p_border_right_color  varchar2 default null,
   p_horizontal_align    varchar2 default null,
   p_vertical_align      varchar2 default null,
   p_textRotation        pls_integer default null,
   p_wrap_text           boolean default false,
   p_format_mask         varchar2 default null
   ) return pls_integer;

/**
<name>addCell_Number</name>
 <type>procedure</type>
 <desc>
 Defines a single cell within the contained row.
 Sets index to the current cell.
 </desc>
 <version>v1.0.0</version>
 <params>
   <param>
     <param-def>p_ws pls_integer default null</param-def>
     <param-desc>Index of the worksheet. null means the current Worksheet.</param-desc>
   </param>
    <param>
     <param-def>p_row pls_integer default null</param-def>
     <param-desc>Index of the row. null means the current Row.</param-desc>
   </param>
   <param>
     <param-def>p_cell varchar2</param-def>
     <param-desc>Specifies a reference to a previously defined cell.</param-desc>
   </param>
   <param>
     <param-def>p_data number</param-def>
     <param-desc>Specifies the value of this cell.</param-desc>
   </param>
   <param>
     <param-def>p_format_id pls_integer default null</param-def>
     <param-desc>Specifies a reference to a previously defined format.</param-desc>
   </param>
 </params>
 <return>-</return>
*/
  procedure addCell_Number(
    p_ws        pls_integer default null,
    p_row       pls_integer default null,
    p_cell      varchar2,
    p_data      number,
    p_format_id pls_integer default null
    );

/**
<name>addCell_String</name>
 <type>procedure</type>
 <desc>
 Defines a single cell within the contained row.
 Sets index to the current cell.
 </desc>
 <version>v1.0.0</version>
 <params>
   <param>
     <param-def>p_ws pls_integer default null</param-def>
     <param-desc>Index of the worksheet. null means the current Worksheet.</param-desc>
   </param>
    <param>
     <param-def>p_row pls_integer default null</param-def>
     <param-desc>Index of the row. null means the current Row.</param-desc>
   </param>
   <param>
     <param-def>p_cell varchar2</param-def>
     <param-desc>Specifies a reference to a previously defined cell.</param-desc>
   </param>
   <param>
     <param-def>p_data varchar2</param-def>
     <param-desc>Specifies the value of this cell.</param-desc>
   </param>
   <param>
     <param-def>p_format_id pls_integer default null</param-def>
     <param-desc>Specifies a reference to a previously defined format.</param-desc>
   </param>
 </params>
 <return>-</return>
*/
procedure addCell_String(
  p_ws        pls_integer default null,
  p_row       pls_integer default null,
  p_cell      varchar2,
  p_data      varchar2,
  p_format_id pls_integer default null
  );

--v4.9.7
function addStringItem
  return pls_integer;

--v4.9.7
procedure addRichText(
  p_StringItem pls_integer,
  p_rPr ploffx_sharedStrings.CT_RPrElt default null,
  p_value varchar2 --text
  );

--v4.9.7
procedure addRichText1(
  p_StringItem pls_integer,
  --
  p_font_name      varchar2 default null,
  p_font_size      pls_integer default null,
  p_font_italic    boolean default false,
  p_font_underline boolean default false,
  p_font_bold      boolean default false,
  p_font_color     varchar2 default null,
  --
  p_value varchar2 --text
  );

--v4.9.7
procedure addCell_StringItem(
  p_ws pls_integer default null,
  p_row pls_integer default null,
  p_cell varchar2,
  p_StringItem pls_integer,
  p_format_id pls_integer default null
  );

/**
<name>addCell_Date</name>
 <type>procedure</type>
 <desc>
 Defines a single cell within the contained row.
 Sets index to the current cell.
 </desc>
 <version>v1.0.0</version>
 <params>
   <param>
     <param-def>p_ws pls_integer default null</param-def>
     <param-desc>Index of the worksheet. null means the current Worksheet.</param-desc>
   </param>
    <param>
     <param-def>p_row pls_integer default null</param-def>
     <param-desc>Index of the row. null means the current Row.</param-desc>
   </param>
    <param>
     <param-def>p_cell varchar2</param-def>
     <param-desc>Specifies a reference to a previously defined cell.</param-desc>
   </param>
   <param>
     <param-def>p_data date</param-def>
     <param-desc>Specifies the value of this cell.</param-desc>
   </param>
   <param>
     <param-def>p_format varchar2 default c_default_date_format</param-def>
     <param-desc>Specifies the date format strings.</param-desc>
   </param>
   <param>
     <param-def>p_format_mask varchar2 default c_default_date_format</param-def>
     <param-desc>Specifies a custom date format.</param-desc>
   </param>
   <param>
     <param-def>p_format_id pls_integer default null</param-def>
     <param-desc>Specifies a reference to a previously defined format.</param-desc>
   </param>
 </params>
 <return>-</return>
*/
  procedure addCell_Date(
    p_ws          pls_integer default null,
    p_row         pls_integer default null,
    p_cell        varchar2,
    p_data        date,
    p_format_mask varchar2 default c_default_date_format,
    p_format_id   pls_integer default null
    );

/**
<name>Cell_setFont</name>
 <type>procedure</type>
 <desc>
 Set cell font properties
 </desc>
 <version>v1.0.0</version>
 <params>
   <param>
     <param-def>p_ws pls_integer default null</param-def>
     <param-desc>Index of the worksheet. null means the current Worksheet.</param-desc>
   </param>
    <param>
     <param-def>p_row pls_integer default null</param-def>
     <param-desc>Index of the row. null means the current Row.</param-desc>
   </param>
    <param>
     <param-def>p_cell varchar2</param-def>
     <param-desc>Specifies a reference to a previously defined cell.</param-desc>
   </param>
   <param>
     <param-def>p_FontName varchar2</param-def>
     <param-desc>Sets the font name.</param-desc>
   </param>
   <param>
     <param-def>p_Size pls_integer default 11</param-def>
     <param-desc>Sets the font size. Default value is 11.</param-desc>
   </param>
 </params>
 <return>-</return>
*/
  procedure Cell_setFont(
    p_ws       pls_integer default null,
    p_row      pls_integer default null,
    p_cell     varchar2,
    p_FontName varchar2,
    p_Size     pls_integer default 11 --??number
    );

/**
<name>Cell_setBold</name>
 <type>procedure</type>
 <desc>
 Set cell style to bold.
 </desc>
 <version>v1.0.0</version>
 <params>
   <param>
     <param-def>p_ws pls_integer default null</param-def>
     <param-desc>Index of the worksheet. null means the current Worksheet.</param-desc>
   </param>
    <param>
     <param-def>p_row pls_integer default null</param-def>
     <param-desc>Index of the row. null means the current Row.</param-desc>
   </param>
    <param>
     <param-def>p_cell varchar2</param-def>
     <param-desc>Specifies a reference to a previously defined cell.</param-desc>
   </param>
 </params>
 <return>-</return>
*/
  procedure Cell_setBold(
    p_ws   pls_integer default null,
    p_row  pls_integer default null,
    p_cell varchar2
    );

/**
<name>Cell_setItalic</name>
 <type>procedure</type>
 <desc>
 Set cell style to italic.
 </desc>
 <version>v1.0.0</version>
 <params>
   <param>
     <param-def>p_ws pls_integer default null</param-def>
     <param-desc>Index of the worksheet. null means the current Worksheet.</param-desc>
   </param>
    <param>
     <param-def>p_row pls_integer default null</param-def>
     <param-desc>Index of the row. null means the current Row.</param-desc>
   </param>
    <param>
     <param-def>p_cell varchar2</param-def>
     <param-desc>Specifies a reference to a previously defined cell.</param-desc>
   </param>
 </params>
 <return>-</return>
*/
  procedure Cell_setItalic(
    p_ws   pls_integer default null,
    p_row  pls_integer default null,
    p_cell varchar2
    );

/**
<name>Cell_setUnderline</name>
 <type>procedure</type>
 <desc>
 Set cell style to underline.
 </desc>
 <version>v4.11</version>
 <params>
   <param>
     <param-def>p_ws pls_integer default null</param-def>
     <param-desc>Index of the worksheet. null means the current Worksheet.</param-desc>
   </param>
    <param>
     <param-def>p_row pls_integer default null</param-def>
     <param-desc>Index of the row. null means the current Row.</param-desc>
   </param>
    <param>
     <param-def>p_cell varchar2</param-def>
     <param-desc>Specifies a reference to a previously defined cell.</param-desc>
   </param>
   <param>
     <param-def>p_style varchar2 default 'single'</param-def>
     <param-desc>The style is single or double</param-desc>
   </param>
 </params>
 <return>-</return>
*/
  procedure Cell_setUnderline(
    p_ws   pls_integer default null,
    p_row  pls_integer default null,
    p_cell varchar2,
    p_style varchar2 default 'single'
    );

/**
<name>Cell_setColor</name>
 <type>procedure</type>
 <desc>
 Specifies the color of celltext.
 </desc>
 <version>v1.0.0</version>
 <params>
   <param>
     <param-def>p_ws pls_integer default null</param-def>
     <param-desc>Index of the worksheet. null means the current Worksheet.</param-desc>
   </param>
    <param>
     <param-def>p_row pls_integer default null</param-def>
     <param-desc>Index of the row. null means the current Row.</param-desc>
   </param>
    <param>
     <param-def>p_cell varchar2</param-def>
     <param-desc>Specifies a reference to a previously defined cell.</param-desc>
   </param>
    <param>
     <param-def>p_color varchar2</param-def>
     <param-desc>Specifies a reference to a previously defined color.</param-desc>
   </param>
 </params>
 <return>-</return>
*/
  procedure Cell_setColor(
    p_ws    pls_integer default null,
    p_row   pls_integer default null,
    p_cell  varchar2,
    p_color varchar2
    );

/**
<name>Cell_setBGColor</name>
 <type>procedure</type>
 <desc>
 Specifies the cells background color.
 </desc>
 <version>v1.0.0</version>
 <params>
   <param>
     <param-def>p_ws pls_integer default null</param-def>
     <param-desc>Index of the worksheet. null means the current Worksheet.</param-desc>
   </param>
    <param>
     <param-def>p_row pls_integer default null</param-def>
     <param-desc>Index of the row. null means the current Row.</param-desc>
   </param>
    <param>
     <param-def>p_cell varchar2</param-def>
     <param-desc>Specifies a reference to a previously defined cell.</param-desc>
   </param>
    <param>
     <param-def>p_color varchar2</param-def>
     <param-desc>Specifies a reference to a previously defined color.</param-desc>
   </param>
 </params>
 <return>-</return>
*/
  procedure Cell_setBGColor(
    p_ws    pls_integer default null,
    p_row   pls_integer default null,
    p_cell  varchar2,
    p_color varchar2
    );

/**
<name>Cell_setHLeft</name>
 <type>procedure</type>
 <desc>
 Set the horizontal alignment to left.
 </desc>
 <version>v1.0.0</version>
 <params>
   <param>
     <param-def>p_ws pls_integer default null</param-def>
     <param-desc>Index of the worksheet. null means the current Worksheet.</param-desc>
   </param>
    <param>
     <param-def>p_row pls_integer default null</param-def>
     <param-desc>Index of the row. null means the current Row.</param-desc>
   </param>
    <param>
     <param-def>p_cell varchar2</param-def>
     <param-desc>Specifies a reference to a previously defined cell.</param-desc>
   </param>
 </params>
 <return>-</return>
*/
  procedure Cell_setHLeft(
    p_ws   pls_integer default null,
    p_row  pls_integer default null,
    p_cell varchar2
    );

/**
<name>Cell_setHCenter</name>
 <type>procedure</type>
 <desc>
 Set the horizontal alignment to center.
 </desc>
 <version>v1.0.0</version>
 <params>
   <param>
     <param-def>p_ws pls_integer default null</param-def>
     <param-desc>Index of the worksheet. null means the current Worksheet.</param-desc>
   </param>
    <param>
     <param-def>p_row pls_integer default null</param-def>
     <param-desc>Index of the row. null means the current Row.</param-desc>
   </param>
    <param>
     <param-def>p_cell varchar2</param-def>
     <param-desc>Specifies a reference to a previously defined cell.</param-desc>
   </param>
 </params>
 <return>-</return>
*/
  procedure Cell_setHCenter(
    p_ws   pls_integer default null,
    p_row  pls_integer default null,
    p_cell varchar2
    );

/**
<name>Cell_setHRight</name>
 <type>procedure</type>
 <desc>
 Set the horizontal alignment to right.
 </desc>
 <version>v1.0.0</version>
 <params>
   <param>
     <param-def>p_ws pls_integer default null</param-def>
     <param-desc>Index of the worksheet. null means the current Worksheet.</param-desc>
   </param>
    <param>
     <param-def>p_row pls_integer default null</param-def>
     <param-desc>Index of the row. null means the current Row.</param-desc>
   </param>
    <param>
     <param-def>p_cell varchar2</param-def>
     <param-desc>Specifies a reference to a previously defined cell.</param-desc>
   </param>
 </params>
 <return>-</return>
*/
  procedure Cell_setHRight(
    p_ws   pls_integer default null,
    p_row  pls_integer default null,
    p_cell varchar2
    );

/**
<name>Cell_setVTOP</name>
 <type>procedure</type>
 <desc>
 Set the vertical alignment to top.
 </desc>
 <version>v1.0.0</version>
 <params>
   <param>
     <param-def>p_ws pls_integer default null</param-def>
     <param-desc>Index of the worksheet. null means the current Worksheet.</param-desc>
   </param>
    <param>
     <param-def>p_row pls_integer default null</param-def>
     <param-desc>Index of the row. null means the current Row.</param-desc>
   </param>
    <param>
     <param-def>p_cell varchar2</param-def>
     <param-desc>Specifies a reference to a previously defined cell.</param-desc>
   </param>
 </params>
 <return>-</return>
*/
  procedure Cell_setVTop(
    p_ws   pls_integer default null,
    p_row  pls_integer default null,
    p_cell varchar2
    );

/**
<name>Cell_setVCenter</name>
 <type>procedure</type>
 <desc>
 Set the vertical alignment to center.
 </desc>
 <version>v1.0.0</version>
 <params>
   <param>
     <param-def>p_ws pls_integer default null</param-def>
     <param-desc>Index of the worksheet. null means the current Worksheet.</param-desc>
   </param>
    <param>
     <param-def>p_row pls_integer default null</param-def>
     <param-desc>Index of the row. null means the current Row.</param-desc>
   </param>
    <param>
     <param-def>p_cell varchar2</param-def>
     <param-desc>Specifies a reference to a previously defined cell.</param-desc>
   </param>
 </params>
 <return>-</return>
*/
  procedure Cell_setVCenter(
    p_ws   pls_integer default null,
    p_row  pls_integer default null,
    p_cell varchar2
    );

/**
<name>Cell_setVBottom</name>
 <type>procedure</type>
 <desc>
 Set the vertical alignment to bottom.
 </desc>
 <version>v1.0.0</version>
 <params>
   <param>
     <param-def>p_ws pls_integer default null</param-def>
     <param-desc>Index of the worksheet. null means the current Worksheet.</param-desc>
   </param>
    <param>
     <param-def>p_row pls_integer default null</param-def>
     <param-desc>Index of the row. null means the current Row.</param-desc>
   </param>
    <param>
     <param-def>p_cell varchar2</param-def>
     <param-desc>Specifies a reference to a previously defined cell.</param-desc>
   </param>
 </params>
 <return>-</return>
*/
  procedure Cell_setVBottom(
    p_ws   pls_integer default null,
    p_row  pls_integer default null,
    p_cell varchar2
    );

/**
<name>Cell_setBorder_Top</name>
 <type>procedure</type>
 <desc>
 Sets a top side border for the cell.
 </desc>
 <version>v1.0.0</version>
 <params>
   <param>
     <param-def>p_ws pls_integer default null</param-def>
     <param-desc>Index of the worksheet. null means the current Worksheet.</param-desc>
   </param>
    <param>
     <param-def>p_row pls_integer default null</param-def>
     <param-desc>Index of the row. null means the current Row.</param-desc>
   </param>
    <param>
     <param-def>p_cell varchar2</param-def>
     <param-desc>Specifies a reference to a previously defined cell.</param-desc>
   </param>
    <param>
     <param-def>p_LineStyle varchar2 default c_thin</param-def>
     <param-desc>Specifies the border line style.</param-desc>
   </param>
    <param>
     <param-def>p_color varchar2 default c_black</param-def>
     <param-desc>Specifies the border color</param-desc>
   </param>
 </params>
 <return>-</return>
*/
  procedure Cell_setBorder_Top(
    p_ws        pls_integer default null,
    p_row       pls_integer default null,
    p_cell      varchar2,
    p_LineStyle varchar2 default ploffx_xlsx_styles.c_thin,
    p_color     varchar2 default ploffx_xlsx_styles.c_black
    );

/**
<name>Cell_setBorder_Bottom</name>
 <type>procedure</type>
 <desc>
 Sets a bottom side border for the cell.
 </desc>
 <version>v1.0.0</version>
 <params>
   <param>
     <param-def>p_ws pls_integer default null</param-def>
     <param-desc>Index of the worksheet. null means the current Worksheet.</param-desc>
   </param>
    <param>
     <param-def>p_row pls_integer default null</param-def>
     <param-desc>Index of the row. null means the current Row.</param-desc>
   </param>
    <param>
     <param-def>p_cell varchar2</param-def>
     <param-desc>Specifies a reference to a previously defined cell.</param-desc>
   </param>
    <param>
     <param-def>p_LineStyle varchar2 default c_thin</param-def>
     <param-desc>Specifies the border line style.</param-desc>
   </param>
    <param>
     <param-def>p_color varchar2 default c_black</param-def>
     <param-desc>Specifies the border color</param-desc>
   </param>
 </params>
 <return>-</return>
*/
  procedure Cell_setBorder_Bottom(
    p_ws        pls_integer default null,
    p_row       pls_integer default null,
    p_cell      varchar2,
    p_LineStyle varchar2 default ploffx_xlsx_styles.c_thin,
    p_color     varchar2 default ploffx_xlsx_styles.c_black
    );

/**
<name>Cell_setBorder_Left</name>
 <type>procedure</type>
 <desc>
 Sets a left side border for the cell.
 </desc>
 <version>v1.0.0</version>
 <params>
   <param>
     <param-def>p_ws pls_integer default null</param-def>
     <param-desc>Index of the worksheet. null means the current Worksheet.</param-desc>
   </param>
    <param>
     <param-def>p_row pls_integer default null</param-def>
     <param-desc>Index of the row. null means the current Row.</param-desc>
   </param>
    <param>
     <param-def>p_cell varchar2</param-def>
     <param-desc>Specifies a reference to a previously defined cell.</param-desc>
   </param>
    <param>
     <param-def>p_LineStyle varchar2 default c_thin</param-def>
     <param-desc>Specifies the border line style.</param-desc>
   </param>
    <param>
     <param-def>p_color varchar2 default c_black</param-def>
     <param-desc>Specifies the border color</param-desc>
   </param>
 </params>
 <return>-</return>
*/
  procedure Cell_setBorder_Left(
    p_ws pls_integer default null,
    p_row pls_integer default null,
    p_cell varchar2,
    p_LineStyle    varchar2 default ploffx_xlsx_styles.c_thin,
    p_color    varchar2 default ploffx_xlsx_styles.c_black
    );

/**
<name>Cell_setBorder_Right</name>
 <type>procedure</type>
 <desc>
 Sets a right side border for the cell.
 </desc>
 <version>v1.0.0</version>
 <params>
   <param>
     <param-def>p_ws pls_integer default null</param-def>
     <param-desc>Index of the worksheet. null means the current Worksheet.</param-desc>
   </param>
    <param>
     <param-def>p_row pls_integer default null</param-def>
     <param-desc>Index of the row. null means the current Row.</param-desc>
   </param>
    <param>
     <param-def>p_cell varchar2</param-def>
     <param-desc>Specifies a reference to a previously defined cell.</param-desc>
   </param>
    <param>
     <param-def>p_LineStyle varchar2 default c_thin</param-def>
     <param-desc>Specifies the border line style.</param-desc>
   </param>
    <param>
     <param-def>p_color varchar2 default c_black</param-def>
     <param-desc>Specifies the border color</param-desc>
   </param>
 </params>
 <return>-</return>
*/
  procedure Cell_setBorder_Right(
    p_ws        pls_integer default null,
    p_row       pls_integer default null,
    p_cell      varchar2,
    p_LineStyle varchar2 default ploffx_xlsx_styles.c_thin,
    p_color     varchar2 default ploffx_xlsx_styles.c_black
    );

/**
<name>Cell_setBorder</name>
 <type>procedure</type>
 <desc>
 Sets border for the cell.
 </desc>
 <version>v1.0.0</version>
 <params>
   <param>
     <param-def>p_ws pls_integer default null</param-def>
     <param-desc>Index of the worksheet. null means the current Worksheet.</param-desc>
   </param>
    <param>
     <param-def>p_row pls_integer default null</param-def>
     <param-desc>Index of the row. null means the current Row.</param-desc>
   </param>
    <param>
     <param-def>p_cell varchar2</param-def>
     <param-desc>Specifies a reference to a previously defined cell.</param-desc>
   </param>
    <param>
     <param-def>p_LineStyle varchar2 default c_thin</param-def>
     <param-desc>Specifies the border line style.</param-desc>
   </param>
    <param>
     <param-def>p_color varchar2 default c_black</param-def>
     <param-desc>Specifies the border color</param-desc>
   </param>
 </params>
 <return>-</return>
*/
procedure Cell_setBorder(
  p_ws        pls_integer default null,
  p_row       pls_integer default null,
  p_cell      varchar2,
  p_LineStyle varchar2 default ploffx_xlsx_styles.c_thin,
  p_color     varchar2 default ploffx_xlsx_styles.c_black
  );

--v1.3.0
procedure Cell_setTextRotation(
  p_ws pls_integer default null,
  p_row pls_integer default null,
  p_cell varchar2,
  p_textrotation pls_integer
  );

/**
<name>Cell_setWrapText</name>
 <type>procedure</type>
 <desc>
 Wrap up the text.
 </desc>
 <version>v1.0.0</version>
 <params>
   <param>
     <param-def>p_ws pls_integer default null</param-def>
     <param-desc>Index of the worksheet. null means the current Worksheet.</param-desc>
   </param>
    <param>
     <param-def>p_row pls_integer default null</param-def>
     <param-desc>Index of the row. null means the current Row.</param-desc>
   </param>
    <param>
     <param-def>p_cell varchar2</param-def>
     <param-desc>Specifies a reference to a previously defined cell.</param-desc>
   </param>
 </params>
 <return>-</return>
*/
  procedure Cell_setWrapText(
    p_ws   pls_integer default null,
    p_row  pls_integer default null,
    p_cell varchar2
    );

/**
<name>mergeCells</name>
 <type>procedure</type>
 <desc>
  Merge two or more cells into one cell.
 </desc>
 <version>v1.0.0</version>
 <params>
   <param>
     <param-def>p_ws pls_integer default null</param-def>
     <param-desc>Index of the worksheet. null means the current Worksheet.</param-desc>
   </param>
    <param>
     <param-def>p_row pls_integer default null</param-def>
     <param-desc>Index of the row. null means the current Row.</param-desc>
   </param>
   <param>
     <param-def>p_cell_from varchar2</param-def>
     <param-desc>The first cell of merging.</param-desc>
   </param>
   <param>
     <param-def>p_cell_to varchar2</param-def>
     <param-desc>The last cell of merging.</param-desc>
   </param>
 </params>
 <return>-</return>
*/
procedure mergeCells(
  p_ws pls_integer default null,
  p_row pls_integer default null,
  p_cell_from varchar2,
  p_cell_to   varchar2
  );

/**
<name>mergeRows</name>
 <type>procedure</type>
 <desc>
  Merges two or more cells into one cell.
 </desc>
 <version>v1.0.0</version>
 <params>
   <param>
     <param-def>p_ws pls_integer default null</param-def>
     <param-desc>Index of the worksheet. null means the current Worksheet.</param-desc>
   </param>
    <param>
     <param-def>p_row pls_integer default null</param-def>
     <param-desc>Index of the row. null means the current Row.</param-desc>
   </param>
    <param>
     <param-def>p_cell varchar2</param-def>
     <param-desc>Specifies a reference to a previously defined cell.</param-desc>
   </param>
   <param>
     <param-def>p_num_rows pls_integer</param-def>
     <param-desc>Number of rows</param-desc>
   </param>
 </params>
 <return>-</return>
*/
procedure mergeRows(
  p_ws pls_integer default null,
  p_row pls_integer default null,
  p_cell varchar2,
  p_num_rows pls_integer
  );

/**
<name>setShowGridLines</name>
 <type>procedure</type>
 <desc>
  Sets the grid line showing for the worksheet.
 </desc>
 <version>v4.9.4</version>
 <params>
   <param>
     <param-def>p_ws pls_integer default null</param-def>
     <param-desc>Index of the worksheet. null means the current Worksheet.</param-desc>
   </param>
    <param>
     <param-def>p_visible boolean default null</param-def>
     <param-desc>Visible or not</param-desc>
   </param>
 </params>
 <return>-</return>
*/
procedure setShowGridLines(
  p_ws pls_integer default null,
  p_visible boolean default null
  );

/**
<name>setAutoFilter</name>
 <type>procedure</type>
 <desc>
  Create AutoFilter in the table.
 </desc>
 <version>v2.0.0</version>
 <params>
   <param>
     <param-def>p_ws pls_integer default null</param-def>
     <param-desc>Index of the worksheet. null means the current Worksheet.</param-desc>
   </param>
    <param>
     <param-def>p_column_start varchar2 default null</param-def>
     <param-desc>First column of the filter.</param-desc>
   </param>
    <param>
     <param-def>p_column_end varchar2 default null</param-def>
     <param-desc>Last column of the filter.</param-desc>
   </param>
   <param>
     <param-def>p_row_start pls_integer default null</param-def>
     <param-desc>First row of the filter.</param-desc>
   </param>
   <param>
     <param-def>p_row_end pls_integer default null</param-def>
     <param-desc>Last row of the filter.</param-desc>
   </param>
 </params>
 <return>-</return>
*/
procedure setAutofilter(
  p_ws pls_integer default null,
  p_column_start varchar2 default null,
  p_column_end varchar2 default null,
  p_row_start pls_integer default null,
  p_row_end pls_integer default null
  );

/**
<name>Cell_setHyperlink</name>
 <type>procedure</type>
 <desc>
  Add a hyperlink to the cell
 </desc>
 <version>v1.0.0</version>
 <params>
  <param>
   <param-def>p_ws pls_integer default null</param-def>
   <param-desc>Index of the worksheet. null means the current Worksheet.</param-desc>
   </param>
    <param>
     <param-def>p_row pls_integer default null</param-def>
     <param-desc>Index of the row. null means the current Row.</param-desc>
   </param>
    <param>
     <param-def>p_cell varchar2</param-def>
     <param-desc>Specifies a reference to a previously defined cell.</param-desc>
   </param>
   <param>
     <param-def>p_hyperlink varchar2</param-def>
     <param-desc>The URL text.</param-desc>
   </param>
  </params>
 <return>-</return>
*/
procedure Cell_setHyperlink(
    p_ws        pls_integer default null,
    p_row       pls_integer default null,
    p_cell      varchar2,
    p_hyperlink varchar2
    );

/**
<name>Cell_setFormatMask</name>
 <type>procedure</type>
 <desc>
  Sets the cell format mask.
 </desc>
 <version>v1.0.0</version>
 <params>
   <param>
   <param-def>p_ws pls_integer default null</param-def>
     <param-desc>Index of the worksheet. null means the current Worksheet.</param-desc>
   </param>
    <param>
     <param-def>p_row pls_integer default null</param-def>
     <param-desc>Index of the row. null means the current Row.</param-desc>
   </param>
    <param>
     <param-def>p_cell varchar2</param-def>
     <param-desc>Specifies a reference to a previously defined cell.</param-desc>
   </param>
   <param>
     <param-def>p_format_mask varchar2</param-def>
     <param-desc>Specifies a custom format mask.</param-desc>
   </param>
  </params>
 <return>-</return>
*/
procedure Cell_setFormatMask(
  p_ws pls_integer default null,
  p_row pls_integer default null,
  p_cell varchar2,
  p_format_mask    varchar2
  );

/**
<name>Cell_setComment</name>
 <type>procedure</type>
 <desc>Contains the comment to be associated with this cell.</desc>
 <version>v4.9.3</version>
 <params>
   <param>
     <param-def>p_ws pls_integer default null</param-def>
     <param-desc>Index of the worksheet. null means the current Worksheet.</param-desc>
   </param>
    <param>
     <param-def>p_row pls_integer default null</param-def>
     <param-desc>Index of the row. null means the current Row.</param-desc>
   </param>
   <param>
     <param-def>p_cell varchar2</param-def>
     <param-desc>Specifies a reference to a previously defined cell.</param-desc>
   </param>
    <param>
     <param-def>p_text   varchar2</param-def>
     <param-desc>Specifies the text of the comment</param-desc>
   </param>
   <param>
     <param-def>p_author varchar2</param-def>
     <param-desc>Specifies the author of this comment.</param-desc>
   </param>
    <param>
     <param-def>p_width  number default 96</param-def>
     <param-desc>Specifies the width of this comment.</param-desc>
   </param>
     <param>
     <param-def>p_height number default 96</param-def>
     <param-desc>Specifies the height of this comment.</param-desc>
   </param>
 </params>
 <return>-</return>
*/
procedure Cell_setComment(
    p_ws     pls_integer default null,
    p_row    pls_integer default null,
    p_cell   varchar2,
    p_text   varchar2,
    p_author varchar2 default null,
    p_width  number default 96,  -- pixels
    p_height number default 55  -- pixels
    );

/**
<name>Cell_setFormula</name>
 <type>procedure</type>
 <desc>Set formula of the cell.</desc>
 <version>v1.2.0</version>
 <params>
   <param>
     <param-def>p_ws pls_integer default null</param-def>
     <param-desc>Index of the worksheet. null means the current Worksheet.</param-desc>
   </param>
    <param>
     <param-def>p_row pls_integer default null</param-def>
     <param-desc>Index of the row. null means the current Row.</param-desc>
   </param>
   <param>
     <param-def>p_cell varchar2</param-def>
     <param-desc>Specifies a reference to a previously defined cell.</param-desc>
   </param>
    <param>
     <param-def>p_formula   varchar2</param-def>
     <param-desc>Specifies the text of the formula</param-desc>
   </param>
 </params>
 <return>-</return>
*/
procedure Cell_setFormula(
  p_ws      pls_integer default null,
  p_row     pls_integer default null,
  p_cell    varchar2,
  p_formula varchar2
  );

/**
<name>Cell_getFormula</name>
 <type>function</type>
 <desc>Get formula of the cell.</desc>
 <version>v4.9.4</version>
 <params>
   <param>
     <param-def>p_ws pls_integer default null</param-def>
     <param-desc>Index of the worksheet. null means the current Worksheet.</param-desc>
   </param>
    <param>
     <param-def>p_row pls_integer default null</param-def>
     <param-desc>Index of the row. null means the current Row.</param-desc>
   </param>
   <param>
     <param-def>p_cell varchar2</param-def>
     <param-desc>Specifies a reference to a previously defined cell.</param-desc>
   </param>
 </params>
 <return>
   <return-def>varchar2</return-def>
   <return-desc>The formula of the cell</return-desc>
 </return>
*/
function Cell_getFormula(
    p_ws      pls_integer default null,
    p_row     pls_integer default null,
    p_cell    varchar2
    ) return varchar2;

/**
<name>Cell_clearContent</name>
 <type>procedure</type>
 <desc>Clear the value of the cell.</desc>
 <version>v4.9.2</version>
 <params>
   <param>
     <param-def>p_ws pls_integer default null</param-def>
     <param-desc>Index of the worksheet. null means the current Worksheet.</param-desc>
   </param>
    <param>
     <param-def>p_row pls_integer default null</param-def>
     <param-desc>Index of the row. null means the current Row.</param-desc>
   </param>
   <param>
     <param-def>p_cell varchar2</param-def>
     <param-desc>Specifies a reference to a previously defined cell.</param-desc>
   </param>
 </params>
 <return>-</return>
*/
procedure Cell_clearContent(
  p_ws      pls_integer default null,
  p_row     pls_integer default null,
  p_cell    varchar2
  );

/**
<name>Cell_getConditionalFormatting</name>
 <type>function</type>
 <desc>Get formula of the cell.</desc>
 <version>v4.9.4</version>
 <params>
   <param>
     <param-def>p_ws pls_integer default null</param-def>
     <param-desc>Index of the worksheet. null means the current Worksheet.</param-desc>
   </param>
    <param>
     <param-def>p_row pls_integer default null</param-def>
     <param-desc>Index of the row. null means the current Row.</param-desc>
   </param>
   <param>
     <param-def>p_cell varchar2</param-def>
     <param-desc>Specifies a reference to a previously defined cell.</param-desc>
   </param>
 </params>
 <return>
   <return-def>varchar2</return-def>
   <return-desc>The formula of the cell</return-desc>
 </return>
*/
function Cell_getConditionalFormatting(
  p_ws      pls_integer default null,
  p_row     pls_integer default null,
  p_cell    varchar2
  ) return ploffx_sheet.CT_ConditionalFormatting;

/**
<name>Cell_setConditionalFormatting</name>
 <type>procedure</type>
 <desc>Set the conditional formatting of the cell.</desc>
 <version>v4.9.4</version>
 <params>
   <param>
     <param-def>p_ws pls_integer default null</param-def>
     <param-desc>Index of the worksheet. null means the current Worksheet.</param-desc>
   </param>
    <param>
     <param-def>p_row pls_integer default null</param-def>
     <param-desc>Index of the row. null means the current Row.</param-desc>
   </param>
   <param>
     <param-def>p_cell varchar2</param-def>
     <param-desc>Specifies a reference to a previously defined cell.</param-desc>
   </param>
   <param>
     <param-def>p_cf ploffx_sheet.CT_ConditionalFormatting</param-def>
     <param-desc>The defined conditional formatting</param-desc>
   </param>
 </params>
 <return>-</return>
*/
procedure Cell_setConditionalFormatting(
  p_ws      pls_integer default null,
  p_row     pls_integer default null,
  p_cell    varchar2,
  p_cf ploffx_sheet.CT_ConditionalFormatting
  );

/**
<name>Cell_delConditionalFormatting</name>
 <type>procedure</type>
 <desc>Delete the conditional formatting of the cell.</desc>
 <version>v4.9.4</version>
 <params>
   <param>
     <param-def>p_ws pls_integer default null</param-def>
     <param-desc>Index of the worksheet. null means the current Worksheet.</param-desc>
   </param>
    <param>
     <param-def>p_row pls_integer default null</param-def>
     <param-desc>Index of the row. null means the current Row.</param-desc>
   </param>
   <param>
     <param-def>p_cell varchar2</param-def>
     <param-desc>Specifies a reference to a previously defined cell.</param-desc>
   </param>
 </params>
 <return>-</return>
*/
procedure Cell_delConditionalFormatting(
  p_ws      pls_integer default null,
  p_row     pls_integer default null,
  p_cell    varchar2
  );


--v4.9.4
function Cell_getStyleID(
  p_ws      pls_integer default null,
  p_row     pls_integer default null,
  p_cell    varchar2
  ) return pls_integer;

--v4.9.7
procedure Cell_setStyleID(
  p_ws pls_integer default null,
  p_row pls_integer default null,
  p_cell varchar2,
  p_style_id pls_integer
  );

--v4.9.8
function Cell_getFormat(
  p_ws      pls_integer default null,
  p_row     pls_integer default null,
  p_cell    varchar2
  ) return pls_integer;

--v4.9.8
procedure Cell_setFormat(
  p_ws pls_integer default null,
  p_row pls_integer default null,
  p_cell varchar2,
  p_format_id pls_integer
  );

/**
 <name>getDoc</name>
 <type>procedure</type>
 <desc>?
 Returns the generated document. The document is closed and then returned.
 </desc>
 <version>v4.9.2</version>
 <params>
   <param>
     <param-def>p_blob_file in out nocopy blob</param-def>
     <param-desc>The XLSX file as BLOB variable</param-desc>
   </param>
 </params>
 <return>-</return>
*/
procedure getDoc(
  p_blob_file in out nocopy blob,
  p_formulaClearContent boolean default false
  );

/**
 <name>parseDoc</name>
 <type>procedure</type>
 <desc>
 Parses the XLSX file and store in PLXLSX internal structure.
 </desc>
 <version>v4.9.5</version>
 <params>
   <param>
     <param-def>p_blob_file in out blob</param-def>
     <param-desc>The XLSX file as BLOB variable</param-desc>
   </param>
   <param>
     <param-def>p_parse_style boolean default true</param-def>
     <param-desc>Parsing styles or not in XLSX file</param-desc>
   </param>
 </params>
 <return>-</return>
*/
procedure parseDoc(
  p_blob_file in out nocopy blob,
  p_parse_style boolean default true
  );

/**
 <name>getDocument</name>
 <type>function</type>
 <desc>
 Gets the XLSX as PLXLSX internal structure.
 </desc>
 <version>v?4.9.0</version>
 <params>-</params>
 <return>
   <return-def>t_xlsx</return-def>
   <return-desc>The XLSX as t_xlsx structure</return-desc>
 </return>
*/
function getDocument
  return t_xlsx;

/**
 <name>setDocument</name>
 <type>procedure</type>
 <desc>
 Sets the XLSX structure and store in PLXLSX internal structure.
 </desc>
 <version>v4.9.0</version>
 <params>
   <param>
     <param-def>p_xlsx t_xlsx</param-def>
     <param-desc>The XLSX internal structure</param-desc>
   </param>
 </params>
 <return>-</return>
*/
procedure setDocument(
  p_xlsx t_xlsx
  );

/**
 <name>AddImage</name>
 <type>procedure</type>
 <desc>
 Adds image to the worksheet
 </desc>
 <version>v3.0.0</version>
 <params>
    <param>
     <param-def>p_ws pls_integer default null</param-def>
     <param-desc>Index of the worksheet. null means the current Worksheet.</param-desc>
   </param>
   <param>
     <param-def>p_row_from pls_integer</param-def>
     <param-desc>Index of the row.</param-desc>
   </param>
   <param>
     <param-def>p_row_from_offset number default 0</param-def>
     <param-desc>Specifies the offset from the row in pixels</param-desc>
   </param>
   <param>
     <param-def>p_cell_from varchar2</param-def>
     <param-desc>Index of the cell.</param-desc>
   </param>
   <param>
     <param-def>p_cell_from_offset number default 0</param-def>
     <param-desc>Specifies the offset from the cell in pixels</param-desc>
   </param>
    <param>
     <param-def>p_name varchar2</param-def>
     <param-desc>Specifies the name of the image with extension.</param-desc>
   </param>
   <param>
     <param-def>p_image blob</param-def>
     <param-desc>Specifies the image file.</param-desc>
   </param>
   <param>
     <param-def>p_height number</param-def>
     <param-desc>Specifies the height to display in pixels.</param-desc>
   </param>
   <param>
     <param-def>p_width number</param-def>
     <param-desc>Specifies the width to display in pixels.</param-desc>
   </param>
   <param>
     <param-def>p_descr varchar2 default null</param-def>
     <param-desc>Specifies the description for the image.</param-desc>
   </param>
 </params>
 <return>-</return>
*/
procedure addImage(
  p_ws pls_integer default null,
  p_row_from  pls_integer,
  p_row_from_offset number default 0, --pixel
  p_column_from varchar2,
  p_column_from_offset number default 0,
  p_name varchar2, --with extension
  p_image blob,
  p_height number,
  p_width number,
  p_descr varchar2 default null
  );
/**
 <name>setPrinterSettings</name>
 <type>procedure</type>
 <desc>
 Adds printer setting to a worksheet
 </desc>
 <version>v4.9.1</version>
 <params>
    <param>
     <param-def>p_ws pls_integer default null</param-def>
     <param-desc>Index of the worksheet. null means the current Worksheet.</param-desc>
   </param>
    <param>
     <param-def>p_blackAndWhite boolean default null</param-def>
     <param-desc>Specifies the printing setting black and white</param-desc>
   </param>
   <param>
     <param-def>p_orientation ploffx_sheet.ST_Orientation default null</param-def>
     <param-desc>Specifies the orientation of the printing.</param-desc>
   </param>
    <param>
     <param-def>p_paperSize   ploffx_sheet.XT_PaperSize default ploffx_sheet.paperSize_A4</param-def>
     <param-desc>Specifies the paper size for the printing.</param-desc>
   </param>
   <param>
     <param-def>p_fitToHeight pls_integer default null</param-def>
     <param-desc>Number of vertical pages to fit on.</param-desc>
   </param>
   <param>
     <param-def>p_fitToWidth pls_integer default null</param-def>
     <param-desc>Number of horizontal pages to fit on.</param-desc>
   </param>
 </params>
 <return>-</return>
*/
procedure setPrinterSettings(
  p_ws pls_integer default null,
  p_blackAndWhite boolean default null,
  p_orientation ploffx_sheet.ST_Orientation default null,
  p_paperSize ploffx_sheet.XT_PaperSize default ploffx_sheet.paperSize_A4,
  p_fitToHeight pls_integer default null,
  p_fitToWidth pls_integer default null
  );

/**
 <name>loadDocument</name>
 <type>procedure</type>
 <desc>
 Loads XLSX file into the internal PLOFFX structure for modifiing or filling cells and rows.
 </desc>
 <version>v5.4</version>
 <params>
 <param>
   <param-def>p_xls in out nocopy blob</param-def>
   <param-desc>The XLSX file as BLOB variable</param-desc>
  </param>
 </params>
 <return>-</return>
*/
procedure loadDocument(
  p_xls in out nocopy blob,
  p_SharedFormulas2Formulas boolean default true 
  );

--v4.9.6
procedure SharedFormulas2Formulas;

/**
<name>setCellValue</name>
 <type>procedure</type>
 <desc>
 Set the value property for the referenced cell.
 </desc>
 <version>v4.9.0</version>
 <params>
   <param>
     <param-def>p_ws pls_integer default null</param-def>
     <param-desc>Index of the worksheet. null means the current Worksheet.</param-desc>
   </param>
    <param>
     <param-def>p_row pls_integer default null</param-def>
     <param-desc>Index of the row. null means the current Row.</param-desc>
   </param>
   <param>
     <param-def>p_cell varchar2</param-def>
     <param-desc>Specifies a reference to a previously defined cell.</param-desc>
   </param>
   <param>
     <param-def>p_data number</param-def>
     <param-desc>Specifies the value of this cell.</param-desc>
   </param>
 </params>
 <return>-</return>
*/
procedure setCellValue(
  p_ws pls_integer default null,
  p_row pls_integer default null,
  p_cell varchar2,
  p_data    number
  );

/**
<name>setCellValue</name>
 <type>procedure</type>
 <desc>
 Set the value property for the referenced cell.
 </desc>
 <version>v4.9.0</version>
 <params>
   <param>
     <param-def>p_ws pls_integer default null</param-def>
     <param-desc>Index of the worksheet. null means the current Worksheet.</param-desc>
   </param>
    <param>
     <param-def>p_row pls_integer default null</param-def>
     <param-desc>Index of the row. null means the current Row.</param-desc>
   </param>
   <param>
     <param-def>p_cell varchar2</param-def>
     <param-desc>Specifies a reference to a previously defined cell.</param-desc>
   </param>
   <param>
     <param-def>p_data varchar2</param-def>
     <param-desc>Specifies the value of this cell.</param-desc>
   </param>
 </params>
 <return>-</return>
*/
procedure setCellValue(
  p_ws pls_integer default null,
  p_row pls_integer default null,
  p_cell varchar2,
  p_data    varchar2
  );

/**
<name>setCellValue</name>
 <type>procedure</type>
 <desc>
 Set the value property for the referenced cell.
 </desc>
 <version>v4.9.0</version>
 <params>
   <param>
     <param-def>p_ws pls_integer default null</param-def>
     <param-desc>Index of the worksheet. null means the current Worksheet.</param-desc>
   </param>
    <param>
     <param-def>p_row pls_integer default null</param-def>
     <param-desc>Index of the row. null means the current Row.</param-desc>
   </param>
   <param>
     <param-def>p_cell varchar2</param-def>
     <param-desc>Specifies a reference to a previously defined cell.</param-desc>
   </param>
   <param>
     <param-def>p_data date</param-def>
     <param-desc>Specifies the value of this cell.</param-desc>
   </param>
 </params>
 <return>-</return>
*/
procedure setCellValue(
  p_ws pls_integer default null,
  p_row pls_integer default null,
  p_cell varchar2,
  p_data    date
  );

/**
<name>getCellValue</name>
 <type>function</type>
 <desc>
 Gets the text value property for the referenced cell.
 </desc>
 <version>v4.9.0</version>
 <params>
   <param>
     <param-def>p_ws pls_integer default null</param-def>
     <param-desc>Index of the worksheet. null means the current Worksheet.</param-desc>
   </param>
    <param>
     <param-def>p_row pls_integer default null</param-def>
     <param-desc>Index of the row. null means the current Row.</param-desc>
   </param>
   <param>
     <param-def>p_cell varchar2</param-def>
     <param-desc>Specifies a reference to a previously defined cell.</param-desc>
   </param>
 </params>
 <return>
 <return-def>varchar2</return-def>
  <return-desc>The value of the cell</return-desc>
 </return>
*/
function getCellValue(
    p_ws pls_integer default null,
    p_row pls_integer default null,
    p_cell varchar2
    ) return varchar2;

/**
<name>getCellValueString</name>
 <type>function</type>
 <desc>
 Gets the text value of the referenced cell.
 </desc>
 <version>v4.9.0</version>
 <params>
   <param>
     <param-def>p_ws pls_integer default null</param-def>
     <param-desc>Index of the worksheet. null means the current Worksheet.</param-desc>
   </param>
    <param>
     <param-def>p_row pls_integer default null</param-def>
     <param-desc>Index of the row. null means the current Row.</param-desc>
   </param>
   <param>
     <param-def>p_cell varchar2</param-def>
     <param-desc>Specifies a reference to a previously defined cell.</param-desc>
   </param>
 </params>
 <return>
 <return-def>varchar2</return-def>
  <return-desc>The value of the cell</return-desc>
 </return>
*/
function getCellValueString(
  p_ws pls_integer default null,
  p_row pls_integer default null,
  p_cell varchar2
  ) return varchar2;

/**
<name>getCellValueDate</name>
 <type>function</type>
 <desc>
 Gets the date value of the referenced cell.
 </desc>
 <version>v4.9.0</version>
 <params>
   <param>
     <param-def>p_ws pls_integer default null</param-def>
     <param-desc>Index of the worksheet. null means the current Worksheet.</param-desc>
   </param>
    <param>
     <param-def>p_row pls_integer default null</param-def>
     <param-desc>Index of the row. null means the current Row.</param-desc>
   </param>
   <param>
     <param-def>p_cell varchar2</param-def>
     <param-desc>Specifies a reference to a previously defined cell.</param-desc>
   </param>
 </params>
 <return>
 <return-def>date</return-def>
  <return-desc>The value of the cell</return-desc>
 </return>
*/
function getCellValueDate(
  p_ws pls_integer default null,
  p_row pls_integer default null,
  p_cell varchar2
  ) return date;

/**
<name>getCellValueNumber</name>
 <type>function</type>
 <desc>
 Gets the number value of the referenced cell.
 </desc>
 <version>v4.9.0</version>
 <params>
   <param>
     <param-def>p_ws pls_integer default null</param-def>
     <param-desc>Index of the worksheet. null means the current Worksheet.</param-desc>
   </param>
    <param>
     <param-def>p_row pls_integer default null</param-def>
     <param-desc>Index of the row. null means the current Row.</param-desc>
   </param>
   <param>
     <param-def>p_cell varchar2</param-def>
     <param-desc>Specifies a reference to a previously defined cell.</param-desc>
   </param>
 </params>
 <return>
 <return-def>number</return-def>
  <return-desc>The value of the cell</return-desc>
 </return>
*/
function getCellValueNumber(
  p_ws pls_integer default null,
  p_row pls_integer default null,
  p_cell varchar2
  ) return number;

/**
<name>getCellDataType</name>
 <type>function</type>
 <desc>
 Gets the data type of the referenced cell.
 </desc>
 <version>v4.9.0</version>
 <params>
   <param>
     <param-def>p_ws pls_integer default null</param-def>
     <param-desc>Index of the worksheet. null means the current Worksheet.</param-desc>
   </param>
    <param>
     <param-def>p_row pls_integer default null</param-def>
     <param-desc>Index of the row. null means the current Row.</param-desc>
   </param>
   <param>
     <param-def>p_cell varchar2</param-def>
     <param-desc>Specifies a reference to a previously defined cell.</param-desc>
   </param>
 </params>
 <return>
 <return-def>varchar2</return-def>
  <return-desc>The value of the cell, possible values are: text/date/number</return-desc>
 </return>
*/
function getCellDataType(
    p_ws pls_integer default null,
    p_row pls_integer default null,
    p_cell varchar2
  ) return varchar2;

/**
<name>getCellFormatString</name>
 <type>function</type>
 <desc>
 Gets the format string of the referenced cell, it works for number and date cell.
 </desc>
 <version>v4.9.0</version>
 <params>
   <param>
     <param-def>p_ws pls_integer default null</param-def>
     <param-desc>Index of the worksheet. null means the current Worksheet.</param-desc>
   </param>
    <param>
     <param-def>p_row pls_integer default null</param-def>
     <param-desc>Index of the row. null means the current Row.</param-desc>
   </param>
   <param>
     <param-def>p_cell varchar2</param-def>
     <param-desc>Specifies a reference to a previously defined cell.</param-desc>
   </param>
 </params>
 <return>
 <return-def>varchar2</return-def>
  <return-desc>The format string of the cell.</return-desc>
 </return>
*/
function getCellFormatString(
    p_ws pls_integer default null,
    p_row pls_integer default null,
    p_cell varchar2
  ) return varchar2;

/**
<name>getLastCellNum</name>
 <type>function</type>
 <desc>Gets the last cell index of a sheet row.</desc>
 <version>v4.9.0</version>
 <params>
   <param>
     <param-def>p_ws pls_integer default null</param-def>
     <param-desc>Index of the worksheet. null means the current Worksheet.</param-desc>
   </param>
    <param>
     <param-def>p_row pls_integer default null</param-def>
     <param-desc>Index of the row. null means the current Row.</param-desc>
   </param>
 </params>
 <return>
 <return-def>pls_integer</return-def>
  <return-desc>The index of the last cell</return-desc>
 </return>
*/
function getLastCellNum(
    p_ws pls_integer default null,
    p_row pls_integer default null
    ) return pls_integer;


/**
<name>getMergeCells</name>
 <type>function</type>
 <desc>Gets the merged cell range of a sheet row.</desc>
 <version>v4.11</version>
 <params>
   <param>
     <param-def>p_ws pls_integer default null</param-def>
     <param-desc>Index of the worksheet. null means the current Worksheet.</param-desc>
   </param>
    <param>
     <param-def>p_row pls_integer default null</param-def>
     <param-desc>Index of the row. null means the current Row.</param-desc>
   </param>
 </params>
 <return>
 <return-def>ploffx_util.t_cellranges</return-def>
  <return-desc>The cell range</return-desc>
 </return>
*/
function getMergeCells(
  p_ws pls_integer default null,
  p_row pls_integer default null
  ) return ploffx_util.t_cellranges;

/**
<name>insertRowsAfter</name>
 <type>function</type>
 <desc>
 Inserts empty rows after the setted row into the worksheet and sets the last added row as current.
 </desc>
 <version>v4.9.0</version>
 <params>
   <param>
     <param-def>p_ws pls_integer default null</param-def>
     <param-desc>Index of the worksheet. null means the current Worksheet.</param-desc>
   </param>
    <param>
     <param-def>p_row pls_integer default null</param-def>
     <param-desc>Index of the row. null means the current Row.</param-desc>
   </param>
   <param>
     <param-def>p_num pls_integer default 1</param-def>
     <param-desc>The number of new rows.</param-desc>
   </param>
 </params>
 <return>
  <return-def>pls_integer</return-def>
  <return-desc>The index of the last added row</return-desc>
 </return>
*/
function insertRowsAfter(
  p_ws pls_integer default null,
  p_row pls_integer default null,
  p_num pls_integer default 1
  ) return pls_integer;

/**
<name>insertRowsAfter</name>
 <type>function</type>
 <desc>
 Inserts empty rows after the setted row into the worksheet and sets the last added row as current.
 </desc>
 <version>v4.9.0</version>
 <params>
   <param>
     <param-def>p_ws pls_integer default null</param-def>
     <param-desc>Index of the worksheet. null means the current Worksheet.</param-desc>
   </param>
    <param>
     <param-def>p_row pls_integer default null</param-def>
     <param-desc>Index of the row. null means the current Row.</param-desc>
   </param>
   <param>
     <param-def>p_num pls_integer default 1</param-def>
     <param-desc>The number of new rows.</param-desc>
   </param>
 </params>
 <return>-</return>
*/
procedure insertRowsAfter(
  p_ws pls_integer default null,
  p_row pls_integer default null,
  p_num pls_integer default 1
  );

/**
<name>insertRowAfter</name>
 <type>function</type>
 <desc>
 Inserts empty row after the setted row into the worksheet and sets the added row as current.
 </desc>
 <version>v4.9.0</version>
 <params>
   <param>
     <param-def>p_ws pls_integer default null</param-def>
     <param-desc>Index of the worksheet. null means the current Worksheet.</param-desc>
   </param>
    <param>
     <param-def>p_row pls_integer default null</param-def>
     <param-desc>Index of the row. null means the current Row.</param-desc>
   </param>
 </params>
 <return>
  <return-def>pls_integer</return-def>
  <return-desc>The index of the last added row</return-desc>
 </return>
*/
function insertRowAfter(
  p_ws pls_integer default null,
  p_row pls_integer default null
  ) return pls_integer;

/**
<name>insertRowAfter</name>
 <type>procedure</type>
 <desc>
 Inserts empty row after the setted row into the worksheet and sets the added row as current.
 </desc>
 <version>v4.9.0</version>
 <params>
   <param>
     <param-def>p_ws pls_integer default null</param-def>
     <param-desc>Index of the worksheet. null means the current Worksheet.</param-desc>
   </param>
    <param>
     <param-def>p_row pls_integer default null</param-def>
     <param-desc>Index of the row. null means the current Row.</param-desc>
   </param>
 </params>
 <return>-</return>
*/
procedure insertRowAfter(
  p_ws pls_integer default null,
  p_row pls_integer default null
  );

/**
<name>insertRowsAfter</name>
 <type>function</type>
 <desc>
 Inserts existing rows (see <link>getRows</link>) after the setted row into the worksheet and sets the last added row as current.
 </desc>
 <version>v4.9.0</version>
 <params>
   <param>
     <param-def>p_ws pls_integer default null</param-def>
     <param-desc>Index of the worksheet. null means the current Worksheet.</param-desc>
   </param>
    <param>
     <param-def>p_row pls_integer default null</param-def>
     <param-desc>Index of the row. null means the current Row.</param-desc>
   </param>
   <param>
     <param-def>p_data ploffx_sheet.CT_SheetData</param-def>
     <param-desc>The new rows.</param-desc>
   </param>
 </params>
 <return>
  <return-def>pls_integer</return-def>
  <return-desc>The index of the last added row</return-desc>
 </return>
*/
function insertRowsAfter(
  p_ws pls_integer default null,
  p_row pls_integer default null,
  p_data ploffx_sheet.CT_SheetData
  ) return pls_integer;

/**
<name>insertRowsAfter</name>
 <type>procedure</type>
 <desc>
 Inserts existing rows (see <link>getRows</link>) after the setted row into the worksheet and sets the last added row as current.
 </desc>
 <version>v4.9.0</version>
 <params>
   <param>
     <param-def>p_ws pls_integer default null</param-def>
     <param-desc>Index of the worksheet. null means the current Worksheet.</param-desc>
   </param>
    <param>
     <param-def>p_row pls_integer default null</param-def>
     <param-desc>Index of the row. null means the current Row.</param-desc>
   </param>
   <param>
     <param-def>p_data ploffx_sheet.CT_SheetData</param-def>
     <param-desc>The new rows.</param-desc>
   </param>
 </params>
 <return>-</return>
*/
procedure insertRowsAfter(
  p_ws pls_integer default null,
  p_row pls_integer default null,
  p_data ploffx_sheet.CT_SheetData
  );

/**
<name>insertRowsAfterMulti</name>
 <type>function</type>
 <desc>
 Inserts existing rows (see <link>getRows</link>) after the setted row into the worksheet and sets the last added row as current.
 </desc>
 <version>v4.9.4</version>
 <params>
   <param>
     <param-def>p_ws pls_integer default null</param-def>
     <param-desc>Index of the worksheet. null means the current Worksheet.</param-desc>
   </param>
    <param>
     <param-def>p_row pls_integer default null</param-def>
     <param-desc>Index of the row. null means the current Row.</param-desc>
   </param>
   <param>
     <param-def>p_data ploffx_sheet.CT_SheetData</param-def>
     <param-desc>The new rows.</param-desc>
   </param>
   <param>
     <param-def>p_num pls_integer default 1</param-def>
     <param-desc>The number of multiplication.</param-desc>
   </param>
 </params>
 <return>
  <return-def>pls_integer</return-def>
  <return-desc>The index of the last added row</return-desc>
 </return>
*/
function insertRowsAfterMulti(
  p_ws pls_integer default null,
  p_row pls_integer default null,
  p_data ploffx_sheet.CT_SheetData,
  p_num pls_integer default 1
  ) return pls_integer;

/**
<name>insertRowsAfterMulti</name>
 <type>procedure</type>
 <desc>
 Inserts existing rows (see <link>getRows</link>) after the setted row into the worksheet and sets the last added row as current.
 </desc>
 <version>v4.9.4</version>
 <params>
   <param>
     <param-def>p_ws pls_integer default null</param-def>
     <param-desc>Index of the worksheet. null means the current Worksheet.</param-desc>
   </param>
    <param>
     <param-def>p_row pls_integer default null</param-def>
     <param-desc>Index of the row. null means the current Row.</param-desc>
   </param>
   <param>
     <param-def>p_data ploffx_sheet.CT_SheetData</param-def>
     <param-desc>The new rows.</param-desc>
   </param>
   <param>
     <param-def>p_num pls_integer default 1</param-def>
     <param-desc>The number of multiplication.</param-desc>
   </param>
 </params>
 <return>-</return>
*/
procedure insertRowsAfterMulti(
  p_ws pls_integer default null,
  p_row pls_integer default null,
  p_data ploffx_sheet.CT_SheetData,
  p_num pls_integer default 1
  );

/**
<name>insertRowAfter</name>
 <type>function</type>
 <desc>
 Inserts existing row (see <link>getRow</link>) after the setted row into the worksheet and sets the added row as current.
 </desc>
 <version>v4.9.0</version>
 <params>
   <param>
     <param-def>p_ws pls_integer default null</param-def>
     <param-desc>Index of the worksheet. null means the current Worksheet.</param-desc>
   </param>
    <param>
     <param-def>p_row pls_integer default null</param-def>
     <param-desc>Index of the row. null means the current Row.</param-desc>
   </param>
   <param>
     <param-def>p_data ploffx_sheet.CT_row</param-def>
     <param-desc>The new row.</param-desc>
   </param>
 </params>
 <return>
  <return-def>pls_integer</return-def>
  <return-desc>The index of the added row</return-desc>
 </return>
*/
function insertRowAfter(
  p_ws pls_integer default null,
  p_row pls_integer default null,
  p_data ploffx_sheet.CT_row
  ) return pls_integer;

/**
<name>insertRowAfter</name>
 <type>procedure</type>
 <desc>
 Inserts existing row (see <link>getRow</link>) after the setted row into the worksheet and sets the added row as current.
 </desc>
 <version>v4.9.0</version>
 <params>
   <param>
     <param-def>p_ws pls_integer default null</param-def>
     <param-desc>Index of the worksheet. null means the current Worksheet.</param-desc>
   </param>
    <param>
     <param-def>p_row pls_integer default null</param-def>
     <param-desc>Index of the row. null means the current Row.</param-desc>
   </param>
   <param>
     <param-def>p_data ploffx_sheet.CT_row</param-def>
     <param-desc>The new row.</param-desc>
   </param>
 </params>
 <return>-</return>
*/
procedure insertRowAfter(
  p_ws pls_integer default null,
  p_row pls_integer default null,
  p_data ploffx_sheet.CT_row
  );

/**
<name>getRow</name>
 <type>function</type>
 <desc>
 Gets a row from the worksheet and return as the internal data format.
 </desc>
 <version>v4.9.0</version>
 <params>
   <param>
     <param-def>p_ws pls_integer default null</param-def>
     <param-desc>Index of the worksheet. null means the current Worksheet.</param-desc>
   </param>
    <param>
     <param-def>p_row pls_integer default null</param-def>
     <param-desc>Index of the row. null means the current Row.</param-desc>
   </param>
 </params>
 <return>
  <return-def>ploffx_sheet.CT_row</return-def>
  <return-desc>The row as structure</return-desc>
 </return>
*/
function getRow(
  p_ws pls_integer default null,
  p_row pls_integer default null
  ) return ploffx_sheet.CT_row;

/**
<name>getRows</name>
 <type>function</type>
 <desc>
 Gets a rows from the worksheet and return as the internal data format.
 </desc>
 <version>v4.9.0</version>
 <params>
   <param>
     <param-def>p_ws pls_integer default null</param-def>
     <param-desc>Index of the worksheet. null means the current Worksheet.</param-desc>
   </param>
    <param>
     <param-def>p_row pls_integer default null</param-def>
     <param-desc>Index of the start row. null means the current Row.</param-desc>
   </param>
   <param>
     <param-def>p_num pls_integer default 1</param-def>
     <param-desc>The number of the rows.</param-desc>
   </param>
 </params>
 <return>
  <return-def>ploffx_sheet.CT_SheetData</return-def>
  <return-desc>The rows as structure</return-desc>
 </return>
*/
function getRows(
  p_ws pls_integer default null,
  p_row pls_integer default null,
  p_num pls_integer default 1
  ) return ploffx_sheet.CT_SheetData;

/**
<name>duplicateRow</name>
 <type>function</type>
 <desc>
 Duplicates an existing row after the setted row into the worksheet and sets the added row as current.
 </desc>
 <version>v4.9.0</version>
 <params>
   <param>
     <param-def>p_ws pls_integer default null</param-def>
     <param-desc>Index of the worksheet. null means the current Worksheet.</param-desc>
   </param>
    <param>
     <param-def>p_row pls_integer default null</param-def>
     <param-desc>Index of the row. null means the current Row.</param-desc>
   </param>
 </params>
 <return>
  <return-def>pls_integer</return-def>
  <return-desc>The index of the new row</return-desc>
 </return>
*/
function duplicateRow(
  p_ws pls_integer default null,
  p_row pls_integer default null
  ) return pls_integer;

/**
<name>duplicateRow</name>
 <type>procedure</type>
 <desc>
 Duplicates an existing row after the setted row into the worksheet and sets the added row as current.
 </desc>
 <version>v4.9.0</version>
 <params>
   <param>
     <param-def>p_ws pls_integer default null</param-def>
     <param-desc>Index of the worksheet. null means the current Worksheet.</param-desc>
   </param>
    <param>
     <param-def>p_row pls_integer default null</param-def>
     <param-desc>Index of the row. null means the current Row.</param-desc>
   </param>
 </params>
 <return>-</return>
*/
procedure duplicateRow(
  p_ws pls_integer default null,
  p_row pls_integer default null
  );

/**
<name>deleteRow</name>
 <type>procedure</type>
 <desc>
 Deletes row from the worksheet and updates references.
 </desc>
 <version>v4.9.4</version>
 <params>
   <param>
     <param-def>p_ws pls_integer default null</param-def>
     <param-desc>Index of the worksheet. null means the current Worksheet.</param-desc>
   </param>
    <param>
     <param-def>p_start pls_integer default null</param-def>
     <param-desc>Index of the row. null means the current Row.</param-desc>
   </param>
   <param>
     <param-def>p_check_refs boolean default true</param-def>
     <param-desc>Check and update the cell related referencies or not</param-desc>
   </param>
 </params>
 <return>-</return>
*/
procedure deleteRow(
  p_ws pls_integer default null,
  p_row pls_integer default null,
  p_check_refs boolean default true
  );

/**
<name>deleteRows</name>
 <type>procedure</type>
 <desc>
 Deletes rows from the worksheet and updates references.
 </desc>
 <version>v4.9.4</version>
 <params>
   <param>
     <param-def>p_ws pls_integer default null</param-def>
     <param-desc>Index of the worksheet. null means the current Worksheet.</param-desc>
   </param>
    <param>
     <param-def>p_row pls_integer default null</param-def>
     <param-desc>Index of the start row. null means the current Row.</param-desc>
   </param>
   <param>
     <param-def>p_num pls_integer default 1</param-def>
     <param-desc>number of the rows for delete. null means one row.</param-desc>
   </param>
   <param>
     <param-def>p_check_refs boolean default true</param-def>
     <param-desc>Check and update the cell related referencies or not</param-desc>
   </param>
 </params>
 <return>-</return>
*/
procedure deleteRows(
  p_ws pls_integer default null,
  p_row pls_integer default null,
  p_num pls_integer default 1,
  p_check_refs boolean default true
  );

/**
<name>clearRow</name>
 <type>procedure</type>
 <desc>
 Clears the value property of the all cell in the row.
 </desc>
 <version>v4.9.0</version>
 <params>
   <param>
     <param-def>p_ws pls_integer default null</param-def>
     <param-desc>Index of the worksheet. null means the current Worksheet.</param-desc>
   </param>
    <param>
     <param-def>p_row pls_integer default null</param-def>
     <param-desc>Index of the row. null means the current Row.</param-desc>
   </param>
 </params>
 <return>-</return>
*/
procedure clearRow(
  p_ws pls_integer default null,
  p_row pls_integer default null
  );

/**
<name>searchArrayName</name>
 <type>function</type>
 <desc>Searches for a predefined named cell array.</desc>
 <version>v4.9.0</version>
 <params>
   <param>
     <param-def>p_name varchar2</param-def>
     <param-desc>The name of the predefined array</param-desc>
   </param>
 </params>
 <return>
  <return-def>t_cell_arr_ref</return-def>
  <return-desc>the structure of the array reference</return-desc>
 </return>
*/
function searchArrayName(
  p_name varchar2
  ) return t_cell_arr_ref;

/**
<name>searchCellName</name>
 <type>function</type>
 <desc>Searches for a predefined named cell.</desc>
 <version>v4.9.0</version>
 <params>
   <param>
     <param-def>p_name varchar2</param-def>
     <param-desc>The name of the predefined cell</param-desc>
   </param>
 </params>
 <return>
  <return-def>t_cell_ref</return-def>
  <return-desc>the structure of the cell reference</return-desc>
 </return>
*/
function searchCellName(
  p_name varchar2
  ) return t_cell_ref;

/**
<name>searchCellName</name>
 <type>procedure</type>
 <desc>Searches for a predefined named cell.</desc>
 <version>v4.9.0</version>
 <params>
   <param>
     <param-def>p_name varchar2</param-def>
     <param-desc>The name of the predefined cell</param-desc>
   </param>
   <param>
     <param-def>p_ws out pls_integer</param-def>
     <param-desc>Index of the worksheet.</param-desc>
   </param>
    <param>
     <param-def>p_row out pls_integer</param-def>
     <param-desc>Index of the row.</param-desc>
   </param>
   <param>
     <param-def>p_cell varchar2</param-def>
     <param-desc>the referenceof the cell.</param-desc>
   </param>
 </params>
 <return>-</return>
*/
procedure searchCellName(
  p_name varchar2,
  p_ws out pls_integer,
  p_row out pls_integer,
  p_cell out varchar2
  );

/**
<name>searchCellValue</name>
 <type>function</type>
 <desc>Searches for a cell value on the sheet.</desc>
 <version>v4.9.0</version>
 <params>
   <param>
     <param-def>p_ws out pls_integer default null</param-def>
     <param-desc>Index of the worksheet, null means the current sheet.</param-desc>
   </param>
   <param>
     <param-def>p_value varchar2</param-def>
     <param-desc>The value of the cell</param-desc>
   </param>
 </params>
 <return>
  <return-def>t_cell_ref</return-def>
  <return-desc>the structure of the cell reference</return-desc>
 </return>
*/
function searchCellValue(
  p_ws pls_integer default null,
  p_value varchar2
  ) return t_cell_ref;
--
--INTERNAL
--v4.9.1
function isNewXLSX
  return boolean;
--
end;
--------------------------------------------------------------------------------
/

