create or replace package plpdf_oo_form is

/**
Collect information about form's fields.
%version v1.6.0
%param p_blob blob: PDF file
%return plpdf_type.t_form_fields: description of PDF form's fields, see plpdf_type package for returned value
*/
function GetPdfFieldsInfo(
  p_blob blob
  ) return plpdf_type.t_form_fields;

/**
Display (with dbms_output) collected information about form's fields.
%version v1.6.0
%param p_blob blob: PDF file
%return plpdf_type.t_form_fields: description of PDF form's fields, see plpdf_type package for returned value
*/
procedure WriteOutPdfFieldsInfo(
  p_info plpdf_type.t_form_fields
  );

/**
Prepare a PDF for filling.
%version v1.6.0
%param p_blob blob: PDF file
%return plpdf_type.t_prepform: internal structure for filling a PDF (prepared PDF)
*/
function PrepPDF(
  p_blob blob
  ) return plpdf_type.t_prepform;

/**
Check if a field exists in the prepared PDF
%version v1.6.0
%param p_prepform in out plpdf_type.t_prepform: prepared PDF
%param p_field_name varchar2: name of the field
%return boolean:
<ul>
<li>	true: field exists
<li>	false: field does not exist
</ul>
*/
function FieldExist(
  p_prepform in out plpdf_type.t_prepform,
  p_field_name varchar2
  ) return boolean;

/**
Set the value of a field in the prepared PDF.
%version v1.6.0
%param p_prepform in out plpdf_type.t_prepform: prepared PDF
%param p_field_name varchar2: name of the field
%param p_field_value varchar2: value of the field
%return -
*/
procedure SetValue(
  p_prepform in out plpdf_type.t_prepform,
  p_field_name varchar2,
  p_value varchar2
  );

/**
Set a field to read only in the prepared PDF.
%version v1.6.0
%param p_prepform in out plpdf_type.t_prepform: prepared PDF
%param p_field_name varchar2: name of the field
%return -
*/
procedure SetReadonly(
  p_prepform in out plpdf_type.t_prepform,
  p_field_name varchar2
  );

/**
Set the value of a field and then set the field to read only in the prepared PDF.
%version v1.6.0
%param p_prepform in out plpdf_type.t_prepform: prepared PDF
%param p_field_name varchar2: name of the field
%param p_field_value varchar2: value of the field
%return -
*/
procedure SetValueReadonly(
  p_prepform in out plpdf_type.t_prepform,
  p_field_name varchar2,
  p_value varchar2
  );

/**
Get a filled PDF as a BLOB variable.
%version v1.6.0
%param p_prepform in out plpdf_type.t_prepform: prepared PDF
%param p_blob out nocopy  blob: filled PDF as a BLOB
%return -
*/
procedure GetPDF(
  p_prepform plpdf_type.t_prepform,
  p_blob out nocopy  blob
  );

/**
Convert a boolean value to checkbox On/Off value
%version v1.6.0
%param p_boolean boolean: boolean value
%return varchar2: converted value
*/
function boolean2cb(
  p_boolean boolean
  ) return varchar2;

/**
Set a field's read only flag in the prepared PDF.
%version v1.6.0
%param p_prepform in out plpdf_type.t_prepform: prepared PDF
%param p_field_name varchar2: name of the field
%param p_readonly boolean:
<ul>
<li>	true: read only
<li>	false: not read only (can be updated)
</ul>
%return -
*/
procedure SetReadonlyFlag(
  p_prepform in out plpdf_type.t_prepform,
  p_field_name varchar2,
  p_readonly boolean
  );
-----------------------------------------
-- database save and load support
/*
CREATE OR REPLACE TYPE plpdf_type_prepform_obj AS OBJECT (
  obj number,
  gen number,
  field_name varchar2(255 char),
  field_subtype varchar2(20 char),
  value_type varchar2(10 char),
  orig_value varchar2(2000 char),
  orig_read_only number(1,0), -- boolean
  new_value varchar2(2000 char),
  new_read_only number(1,0), -- boolean
  orig_flag number,
  value_ref number,
  flag_ref number
);


CREATE OR REPLACE TYPE plpdf_type_prepform_objs AS TABLE OF plpdf_type_prepform_obj;

CREATE OR REPLACE TYPE plpdf_type_prepform_xref AS TABLE OF number;

CREATE OR REPLACE TYPE plpdf_type_obj_ref AS OBJECT (obj number, gen number);

CREATE OR REPLACE TYPE plpdf_type_prepform AS OBJECT (
    pdf_blob blob,
    last_obj_end number,
    xref plpdf_type_prepform_xref,
    objs plpdf_type_prepform_objs,
    root_ref plpdf_type_obj_ref,
    info_ref plpdf_type_obj_ref
    );

-- sample table
create table plpdf_00_form_store
(
id number not null,
filename varchar2(255 char),
orig_pdf blob,
prepform plpdf_type_prepform
)
nested table prepform.xref store as xref_table,
nested table prepform.objs store as objs_table
;

*/

/**
Convert a prepared PDF to a database level object structure
%version v1.6.0
%param p_prepform plpdf_type.t_prepform: prepared PDF
%return plpdf_type_prepform: database level object structure
*/
function Convert2Object(
  p_prepform plpdf_type.t_prepform
  ) return plpdf_type_prepform;

/**
Convert a database level object structure to a prepared PDF
%version v1.6.0
%param plpdf_type_prepform: database level object structure
%return p_prepform plpdf_type.t_prepform: prepared PDF
*/
function Convert2Type(
  p_prepform plpdf_type_prepform
  ) return plpdf_type.t_prepform;

end plpdf_oo_form;
/

create or replace package body plpdf_oo_form wrapped 
0
abcd
abcd
abcd
abcd
abcd
abcd
abcd
abcd
abcd
abcd
abcd
abcd
abcd
abcd
abcd
3
b
9200000
1
4
0 
7e
2 :e:
1PACKAGE:
1BODY:
1PLPDF_OO_FORM:
1FUNCTION:
1GETVERSION:
1RETURN:
1VARCHAR2:
1PLPDF:
1GETPDFFIELDSINFO:
1P_BLOB:
1BLOB:
1PLPDF_TYPE:
1T_FORM_FIELDS:
1PLPDF_PARSER:
1WRITEOUT:
1P_TXT:
1DBMS_OUTPUT:
1PUT_LINE:
1WRITEOUT2:
1P_TXT2:
1||:
1 :
1WRITEOUTPDFFIELDSINFO:
1P_INFO:
1begin:
1=======================:
1L_I:
11:
1COUNT:
1LOOP:
1OBJ:::
1OBJ:
1GEN:::
1GEN:
1NAME:::
1NAME:
1TYPE:::
1FIELD_SUBTYPE:
1VALUE_CONV:
1IS NOT NULL:
1VALUE(conv.):::
1VALUE_CHAR:
1=:
1FEFF:
1VALUE:::
1checkbox:
1radiobutton:
1SUBSTR:
12:
1choice:
1OPTIONS:
1----------------:
1L_II:
1OPT_VALS:
1VALUE:
1END OPTIONS:
1end:
1SETREADONLYFLAG:
1P_PREPFORM:
1OUT:
1T_PREPFORM:
1P_FIELD_NAME:
1P_READONLY:
1BOOLEAN:
1SETPREPFORMFIELDREADONLY:
1PREPPDF:
1GETPREPFORMPDF:
1FIELDEXIST:
1PREPFORMFIELDEXIST:
1SETVALUE:
1P_VALUE:
1SETPREPFORMFIELDVALUE:
1SETREADONLY:
1TRUE:
1SETVALUEREADONLY:
1GETPDF:
1NOCOPY:
1GETPREPPDF:
1CONVERT2OBJECT:
1PLPDF_TYPE_PREPFORM:
1L_PF:
1L_XREF:
1PLPDF_TYPE_PREPFORM_XREF:
1L_PFO:
1PLPDF_TYPE_PREPFORM_OBJS:
1L_ORO:
1NUMBER:
10:
1L_NRO:
1L_ROOT:
1PLPDF_TYPE_OBJ_REF:
1L_INFO:
1L_BLOB:
1EXTEND:
1XREF:
1I:
1OBJS:
1II:
1ORIG_READ_ONLY:
1NEW_READ_ONLY:
1PLPDF_TYPE_PREPFORM_OBJ:
1FIELD_NAME:
1VALUE_TYPE:
1ORIG_VALUE:
1NEW_VALUE:
1ORIG_FLAG:
1VALUE_REF:
1FLAG_REF:
1ROOT_REF:
1INFO_REF:
1DBMS_LOB:
1CREATETEMPORARY:
1SESSION:
1COPY:
1PDF_BLOB:
1GETLENGTH:
1LAST_OBJ_END:
1CONVERT2TYPE:
1L_RET:
1FALSE:
1BOOLEAN2CB:
1P_BOOLEAN:
1CHAR:
13:
1CHECKBOX_ON:
1CHECKBOX_OFF:
0

0
0
5c6
2
0 :2 a0 97 a0 8d a0 b4 a0
2c 6a :3 a0 6b 65 b7 a4 b1
11 68 4f a0 8d 8f a0 b0
3d b4 :3 a0 6b 2c 6a :3 a0 6b
:2 a0 e a5 b 65 b7 a4 b1
11 68 4f 9a 8f a0 b0 3d
b4 55 6a :2 a0 6b a0 a5 57
b7 a4 b1 11 68 4f 9a 8f
a0 b0 3d 8f a0 b0 3d b4
55 6a :2 a0 6b a0 7e 6e b4
2e 7e a0 b4 2e a5 57 b7
a4 b1 11 68 4f 9a 8f :2 a0
6b b0 3d b4 55 6a a0 6e
a5 57 a0 6e a5 57 91 51
:2 a0 6b a0 63 37 a0 6e :2 a0
a5 b a0 6b a5 57 a0 6e
:2 a0 a5 b a0 6b a5 57 a0
6e :2 a0 a5 b a0 6b a5 57
a0 6e :2 a0 a5 b a0 6b a5
57 :2 a0 a5 b a0 6b 7e b4
2e a0 6e :2 a0 a5 b a0 6b
a5 57 b7 :2 a0 a5 b a0 6b
7e 6e b4 2e a0 6e a5 57
b7 :2 a0 a5 b a0 6b 3e :2 6e
5 48 a0 6e :3 a0 a5 b a0
6b 51 a5 b a5 57 b7 a0
6e :2 a0 a5 b a0 6b a5 57
b7 :2 19 3c b7 :2 19 3c b7 :2 19
3c :2 a0 a5 b a0 6b 3e :2 6e
5 48 a0 6e a5 57 a0 6e
a5 57 91 51 :2 a0 a5 b a0
6b a0 6b a0 63 37 :2 a0 a5
b a0 6b a0 a5 b a0 6b
7e b4 2e :3 a0 a5 b a0 6b
a0 a5 b a0 6b a5 57 b7
:2 a0 a5 b a0 6b 7e 6e b4
2e :4 a0 a5 b a0 6b a0 a5
b a0 6b 51 a5 b a5 57
b7 :3 a0 a5 b a0 6b a0 a5
b a0 6b a5 57 b7 :2 19 3c
b7 :2 19 3c a0 6e a5 57 b7
a0 47 a0 6e a5 57 b7 19
3c a0 6e a5 57 b7 a0 47
a0 6e a5 57 b7 a4 b1 11
68 4f 9a 90 :3 a0 6b b0 3f
8f a0 b0 3d 8f a0 b0 3d
b4 55 6a a0 7e b4 2e a0
7e b4 2e a 10 :2 a0 6b :2 a0
e :2 a0 e :2 a0 e a5 57 b7
19 3c b7 a4 b1 11 68 4f
a0 8d 8f a0 b0 3d b4 :3 a0
6b 2c 6a :3 a0 6b :2 a0 e a5
b 65 b7 a4 b1 11 68 4f
a0 8d 90 :3 a0 6b b0 3f 8f
a0 b0 3d b4 :2 a0 2c 6a :3 a0
6b :2 a0 e :2 a0 e a5 b 65
b7 a4 b1 11 68 4f 9a 90
:3 a0 6b b0 3f 8f a0 b0 3d
8f a0 b0 3d b4 55 6a a0
7e b4 2e :2 a0 6b :2 a0 e :2 a0
e :2 a0 e a5 57 b7 19 3c
b7 a4 b1 11 68 4f 9a 90
:3 a0 6b b0 3f 8f a0 b0 3d
b4 55 6a a0 7e b4 2e :2 a0
6b :2 a0 e :2 a0 e :2 a0 e a5
57 b7 19 3c b7 a4 b1 11
68 4f 9a 90 :3 a0 6b b0 3f
8f a0 b0 3d 8f a0 b0 3d
b4 55 6a a0 7e b4 2e :2 a0
6b :2 a0 e :2 a0 e :2 a0 e a5
57 :2 a0 6b :2 a0 e :2 a0 e :2 a0
e a5 57 b7 19 3c b7 a4
b1 11 68 4f 9a 8f :2 a0 6b
b0 3d 96 :3 a0 b0 54 b4 55
6a :2 a0 6b :2 a0 e :2 a0 e a5
57 b7 a4 b1 11 68 4f a0
8d 8f :2 a0 6b b0 3d b4 :2 a0
2c 6a a3 a0 1c 81 b0 a3
a0 1c 81 b0 a3 a0 1c 81
b0 a3 a0 :2 51 a5 1c 81 b0
a3 a0 :2 51 a5 1c 81 b0 a3
a0 1c 81 b0 a3 a0 1c 81
b0 a3 a0 1c 81 b0 :2 a0 b4
2e d :2 a0 6b :2 a0 6b a0 6b
a5 57 91 51 :2 a0 6b a0 6b
a0 63 37 :2 a0 a5 b :2 a0 6b
a0 a5 b d b7 a0 47 :2 a0
b4 2e d :2 a0 6b :2 a0 6b a0
6b a5 57 91 51 :2 a0 6b a0
6b a0 63 37 :2 a0 6b a0 a5
b a0 6b a0 51 d b7 a0
51 d b7 :2 19 3c :2 a0 6b a0
a5 b a0 6b a0 51 d b7
a0 51 d b7 :2 19 3c :2 a0 a5
b :3 a0 6b a0 a5 b a0 6b
:2 a0 6b a0 a5 b a0 6b :2 a0
6b a0 a5 b a0 6b :2 a0 6b
a0 a5 b a0 6b :2 a0 6b a0
a5 b a0 6b :2 a0 6b a0 a5
b a0 6b :3 a0 6b a0 a5 b
a0 6b :3 a0 6b a0 a5 b a0
6b :2 a0 6b a0 a5 b a0 6b
:2 a0 6b a0 a5 b a0 6b a5
b d b7 a0 47 :4 a0 6b a0
6b :2 a0 6b a0 6b a5 b d
:4 a0 6b a0 6b :2 a0 6b a0 6b
a5 b d :2 a0 6b :4 a0 6b a5
57 :2 a0 6b :3 a0 6b :2 a0 6b :2 a0
6b a5 b a5 57 :5 a0 6b :4 a0
a5 b d :2 a0 65 b7 a4 b1
11 68 4f a0 8d 8f a0 b0
3d b4 :3 a0 6b 2c 6a a3 :2 a0
6b 1c 81 b0 a3 a0 1c 81
b0 a3 a0 1c 81 b0 :2 a0 6b
:2 a0 6b :3 a0 6b a5 57 :2 a0 6b
:2 a0 6b :2 a0 6b :2 a0 6b :2 a0 6b
a5 b a5 57 :2 a0 6b :2 a0 6b
d 91 51 :2 a0 6b a0 6b a0
63 37 :2 a0 6b a0 a5 b :2 a0
6b a0 a5 b d b7 a0 47
91 51 :2 a0 6b a0 6b a0 63
37 :2 a0 6b a0 a5 b a0 6b
7e 51 b4 2e :2 a0 d b7 :2 a0
d b7 :2 19 3c :2 a0 6b a0 a5
b a0 6b 7e 51 b4 2e :2 a0
d b7 :2 a0 d b7 :2 19 3c :2 a0
6b a0 a5 b a0 6b :2 a0 6b
a0 a5 b a0 6b d :2 a0 6b
a0 a5 b a0 6b :2 a0 6b a0
a5 b a0 6b d :2 a0 6b a0
a5 b a0 6b :2 a0 6b a0 a5
b a0 6b d :2 a0 6b a0 a5
b a0 6b :2 a0 6b a0 a5 b
a0 6b d :2 a0 6b a0 a5 b
a0 6b :2 a0 6b a0 a5 b a0
6b d :2 a0 6b a0 a5 b a0
6b :2 a0 6b a0 a5 b a0 6b
d :2 a0 6b a0 a5 b a0 6b
a0 d :2 a0 6b a0 a5 b a0
6b :2 a0 6b a0 a5 b a0 6b
d :2 a0 6b a0 a5 b a0 6b
a0 d :2 a0 6b a0 a5 b a0
6b :2 a0 6b a0 a5 b a0 6b
d :2 a0 6b a0 a5 b a0 6b
:2 a0 6b a0 a5 b a0 6b d
:2 a0 6b a0 a5 b a0 6b :2 a0
6b a0 a5 b a0 6b d b7
a0 47 :2 a0 6b a0 6b :2 a0 6b
a0 6b d :2 a0 6b a0 6b :2 a0
6b a0 6b d :2 a0 6b a0 6b
:2 a0 6b a0 6b d :2 a0 6b a0
6b :2 a0 6b a0 6b d :2 a0 65
b7 a4 b1 11 68 4f a0 8d
8f a0 b0 3d b4 :2 a0 2c 6a
a3 :2 a0 51 a5 1c 81 b0 :4 a0
6b d b7 :3 a0 6b d b7 :2 19
3c :2 a0 65 b7 a4 b1 11 68
4f b1 b7 a4 11 a0 b1 56
4f 1d 17 b5 
5c6
2
0 3 7 b 15 19 2d 31
32 36 3a 3e 42 46 4a 4d
51 53 57 59 65 69 6b 6f
8b 87 86 93 83 98 9c a0
a4 a7 ab af b3 b7 bb be
c2 c6 c8 c9 cb cf d1 d5
d7 e3 e7 e9 105 101 100 10d
fd 112 116 11a 11e 122 125 129
12a 12f 131 135 137 143 147 149
165 161 160 16d 17a 176 15d 182
175 187 18b 18f 193 172 197 19b
19e 1a3 1a4 1a9 1ac 1b0 1b1 1b6
1b7 1bc 1be 1c2 1c4 1d0 1d4 1d6
1f9 1ee 1f2 1f6 1ed 201 1ea 206
20a 20e 212 217 218 21d 221 226
227 22c 230 233 237 23b 23e 242
246 248 24c 251 255 259 25a 25c
260 263 264 269 26d 272 276 27a
27b 27d 281 284 285 28a 28e 293
297 29b 29c 29e 2a2 2a5 2a6 2ab
2af 2b4 2b8 2bc 2bd 2bf 2c3 2c6
2c7 2cc 2d0 2d4 2d5 2d7 2db 2de
2e1 2e2 2e7 2eb 2f0 2f4 2f8 2f9
2fb 2ff 302 303 308 30a 30e 312
313 315 319 31c 31f 324 325 32a
32e 333 334 339 33b 33f 343 344
346 34a 1 34d 352 357 35b 35e
362 367 36b 36f 373 374 376 37a
37d 380 381 383 384 389 38b 38f
394 398 39c 39d 39f 3a3 3a6 3a7
3ac 3ae 3b2 3b6 3b9 3bb 3bf 3c3
3c6 3c8 3cc 3d0 3d3 3d7 3db 3dc
3de 3e2 1 3e5 3ea 3ef 3f3 3f6
3fa 3ff 400 405 409 40e 40f 414
418 41b 41f 423 424 426 42a 42d
431 434 438 43c 43e 442 446 447
449 44d 450 454 455 457 45b 45e
461 462 467 46b 46f 473 474 476
47a 47d 481 482 484 488 48b 48c
491 493 497 49b 49c 49e 4a2 4a5
4a8 4ad 4ae 4b3 4b7 4bb 4bf 4c3
4c4 4c6 4ca 4cd 4d1 4d2 4d4 4d8
4db 4de 4df 4e1 4e2 4e7 4e9 4ed
4f1 4f5 4f6 4f8 4fc 4ff 503 504
506 50a 50d 50e 513 515 519 51d
520 522 526 52a 52d 531 536 537
53c 53e 542 549 54d 552 553 558
55a 55e 561 565 56a 56b 570 572
576 57d 581 586 587 58c 58e 592
594 5a0 5a4 5a6 5cd 5be 5c2 5c6
5ca 5bd 5d4 5e1 5dd 5ba 5e9 5f2
5ee 5dc 5fa 5d9 5ff 603 607 60b
60e 60f 614 618 61b 61c 1 621
626 62a 62e 631 635 639 63b 63f
643 645 649 64d 64f 650 655 657
65b 65e 660 664 666 672 676 678
67c 698 694 693 6a0 690 6a5 6a9
6ad 6b1 6b4 6b8 6bc 6c0 6c4 6c8
6cb 6cf 6d3 6d5 6d6 6d8 6dc 6de
6e2 6e4 6f0 6f4 6f6 6fa 721 712
716 71a 71e 711 728 735 731 70e
73d 730 742 746 74a 74e 752 756
75a 72d 75e 762 766 768 76c 770
772 773 775 779 77b 77f 781 78d
791 793 7ba 7ab 7af 7b3 7b7 7aa
7c1 7ce 7ca 7a7 7d6 7df 7db 7c9
7e7 7c6 7ec 7f0 7f4 7f8 7fb 7fc
801 805 809 80c 810 814 816 81a
81e 820 824 828 82a 82b 830 832
836 839 83b 83f 841 84d 851 853
87a 86b 86f 873 877 86a 881 88e
88a 867 896 889 89b 89f 8a3 886
8a7 8a8 8ad 8b1 8b5 8b8 8bc 8c0
8c2 8c6 8ca 8cc 8d0 8d4 8d6 8d7
8dc 8de 8e2 8e5 8e7 8eb 8ed 8f9
8fd 8ff 926 917 91b 91f 923 916
92d 93a 936 913 942 94b 947 935
953 932 958 95c 960 964 967 968
96d 971 975 978 97c 980 982 986
98a 98c 990 994 996 997 99c 9a0
9a4 9a7 9ab 9af 9b1 9b5 9b9 9bb
9bf 9c3 9c5 9c6 9cb 9cd 9d1 9d4
9d6 9da 9dc 9e8 9ec 9ee a11 a06
a0a a0e a05 a19 a2e a22 a26 a2a
a02 a35 a21 a3a a3e a42 a46 a1e
a4a a4e a52 a54 a58 a5c a5e a5f
a64 a66 a6a a6c a78 a7c a7e a82
aa5 a9a a9e aa2 a99 aad a96 ab2
ab6 aba abe ad7 ac6 aca ad2 ac5
af3 ae2 ae6 aee ac2 b0b afa afe
b06 ae1 b2b b16 ade b1a b1d b1e
b26 b15 b4b b36 b12 b3a b3d b3e
b46 b35 b67 b56 b5a b62 b32 b7f
b6e b72 b7a b55 b9b b8a b8e b96
b52 b86 ba2 ba6 ba7 bac bb0 bb4
bb8 bbb bbf bc3 bc6 bca bcd bce
bd3 bd7 bda bde be2 be5 be9 bec
bf0 bf4 bf6 bfa bfe bff c01 c05
c09 c0c c10 c11 c13 c17 c19 c1d
c24 c28 c2c c2d c32 c36 c3a c3e
c41 c45 c49 c4c c50 c53 c54 c59
c5d c60 c64 c68 c6b c6f c72 c76
c7a c7c c80 c84 c87 c8b c8c c8e
c92 c95 c99 c9c ca0 ca2 ca6 ca9
cad caf cb3 cb7 cba cbe cc2 cc5
cc9 cca ccc cd0 cd3 cd7 cda cde
ce0 ce4 ce7 ceb ced cf1 cf5 cf8
cfc d00 d01 d03 d07 d0b d0f d12
d16 d17 d19 d1d d20 d24 d28 d2b
d2f d30 d32 d36 d39 d3d d41 d44
d48 d49 d4b d4f d52 d56 d5a d5d
d61 d62 d64 d68 d6b d6f d73 d76
d7a d7b d7d d81 d84 d88 d8c d8f
d93 d94 d96 d9a d9d da1 da5 da9
dac db0 db1 db3 db7 dba dbe dc2
dc6 dc9 dcd dce dd0 dd4 dd7 ddb
ddf de2 de6 de7 de9 ded df0 df4
df8 dfb dff e00 e02 e06 e09 e0a
e0c e10 e12 e16 e1d e21 e25 e29
e2d e30 e34 e37 e3b e3f e42 e46
e49 e4a e4c e50 e54 e58 e5c e60
e63 e67 e6a e6e e72 e75 e79 e7c
e7d e7f e83 e87 e8b e8e e92 e96
e9a e9e ea1 ea2 ea7 eab eaf eb2
eb6 eba ebe ec1 ec5 ec9 ecc ed0
ed4 ed7 ed8 eda edb ee0 ee4 ee8
eec ef0 ef4 ef7 efb eff f03 f07
f08 f0a f0e f12 f16 f1a f1c f20
f22 f2e f32 f34 f38 f54 f50 f4f
f5c f4c f61 f65 f69 f6d f70 f74
f94 f7c f80 f84 f87 f8f f7b fb0
f9f fa3 fab f78 fc8 fb7 fbb fc3
f9e fcf fd3 f9b fd7 fdb fdf fe2
fe6 fea fee ff1 ff2 ff7 ffb fff
1002 1006 100a 100d 1011 1015 1018 101c
1020 1023 1027 102b 102e 102f 1031 1032
1037 103b 103f 1042 1046 104a 104d 1051
1055 1058 105c 1060 1063 1067 106a 106e
1072 1074 1078 107c 107f 1083 1084 1086
108a 108e 1091 1095 1096 1098 109c 109e
10a2 10a9 10ad 10b0 10b4 10b8 10bb 10bf
10c2 10c6 10ca 10cc 10d0 10d4 10d7 10db
10dc 10de 10e2 10e5 10e8 10eb 10ec 10f1
10f5 10f9 10fd 10ff 1103 1107 110b 110d
1111 1115 1118 111c 1120 1123 1127 1128
112a 112e 1131 1134 1137 1138 113d 1141
1145 1149 114b 114f 1153 1157 1159 115d
1161 1164 1168 116c 116f 1173 1174 1176
117a 117d 1181 1185 1188 118c 118d 118f
1193 1196 119a 119e 11a2 11a5 11a9 11aa
11ac 11b0 11b3 11b7 11bb 11be 11c2 11c3
11c5 11c9 11cc 11d0 11d4 11d8 11db 11df
11e0 11e2 11e6 11e9 11ed 11f1 11f4 11f8
11f9 11fb 11ff 1202 1206 120a 120e 1211
1215 1216 1218 121c 121f 1223 1227 122a
122e 122f 1231 1235 1238 123c 1240 1244
1247 124b 124c 124e 1252 1255 1259 125d
1260 1264 1265 1267 126b 126e 1272 1276
127a 127d 1281 1282 1284 1288 128b 128f
1293 1296 129a 129b 129d 12a1 12a4 12a8
12ac 12b0 12b3 12b7 12b8 12ba 12be 12c1
12c5 12c9 12cd 12d1 12d4 12d8 12d9 12db
12df 12e2 12e6 12ea 12ed 12f1 12f2 12f4
12f8 12fb 12ff 1303 1307 130a 130e 130f
1311 1315 1318 131c 1320 1324 1328 132b
132f 1330 1332 1336 1339 133d 1341 1344
1348 1349 134b 134f 1352 1356 135a 135e
1361 1365 1366 1368 136c 136f 1373 1377
137a 137e 137f 1381 1385 1388 138c 1390
1394 1397 139b 139c 139e 13a2 13a5 13a9
13ad 13b0 13b4 13b5 13b7 13bb 13be 13c2
13c4 13c8 13cf 13d3 13d7 13da 13de 13e1
13e5 13e9 13ec 13f0 13f3 13f7 13fb 13ff
1402 1406 1409 140d 1411 1414 1418 141b
141f 1423 1427 142a 142e 1431 1435 1439
143c 1440 1443 1447 144b 144f 1452 1456
1459 145d 1461 1464 1468 146b 146f 1473
1477 147b 147d 1481 1483 148f 1493 1495
1499 14b5 14b1 14b0 14bd 14ad 14c2 14c6
14ca 14ce 14ef 14d6 14da 14de 14e1 14e2
14ea 14d5 14f6 14fa 14fe 1502 14d2 1506
150a 150c 1510 1514 1518 151b 151f 1521
1525 1529 152c 1530 1534 1538 153a 153e
1540 154c 1550 1552 1554 1556 155a 1566
156a 156c 156f 1571 1572 157b 
5c6
2
0 1 9 e 1 a 3 0
a :2 1 3 a :2 10 3 :7 1 a
3 a :2 3 1a 5 c 17 c
:2 1 3 5 :2 12 7 11 7 :2 5
3 :6 1 b 3 9 :2 3 13 :2 1
3 :2 f 18 :2 3 :6 1 b 3 9
:3 3 a :2 3 14 :2 1 3 :2 f 18
1f 22 :2 18 26 29 :2 18 :2 3 :6 1
b 3 a 15 a :2 3 20 :2 1
3 c :3 3 c :2 3 7 e 11
:2 18 1e e 3 5 f 17 1e
:2 17 :2 23 :3 5 f 16 1d :2 16 :2 22
:3 5 f 17 1e :2 17 :2 23 :3 5 f
17 1e :2 17 :2 23 :2 5 8 f :2 8
:2 14 :3 8 7 11 21 28 :2 21 :2 2d
:2 7 2b a 11 :2 a :2 16 21 23
:2 21 9 12 :2 9 2a c 13 :2 c
:2 18 c 2a 35 :2 c b 15 1e
25 2c :2 25 :2 31 3c :2 1e :2 b 44
b 15 1e 25 :2 1e :2 2a :2 b :4 9
:4 7 :4 5 8 f :2 8 :2 14 8 27
31 :2 8 7 10 :3 7 10 :2 7 b
13 16 1d :2 16 :2 22 :2 2b 31 13
7 c 13 :2 c :2 18 21 :2 c :2 27
:3 c b 14 1b :2 14 :2 20 29 :2 14
:2 2f :2 b 3e e 15 :2 e :2 1a 28
2a :2 28 d 16 1d 24 :2 1d :2 29
32 :2 1d :2 38 3e :2 16 :2 d 38 d
16 1d :2 16 :2 22 2b :2 16 :2 31 :2 d
:4 b :5 9 12 :2 9 31 b :2 7 10
:2 7 40 :3 5 e :2 5 1e 7 :2 3
c :2 3 :6 1 b 3 11 15 20
15 :3 3 10 :3 3 e :2 3 1a :2 1
:4 6 :4 23 :2 6 5 :2 12 7 15 :2 7
17 :2 7 15 7 :2 5 3b :2 3 :7 1
a 3 a :2 3 11 5 c 17
c :2 1 3 5 :2 12 7 11 7
:2 5 3 :7 1 a 3 11 15 20
15 :3 3 10 :2 3 14 5 c :2 1
3 5 :2 12 7 15 :2 7 17 7
:2 5 3 :6 1 b 3 11 15 20
15 :3 3 10 :3 3 b :2 3 13 :2 1
:4 6 5 :2 12 7 15 :2 7 17 :2 7
12 7 :2 5 1f :2 3 :6 1 b 3
11 15 20 15 :3 3 10 :2 3 16
:2 1 :4 6 5 :2 12 7 15 :2 7 17
:2 7 15 7 :2 5 1f :2 3 :6 1 b
3 11 15 20 15 :3 3 10 :3 3
b :2 3 1b :2 1 :4 6 5 :2 12 7
15 :2 7 17 :2 7 12 7 :3 5 :2 12
7 15 :2 7 17 :2 7 15 7 :2 5
1f :2 3 :6 1 b 3 e 19 e
:3 3 a e 16 :2 3 11 :2 1 3
:2 10 5 13 :2 5 f 5 :2 3 :7 1
a 3 e 19 e :2 3 18 5
c :2 1 3 :3 8 :2 3 :3 a :2 3 :3 9
:2 3 9 10 12 f :2 9 :2 3 9
10 12 f :2 9 :2 3 :3 a :2 3 :3 a
:2 3 :3 a :2 3 :3 d :2 3 :2 a 11 :2 1c
:2 21 :2 3 7 c f :2 1a :2 1f 25
c 3 5 c :2 5 12 :2 1d 22
:2 12 5 25 7 :2 3 :3 c :2 3 :2 9
10 :2 1b :2 20 :2 3 7 d 10 :2 1b
:2 20 26 d 3 8 :2 13 18 :2 8
:2 1c 7 10 7 2b 7 10 7
:4 5 8 :2 13 18 :2 8 :2 1c 7 10
7 2a 7 10 7 :5 5 b :2 5
12 13 :2 1e 23 :2 13 :2 27 13 :2 1e
23 :2 13 :2 27 13 :2 1e 23 :2 13 :2 27
13 :2 1e 23 :2 13 :2 27 13 :2 1e 23
:2 13 :2 27 13 :2 1e 23 :2 13 :2 27 :2 13
:2 1e 23 :2 13 :2 27 :2 13 :2 1e 23 :2 13
:2 27 13 :2 1e 23 :2 13 :2 27 13 :2 1e
23 :2 13 :2 27 :2 12 5 26 8 3
4 e 21 :2 2c :2 35 39 :2 44 :2 4d
:2 e :2 4 e 21 :2 2c :2 35 39 :2 44
:2 4d :2 e :2 4 :2 d 1d 25 2b :2 34
:3 4 :2 d 12 19 :2 24 2d :2 36 40
:2 4b :2 2d :3 4 c :2 b :2 16 :4 b :2 c
4 3 a 3 :7 1 a 3 e
:2 3 16 5 c 17 c :2 1 3
9 14 :3 9 :2 3 :3 9 :2 3 :3 9 :2 3
:2 c 1c :2 22 2c 32 :2 3b :3 3 :2 c
11 :2 17 20 :2 2b 34 :2 3d 47 :2 52
:2 34 :3 3 :2 9 19 :2 24 3 7 c
f :2 1a :2 1f 25 c 3 4 :2 a
f :2 4 15 :2 20 25 :2 15 4 25
7 3 7 d 10 :2 1b :2 20 26
d 3 8 :2 13 18 :2 8 :2 1c 2b
2d :2 2b 7 10 7 2f 7 10
7 :4 5 8 :2 13 18 :2 8 :2 1c 2a
2c :2 2a 7 10 7 2e 7 10
7 :5 5 :2 b 10 :2 5 :2 14 1b :2 26
2b :2 1b :2 2f :2 5 :2 b 10 :2 5 :2 14
1b :2 26 2b :2 1b :2 2f :2 5 :2 b 10
:2 5 :2 14 22 :2 2d 32 :2 22 :2 36 :2 5
:2 b 10 :2 5 :2 14 25 :2 30 35 :2 25
:2 39 :2 5 :2 b 10 :2 5 :2 14 22 :2 2d
32 :2 22 :2 36 :2 5 :2 b 10 :2 5 :2 14
22 :2 2d 32 :2 22 :2 36 :2 5 :2 b 10
:2 5 :2 14 26 :2 5 :2 b 10 :2 5 :2 14
21 :2 2c 31 :2 21 :2 35 :2 5 :2 b 10
:2 5 :2 14 25 :2 5 :2 b 10 :2 5 :2 14
21 :2 2c 31 :2 21 :2 35 :2 5 :2 b 10
:2 5 :2 14 21 :2 2c 31 :2 21 :2 35 :2 5
:2 b 10 :2 5 :2 14 20 :2 2b 30 :2 20
:2 34 5 26 7 :2 3 :2 9 :2 12 19
:2 24 :2 2d :2 3 :2 9 :2 12 19 :2 24 :2 2d
:2 3 :2 9 :2 12 19 :2 24 :2 2d :2 3 :2 9
:2 12 19 :2 24 :2 2d :2 3 a 3 :7 1
a 3 d :2 3 14 5 c :2 1
3 9 14 :2 11 :2 9 3 6 5
e :2 1b 5 10 5 e :2 1b 5
:5 3 a 3 :a 1 5 :6 1 
5c6
4
0 :3 1 :2 3 4
0 4 :2 3 :5 6
:2 5 :4 3 :2 a :4 b
a :4 c :2 a e
:3 f :3 10 :2 f e
:2 d :4 a 15 :4 16
:3 15 :6 19 :2 18 :4 15
1d :4 1e :4 1f :3 1d
:e 22 :2 21 :4 1d 26
:6 27 :3 26 :4 2a :4 2b
:8 2c :a 2d :a 2e :a 2f
:a 30 :9 31 :a 32 31
:a 34 :4 35 34 :b 37
:e 38 37 :a 3a 39
:3 37 36 :3 34 33
:3 31 :b 3e :4 3f :4 40
:d 41 :e 42 :e 43 42
:a 45 :12 46 45 :e 48
47 :3 45 44 :3 42
:4 4b 41 4c 41
:4 4d :3 3e :4 4f 2c
50 2c :4 51 :2 29
:4 26 55 :7 56 :4 57
:4 58 :3 55 :a 5b :3 5c
:3 5d :3 5e :3 5f :2 5c
:3 5b :2 5a :4 55 :2 64
:4 65 64 :4 66 :2 64
68 :3 69 :3 6a :2 69
68 :2 67 :4 64 :2 6e
:7 6f :4 70 6e :2 71
:2 6e 73 :3 74 :3 75
:3 76 :2 74 73 :2 72
:4 6e 7a :7 7b :4 7c
:4 7d :3 7a :4 80 :3 81
:3 82 :3 83 :3 84 :2 81
:3 80 :2 7f :4 7a 89
:7 8a :4 8b :3 89 :4 8e
:3 8f :3 90 :3 91 :3 92
:2 8f :3 8e :2 8d :4 89
97 :7 98 :4 99 :4 9a
:3 97 :4 9d :3 9e :3 9f
:3 a0 :3 a1 :2 9e :3 a4
:3 a5 :3 a6 :3 a7 :2 a4
:3 9d :2 9c :4 97 ac
:6 ad :6 ae :3 ac :3 b1
:3 b2 :3 b3 :2 b1 :2 b0
:4 ac :2 ba :6 bb ba
:2 bc :2 ba :5 bd :5 be
:5 bf :8 c0 :8 c1 :5 c2
:5 c3 :5 c4 :5 c7 :a c8
:a c9 :b ca c9 cb
c9 :5 ce :a cf :a d0
:8 d1 :3 d2 d1 :3 d4
d3 :3 d1 :8 d6 :3 d7
d6 :3 d9 d8 :3 d6
:5 db :8 dc :8 dd :8 de
:8 df :8 e0 :8 e1 e2
:8 e3 e4 :8 e5 :8 e6
:8 e7 :3 db d0 ea
d0 :f ec :f ed :a ef
:11 f0 :2 f2 f3 :3 f4
f5 f6 f7 f8
:3 f2 :3 fb :2 c5 :4 ba
:2 ff :4 100 ff :4 101
:2 ff :7 102 :5 103 :5 104
:c 107 :13 108 :7 10a :a 10c
:d 10d 10c 10e 10c
:a 110 :c 111 :3 112 111
:3 114 113 :3 111 :c 116
:3 117 116 :3 119 118
:3 116 :11 11b :11 11c :11 11d
:11 11e :11 11f :11 120 :a 121
:11 122 :a 123 :11 124 :11 125
:11 126 110 127 110
:b 129 :b 12a :b 12c :b 12d
:3 130 :2 105 :4 ff :2 134
:4 135 134 :2 136 :2 134
:8 137 139 :5 13a 139
:5 13c 13b :3 139 :3 13e
:2 138 :4 134 :4 3 141
:6 1 
157d
4
:3 0 1 :3 0 2
:3 0 3 :6 0 1
:2 0 4 :3 0 5
:a 0 14 2 :7 0
6 :4 0 7 :3 0
7 8 0 14
5 9 :2 0 6
:3 0 8 :3 0 5
:3 0 c d 0
e :2 0 10 3
13 :3 0 13 0
13 12 10 11
:6 0 14 1 0
5 9 13 5c0
:2 0 4 :3 0 9
:a 0 31 3 :7 0
7 :2 0 5 b
:3 0 a :7 0 1a
19 :3 0 6 :3 0
c :3 0 d :2 0
4 1e 1f 0
1c 20 0 31
17 21 :2 0 6
:3 0 e :3 0 9
:3 0 24 25 0
a :3 0 a :3 0
27 28 9 26
2a 2b :2 0 2d
b 30 :3 0 30
0 30 2f 2d
2e :6 0 31 1
0 17 21 30
5c0 :2 0 f :a 0
45 4 :7 0 f
:2 0 d 7 :3 0
10 :7 0 36 35
:3 0 38 :2 0 45
33 39 :2 0 11
:3 0 12 :3 0 3b
3c 0 10 :3 0
11 3d 3f :2 0
41 13 44 :3 0
44 0 44 43
41 42 :6 0 45
1 0 33 39
44 5c0 :2 0 13
:a 0 65 5 :7 0
17 172 0 15
7 :3 0 10 :7 0
4a 49 :3 0 53
54 0 19 7
:3 0 14 :7 0 4e
4d :3 0 50 :2 0
65 47 51 :2 0
11 :3 0 12 :3 0
10 :3 0 15 :2 0
16 :4 0 1c 57
59 :3 0 15 :2 0
14 :3 0 1f 5b
5d :3 0 22 55
5f :2 0 61 24
64 :3 0 64 0
64 63 61 62
:6 0 65 1 0
47 51 64 5c0
:2 0 17 :a 0 189
6 :7 0 28 :2 0
26 c :3 0 d
:2 0 4 69 6a
0 18 :7 0 6c
6b :3 0 6e :2 0
189 67 6f :2 0
f :3 0 19 :4 0
2a 71 73 :2 0
185 f :3 0 1a
:4 0 2c 75 77
:2 0 185 1b :3 0
1c :2 0 18 :3 0
1d :3 0 7b 7c
0 1e :3 0 7a
7d :2 0 79 7f
13 :3 0 1f :4 0
18 :3 0 1b :3 0
2e 83 85 20
:3 0 86 87 0
30 81 89 :2 0
17e 13 :3 0 21
:4 0 18 :3 0 1b
:3 0 33 8d 8f
22 :3 0 90 91
0 35 8b 93
:2 0 17e 13 :3 0
23 :4 0 18 :3 0
1b :3 0 38 97
99 24 :3 0 9a
9b 0 3a 95
9d :2 0 17e 13
:3 0 25 :4 0 18
:3 0 1b :3 0 3d
a1 a3 26 :3 0
a4 a5 0 3f
9f a7 :2 0 17e
18 :3 0 1b :3 0
42 a9 ab 27
:3 0 ac ad 0
28 :2 0 44 af
b0 :3 0 13 :3 0
29 :4 0 18 :3 0
1b :3 0 46 b4
b6 27 :3 0 b7
b8 0 48 b2
ba :2 0 bc 4b
f9 18 :3 0 1b
:3 0 4d bd bf
2a :3 0 c0 c1
0 2b :2 0 2c
:4 0 51 c3 c5
:3 0 f :3 0 2d
:4 0 54 c7 c9
:2 0 cb 56 f5
18 :3 0 1b :3 0
58 cc ce 26
:3 0 cf d0 0
2e :4 0 2f :4 0
5a :3 0 d1 d2
d5 13 :3 0 2d
:4 0 30 :3 0 18
:3 0 1b :3 0 5d
da dc 2a :3 0
dd de 0 31
:2 0 5f d9 e1
62 d7 e3 :2 0
e5 65 f1 13
:3 0 2d :4 0 18
:3 0 1b :3 0 67
e8 ea 2a :3 0
eb ec 0 69
e6 ee :2 0 f0
6c f2 d6 e5
0 f3 0 f0
0 f3 6e 0
f4 71 f6 c6
cb 0 f7 0
f4 0 f7 73
0 f8 76 fa
b1 bc 0 fb
0 f8 0 fb
78 0 17e 18
:3 0 1b :3 0 7b
fc fe 26 :3 0
ff 100 0 32
:4 0 2f :4 0 7d
:3 0 101 102 105
f :3 0 33 :4 0
80 107 109 :2 0
177 f :3 0 34
:4 0 82 10b 10d
:2 0 177 35 :3 0
1c :2 0 18 :3 0
1b :3 0 84 111
113 36 :3 0 114
115 0 1d :3 0
116 117 0 1e
:3 0 110 118 :2 0
10f 11a 18 :3 0
1b :3 0 86 11c
11e 36 :3 0 11f
120 0 35 :3 0
88 121 123 27
:3 0 124 125 0
28 :2 0 8a 127
128 :3 0 f :3 0
18 :3 0 1b :3 0
8c 12b 12d 36
:3 0 12e 12f 0
35 :3 0 8e 130
132 27 :3 0 133
134 0 90 12a
136 :2 0 138 92
169 18 :3 0 1b
:3 0 94 139 13b
26 :3 0 13c 13d
0 2b :2 0 2f
:4 0 98 13f 141
:3 0 f :3 0 30
:3 0 18 :3 0 1b
:3 0 9b 145 147
36 :3 0 148 149
0 35 :3 0 9d
14a 14c 37 :3 0
14d 14e 0 31
:2 0 9f 144 151
a2 143 153 :2 0
155 a4 165 f
:3 0 18 :3 0 1b
:3 0 a6 157 159
36 :3 0 15a 15b
0 35 :3 0 a8
15c 15e 37 :3 0
15f 160 0 aa
156 162 :2 0 164
ac 166 142 155
0 167 0 164
0 167 ae 0
168 b1 16a 129
138 0 16b 0
168 0 16b b3
0 170 f :3 0
34 :4 0 b6 16c
16e :2 0 170 b8
172 1e :3 0 11b
170 :4 0 177 f
:3 0 38 :4 0 bb
173 175 :2 0 177
bd 178 106 177
0 179 c2 0
17e f :3 0 1a
:4 0 c4 17a 17c
:2 0 17e c6 180
1e :3 0 80 17e
:4 0 185 f :3 0
39 :4 0 ce 181
183 :2 0 185 d0
188 :3 0 188 0
188 187 185 186
:6 0 189 1 0
67 6f 188 5c0
:2 0 3a :a 0 1bd
9 :7 0 d7 5d9
0 d5 3c :3 0
c :3 0 3d :2 0
4 18e 18f 0
3b :6 0 191 190
:3 0 db :2 0 d9
7 :3 0 3e :7 0
195 194 :3 0 40
:3 0 3f :7 0 199
198 :3 0 19b :2 0
1bd 18b 19c :2 0
3e :3 0 28 :2 0
df 19f 1a0 :3 0
3f :3 0 28 :2 0
e1 1a3 1a4 :3 0
1a1 1a6 1a5 :2 0
e :3 0 41 :3 0
1a8 1a9 0 3b
:3 0 3b :3 0 1ab
1ac 3e :3 0 3e
:3 0 1ae 1af 3f
:3 0 3f :3 0 1b1
1b2 e3 1aa 1b4
:2 0 1b6 e7 1b7
1a7 1b6 0 1b8
e9 0 1b9 eb
1bc :3 0 1bc 0
1bc 1bb 1b9 1ba
:6 0 1bd 1 0
18b 19c 1bc 5c0
:2 0 4 :3 0 42
:a 0 1da a :7 0
ef :2 0 ed b
:3 0 a :7 0 1c3
1c2 :3 0 6 :3 0
c :3 0 3d :2 0
4 1c7 1c8 0
1c5 1c9 0 1da
1c0 1ca :2 0 6
:3 0 e :3 0 43
:3 0 1cd 1ce 0
a :3 0 a :3 0
1d0 1d1 f1 1cf
1d3 1d4 :2 0 1d6
f3 1d9 :3 0 1d9
0 1d9 1d8 1d6
1d7 :6 0 1da 1
0 1c0 1ca 1d9
5c0 :2 0 4 :3 0
44 :a 0 1ff b
:7 0 f7 72d 0
f5 3c :3 0 c
:3 0 3d :2 0 4
1e0 1e1 0 3b
:6 0 1e3 1e2 :3 0
1ef 1f0 0 f9
7 :3 0 3e :7 0
1e7 1e6 :3 0 6
:3 0 40 :3 0 1e9
1eb 0 1ff 1dd
1ec :2 0 6 :3 0
e :3 0 45 :3 0
3b :3 0 3b :3 0
1f2 1f3 3e :3 0
3e :3 0 1f5 1f6
fc 1f1 1f8 1f9
:2 0 1fb ff 1fe
:3 0 1fe 0 1fe
1fd 1fb 1fc :6 0
1ff 1 0 1dd
1ec 1fe 5c0 :2 0
46 :a 0 22d c
:7 0 103 7c6 0
101 3c :3 0 c
:3 0 3d :2 0 4
204 205 0 3b
:6 0 207 206 :3 0
107 :2 0 105 7
:3 0 3e :7 0 20b
20a :3 0 7 :3 0
47 :7 0 20f 20e
:3 0 211 :2 0 22d
201 212 :2 0 3e
:3 0 28 :2 0 10b
215 216 :3 0 e
:3 0 48 :3 0 218
219 0 3b :3 0
3b :3 0 21b 21c
3e :3 0 3e :3 0
21e 21f 47 :3 0
47 :3 0 221 222
10d 21a 224 :2 0
226 111 227 217
226 0 228 113
0 229 115 22c
:3 0 22c 0 22c
22b 229 22a :6 0
22d 1 0 201
212 22c 5c0 :2 0
49 :a 0 257 d
:7 0 119 886 0
117 3c :3 0 c
:3 0 3d :2 0 4
232 233 0 3b
:6 0 235 234 :3 0
28 :2 0 11b 7
:3 0 3e :7 0 239
238 :3 0 23b :2 0
257 22f 23c :2 0
3e :3 0 11e 23f
240 :3 0 e :3 0
41 :3 0 242 243
0 3b :3 0 3b
:3 0 245 246 3e
:3 0 3e :3 0 248
249 3f :3 0 4a
:3 0 24b 24c 120
244 24e :2 0 250
124 251 241 250
0 252 126 0
253 128 256 :3 0
256 0 256 255
253 254 :6 0 257
1 0 22f 23c
256 5c0 :2 0 4b
:a 0 293 e :7 0
12c 932 0 12a
3c :3 0 c :3 0
3d :2 0 4 25c
25d 0 3b :6 0
25f 25e :3 0 130
:2 0 12e 7 :3 0
3e :7 0 263 262
:3 0 7 :3 0 47
:7 0 267 266 :3 0
269 :2 0 293 259
26a :2 0 3e :3 0
28 :2 0 134 26d
26e :3 0 e :3 0
48 :3 0 270 271
0 3b :3 0 3b
:3 0 273 274 3e
:3 0 3e :3 0 276
277 47 :3 0 47
:3 0 279 27a 136
272 27c :2 0 28c
e :3 0 41 :3 0
27e 27f 0 3b
:3 0 3b :3 0 281
282 3e :3 0 3e
:3 0 284 285 3f
:3 0 4a :3 0 287
288 13a 280 28a
:2 0 28c 13e 28d
26f 28c 0 28e
141 0 28f 143
292 :3 0 292 0
292 291 28f 290
:6 0 293 1 0
259 26a 292 5c0
:2 0 4c :a 0 2b4
f :7 0 147 a1e
0 145 c :3 0
3d :2 0 4 297
298 0 3b :7 0
29a 299 :3 0 2a5
2a6 0 149 3c
:3 0 4d :3 0 b
:3 0 a :5 0 1
2a0 29f :3 0 2a2
:2 0 2b4 295 2a3
:2 0 e :3 0 4e
:3 0 3b :3 0 3b
:3 0 2a8 2a9 a
:3 0 a :3 0 2ab
2ac 14c 2a7 2ae
:2 0 2b0 14f 2b3
:3 0 2b3 0 2b3
2b2 2b0 2b1 :6 0
2b4 1 0 295
2a3 2b3 5c0 :2 0
4 :3 0 4f :a 0
401 10 :7 0 153
:2 0 151 c :3 0
3d :2 0 4 2b9
2ba 0 3b :7 0
2bc 2bb :3 0 6
:3 0 50 :3 0 2be
2c0 0 401 2b7
2c1 :2 0 157 ade
0 155 50 :3 0
2c4 :7 0 2c7 2c5
0 3ff 0 51
:6 0 1c :2 0 159
53 :3 0 2c9 :7 0
2cc 2ca 0 3ff
0 52 :6 0 55
:3 0 2ce :7 0 2d1
2cf 0 3ff 0
54 :6 0 1c :2 0
15e 57 :3 0 58
:2 0 15b 2d3 2d6
:6 0 2d9 2d7 0
3ff 0 56 :6 0
165 b52 0 163
57 :3 0 58 :2 0
160 2db 2de :6 0
2e1 2df 0 3ff
0 59 :6 0 169
b86 0 167 5b
:3 0 2e3 :7 0 2e6
2e4 0 3ff 0
5a :6 0 5b :3 0
2e8 :7 0 2eb 2e9
0 3ff 0 5c
:6 0 52 :3 0 b
:3 0 2ed :7 0 2f0
2ee 0 3ff 0
5d :6 0 53 :4 0
2f2 2f3 :3 0 2f1
2f4 0 3fd 52
:3 0 5e :3 0 2f6
2f7 0 3b :3 0
5f :3 0 2f9 2fa
0 1d :3 0 2fb
2fc 0 16b 2f8
2fe :2 0 3fd 60
:3 0 1c :2 0 3b
:3 0 5f :3 0 302
303 0 1d :3 0
304 305 0 1e
:3 0 301 306 :2 0
300 308 52 :3 0
60 :3 0 16d 30a
30c 3b :3 0 5f
:3 0 30e 30f 0
60 :3 0 16f 310
312 30d 313 0
315 171 317 1e
:3 0 309 315 :4 0
3fd 54 :3 0 55
:4 0 319 31a :3 0
318 31b 0 3fd
54 :3 0 5e :3 0
31d 31e 0 3b
:3 0 61 :3 0 320
321 0 1d :3 0
322 323 0 173
31f 325 :2 0 3fd
62 :3 0 1c :2 0
3b :3 0 61 :3 0
329 32a 0 1d
:3 0 32b 32c 0
1e :3 0 328 32d
:2 0 327 32f 3b
:3 0 61 :3 0 331
332 0 62 :3 0
175 333 335 63
:3 0 336 337 0
56 :3 0 1c :2 0
339 33a 0 33c
177 341 56 :3 0
58 :2 0 33d 33e
0 340 179 342
338 33c 0 343
0 340 0 343
17b 0 3b1 3b
:3 0 61 :3 0 344
345 0 62 :3 0
17e 346 348 64
:3 0 349 34a 0
59 :3 0 1c :2 0
34c 34d 0 34f
180 354 59 :3 0
58 :2 0 350 351
0 353 182 355
34b 34f 0 356
0 353 0 356
184 0 3b1 54
:3 0 62 :3 0 187
357 359 65 :3 0
3b :3 0 61 :3 0
35c 35d 0 62
:3 0 189 35e 360
20 :3 0 361 362
0 3b :3 0 61
:3 0 364 365 0
62 :3 0 18b 366
368 22 :3 0 369
36a 0 3b :3 0
61 :3 0 36c 36d
0 62 :3 0 18d
36e 370 66 :3 0
371 372 0 3b
:3 0 61 :3 0 374
375 0 62 :3 0
18f 376 378 26
:3 0 379 37a 0
3b :3 0 61 :3 0
37c 37d 0 62
:3 0 191 37e 380
67 :3 0 381 382
0 3b :3 0 61
:3 0 384 385 0
62 :3 0 193 386
388 68 :3 0 389
38a 0 56 :3 0
3b :3 0 61 :3 0
38d 38e 0 62
:3 0 195 38f 391
69 :3 0 392 393
0 59 :3 0 3b
:3 0 61 :3 0 396
397 0 62 :3 0
197 398 39a 6a
:3 0 39b 39c 0
3b :3 0 61 :3 0
39e 39f 0 62
:3 0 199 3a0 3a2
6b :3 0 3a3 3a4
0 3b :3 0 61
:3 0 3a6 3a7 0
62 :3 0 19b 3a8
3aa 6c :3 0 3ab
3ac 0 19d 35b
3ae 35a 3af 0
3b1 1aa 3b3 1e
:3 0 330 3b1 :4 0
3fd 5a :3 0 5b
:3 0 3b :3 0 6d
:3 0 3b6 3b7 0
20 :3 0 3b8 3b9
0 3b :3 0 6d
:3 0 3bb 3bc 0
22 :3 0 3bd 3be
0 1ae 3b5 3c0
3b4 3c1 0 3fd
5c :3 0 5b :3 0
3b :3 0 6e :3 0
3c5 3c6 0 20
:3 0 3c7 3c8 0
3b :3 0 6e :3 0
3ca 3cb 0 22
:3 0 3cc 3cd 0
1b1 3c4 3cf 3c3
3d0 0 3fd 6f
:3 0 70 :3 0 3d2
3d3 0 5d :3 0
4a :3 0 6f :3 0
71 :3 0 3d7 3d8
0 1b4 3d4 3da
:2 0 3fd 6f :3 0
72 :3 0 3dc 3dd
0 5d :3 0 3b
:3 0 73 :3 0 3e0
3e1 0 6f :3 0
74 :3 0 3e3 3e4
0 3b :3 0 73
:3 0 3e6 3e7 0
1b8 3e5 3e9 1ba
3de 3eb :2 0 3fd
51 :3 0 50 :3 0
5d :3 0 3b :3 0
75 :3 0 3f0 3f1
0 52 :3 0 54
:3 0 5a :3 0 5c
:3 0 1be 3ee 3f7
3ed 3f8 0 3fd
6 :3 0 51 :3 0
3fb :2 0 3fd 1c5
400 :3 0 400 1d2
400 3ff 3fd 3fe
:6 0 401 1 0
2b7 2c1 400 5c0
:2 0 4 :3 0 76
:a 0 58d 13 :7 0
1dd :2 0 1db 50
:3 0 3b :7 0 407
406 :3 0 6 :3 0
c :3 0 3d :2 0
4 40b 40c 0
409 40d 0 58d
404 40e :2 0 1e1
f9b 0 1df c
:3 0 3d :2 0 4
411 412 0 413
:7 0 416 414 0
58b 0 77 :6 0
421 422 0 1e3
40 :3 0 418 :7 0
41b 419 0 58b
0 56 :6 0 40
:3 0 41d :7 0 420
41e 0 58b 0
59 :6 0 6f :3 0
70 :3 0 77 :3 0
73 :3 0 424 425
0 4a :3 0 6f
:3 0 71 :3 0 428
429 0 1e5 423
42b :2 0 589 6f
:3 0 72 :3 0 42d
42e 0 77 :3 0
73 :3 0 430 431
0 3b :3 0 73
:3 0 433 434 0
6f :3 0 74 :3 0
436 437 0 3b
:3 0 73 :3 0 439
43a 0 1e9 438
43c 1eb 42f 43e
:2 0 589 77 :3 0
75 :3 0 440 441
0 3b :3 0 75
:3 0 443 444 0
442 445 0 589
60 :3 0 1c :2 0
3b :3 0 5f :3 0
449 44a 0 1d
:3 0 44b 44c 0
1e :3 0 448 44d
:2 0 447 44f 77
:3 0 5f :3 0 451
452 0 60 :3 0
1ef 453 455 3b
:3 0 5f :3 0 457
458 0 60 :3 0
1f1 459 45b 456
45c 0 45e 1f3
460 1e :3 0 450
45e :4 0 589 62
:3 0 1c :2 0 3b
:3 0 61 :3 0 463
464 0 1d :3 0
465 466 0 1e
:3 0 462 467 :2 0
461 469 3b :3 0
61 :3 0 46b 46c
0 62 :3 0 1f5
46d 46f 63 :3 0
470 471 0 2b
:2 0 1c :2 0 1f9
473 475 :3 0 56
:3 0 4a :3 0 477
478 0 47a 1fc
47f 56 :3 0 78
:3 0 47b 47c 0
47e 1fe 480 476
47a 0 481 0
47e 0 481 200
0 557 3b :3 0
61 :3 0 482 483
0 62 :3 0 203
484 486 64 :3 0
487 488 0 2b
:2 0 1c :2 0 207
48a 48c :3 0 59
:3 0 4a :3 0 48e
48f 0 491 20a
496 59 :3 0 78
:3 0 492 493 0
495 20c 497 48d
491 0 498 0
495 0 498 20e
0 557 77 :3 0
61 :3 0 499 49a
0 62 :3 0 211
49b 49d 20 :3 0
49e 49f 0 3b
:3 0 61 :3 0 4a1
4a2 0 62 :3 0
213 4a3 4a5 20
:3 0 4a6 4a7 0
4a0 4a8 0 557
77 :3 0 61 :3 0
4aa 4ab 0 62
:3 0 215 4ac 4ae
22 :3 0 4af 4b0
0 3b :3 0 61
:3 0 4b2 4b3 0
62 :3 0 217 4b4
4b6 22 :3 0 4b7
4b8 0 4b1 4b9
0 557 77 :3 0
61 :3 0 4bb 4bc
0 62 :3 0 219
4bd 4bf 66 :3 0
4c0 4c1 0 3b
:3 0 61 :3 0 4c3
4c4 0 62 :3 0
21b 4c5 4c7 66
:3 0 4c8 4c9 0
4c2 4ca 0 557
77 :3 0 61 :3 0
4cc 4cd 0 62
:3 0 21d 4ce 4d0
26 :3 0 4d1 4d2
0 3b :3 0 61
:3 0 4d4 4d5 0
62 :3 0 21f 4d6
4d8 26 :3 0 4d9
4da 0 4d3 4db
0 557 77 :3 0
61 :3 0 4dd 4de
0 62 :3 0 221
4df 4e1 67 :3 0
4e2 4e3 0 3b
:3 0 61 :3 0 4e5
4e6 0 62 :3 0
223 4e7 4e9 67
:3 0 4ea 4eb 0
4e4 4ec 0 557
77 :3 0 61 :3 0
4ee 4ef 0 62
:3 0 225 4f0 4f2
68 :3 0 4f3 4f4
0 3b :3 0 61
:3 0 4f6 4f7 0
62 :3 0 227 4f8
4fa 68 :3 0 4fb
4fc 0 4f5 4fd
0 557 77 :3 0
61 :3 0 4ff 500
0 62 :3 0 229
501 503 63 :3 0
504 505 0 56
:3 0 506 507 0
557 77 :3 0 61
:3 0 509 50a 0
62 :3 0 22b 50b
50d 69 :3 0 50e
50f 0 3b :3 0
61 :3 0 511 512
0 62 :3 0 22d
513 515 69 :3 0
516 517 0 510
518 0 557 77
:3 0 61 :3 0 51a
51b 0 62 :3 0
22f 51c 51e 64
:3 0 51f 520 0
59 :3 0 521 522
0 557 77 :3 0
61 :3 0 524 525
0 62 :3 0 231
526 528 6a :3 0
529 52a 0 3b
:3 0 61 :3 0 52c
52d 0 62 :3 0
233 52e 530 6a
:3 0 531 532 0
52b 533 0 557
77 :3 0 61 :3 0
535 536 0 62
:3 0 235 537 539
6b :3 0 53a 53b
0 3b :3 0 61
:3 0 53d 53e 0
62 :3 0 237 53f
541 6b :3 0 542
543 0 53c 544
0 557 77 :3 0
61 :3 0 546 547
0 62 :3 0 239
548 54a 6c :3 0
54b 54c 0 3b
:3 0 61 :3 0 54e
54f 0 62 :3 0
23b 550 552 6c
:3 0 553 554 0
54d 555 0 557
23d 559 1e :3 0
46a 557 :4 0 589
77 :3 0 6d :3 0
55a 55b 0 20
:3 0 55c 55d 0
3b :3 0 6d :3 0
55f 560 0 20
:3 0 561 562 0
55e 563 0 589
77 :3 0 6d :3 0
565 566 0 22
:3 0 567 568 0
3b :3 0 6d :3 0
56a 56b 0 22
:3 0 56c 56d 0
569 56e 0 589
77 :3 0 6e :3 0
570 571 0 20
:3 0 572 573 0
3b :3 0 6e :3 0
575 576 0 20
:3 0 577 578 0
574 579 0 589
77 :3 0 6e :3 0
57b 57c 0 22
:3 0 57d 57e 0
3b :3 0 6e :3 0
580 581 0 22
:3 0 582 583 0
57f 584 0 589
6 :3 0 77 :3 0
587 :2 0 589 24c
58c :3 0 58c 257
58c 58b 589 58a
:6 0 58d 1 0
404 40e 58c 5c0
:2 0 4 :3 0 79
:a 0 5b9 16 :7 0
25d :2 0 25b 40
:3 0 7a :7 0 593
592 :3 0 6 :3 0
7 :3 0 595 597
0 5b9 590 598
:2 0 5a4 5a5 0
262 7 :3 0 7b
:3 0 7c :2 0 25f
59b 59e :6 0 5a1
59f 0 5b7 0
77 :6 0 7a :3 0
77 :3 0 e :3 0
7d :3 0 5a3 5a6
0 5a8 264 5af
77 :3 0 e :3 0
7e :3 0 5aa 5ab
0 5a9 5ac 0
5ae 266 5b0 5a2
5a8 0 5b1 0
5ae 0 5b1 268
0 5b5 6 :3 0
77 :3 0 5b3 :2 0
5b5 26b 5b8 :3 0
5b8 26e 5b8 5b7
5b5 5b6 :6 0 5b9
1 0 590 598
5b8 5c0 :3 0 5be
0 5be :3 0 5be
5c0 5bc 5bd :6 0
5c1 :2 0 3 :3 0
270 0 3 5be
5c4 :3 0 5c3 5c1
5c5 :8 0 
280
4
:3 0 1 f 1
18 1 1b 1
29 1 2c 1
34 1 37 1
3e 1 40 1
48 1 4c 2
4b 4f 2 56
58 2 5a 5c
1 5e 1 60
1 68 1 6d
1 72 1 76
1 84 2 82
88 1 8e 2
8c 92 1 98
2 96 9c 1
a2 2 a0 a6
1 aa 1 ae
1 b5 2 b3
b9 1 bb 1
be 1 c4 2
c2 c4 1 c8
1 ca 1 cd
2 d3 d4 1
db 2 df e0
2 d8 e2 1
e4 1 e9 2
e7 ed 1 ef
2 f1 f2 1
f3 2 f5 f6
1 f7 2 f9
fa 1 fd 2
103 104 1 108
1 10c 1 112
1 11d 1 122
1 126 1 12c
1 131 1 135
1 137 1 13a
1 140 2 13e
140 1 146 1
14b 2 14f 150
1 152 1 154
1 158 1 15d
1 161 1 163
2 165 166 1
167 2 169 16a
1 16d 2 16b
16f 1 174 4
10a 10e 172 176
1 178 1 17b
7 8a 94 9e
a8 fb 179 17d
1 182 4 74
78 180 184 1
18c 1 193 1
197 3 192 196
19a 1 19e 1
1a2 3 1ad 1b0
1b3 1 1b5 1
1b7 1 1b8 1
1c1 1 1c4 1
1d2 1 1d5 1
1de 1 1e5 2
1e4 1e8 2 1f4
1f7 1 1fa 1
202 1 209 1
20d 3 208 20c
210 1 214 3
21d 220 223 1
225 1 227 1
228 1 230 1
237 2 236 23a
1 23e 3 247
24a 24d 1 24f
1 251 1 252
1 25a 1 261
1 265 3 260
264 268 1 26c
3 275 278 27b
3 283 286 289
2 27d 28b 1
28d 1 28e 1
296 1 29c 2
29b 2a1 2 2aa
2ad 1 2af 1
2b8 1 2bd 1
2c3 1 2c8 1
2cd 2 2d4 2d5
1 2d2 2 2dc
2dd 1 2da 1
2e2 1 2e7 1
2ec 1 2fd 1
30b 1 311 1
314 1 324 1
334 1 33b 1
33f 2 341 342
1 347 1 34e
1 352 2 354
355 1 358 1
35f 1 367 1
36f 1 377 1
37f 1 387 1
390 1 399 1
3a1 1 3a9 c
363 36b 373 37b
383 38b 38c 394
395 39d 3a5 3ad
3 343 356 3b0
2 3ba 3bf 2
3c9 3ce 3 3d5
3d6 3d9 1 3e8
3 3df 3e2 3ea
6 3ef 3f2 3f3
3f4 3f5 3f6 c
2f5 2ff 317 31c
326 3b3 3c2 3d1
3db 3ec 3f9 3fc
8 2c6 2cb 2d0
2d8 2e0 2e5 2ea
2ef 1 405 1
408 1 410 1
417 1 41c 3
426 427 42a 1
43b 3 432 435
43d 1 454 1
45a 1 45d 1
46e 1 474 2
472 474 1 479
1 47d 2 47f
480 1 485 1
48b 2 489 48b
1 490 1 494
2 496 497 1
49c 1 4a4 1
4ad 1 4b5 1
4be 1 4c6 1
4cf 1 4d7 1
4e0 1 4e8 1
4f1 1 4f9 1
502 1 50c 1
514 1 51d 1
527 1 52f 1
538 1 540 1
549 1 551 e
481 498 4a9 4ba
4cb 4dc 4ed 4fe
508 519 523 534
545 556 a 42c
43f 446 460 559
564 56f 57a 585
588 3 415 41a
41f 1 591 1
594 2 59d 59c
1 59a 1 5a7
1 5ad 2 5af
5b0 2 5b1 5b4
1 5a0 f 14
31 45 65 189
1bd 1da 1ff 22d
257 293 2b4 401
58d 5b9 
1
4
0 
5c4
0
1
28
16
3a
0 1 1 1 1 1 6 7
1 1 1 1 1 1 1 1
10 10 1 13 13 1 0 0
0 0 0 0 0 0 0 0
0 0 0 0 0 0 0 0

10f 8 0
2ec 10 0
1c0 1 a
2cd 10 0
259 1 e
2c3 10 0
461 15 0
327 12 0
17 1 3
4c 5 0
47 1 5
59a 16 0
410 13 0
41c 13 0
2da 10 0
417 13 0
2d2 10 0
29c f 0
1c1 a 0
48 5 0
34 4 0
18 3 0
197 9 0
261 e 0
237 d 0
209 c 0
1e5 b 0
193 9 0
591 16 0
1dd 1 b
590 1 16
3 0 1
2e2 10 0
2e7 10 0
79 7 0
201 1 c
447 14 0
405 13 0
300 11 0
2b8 10 0
296 f 0
25a e 0
230 d 0
202 c 0
1de b 0
18c 9 0
295 1 f
67 1 6
5 1 2
33 1 4
404 1 13
22f 1 d
18b 1 9
2c8 10 0
68 6 0
2b7 1 10
265 e 0
20d c 0
0

/
